# Copyright (C) 2003  CAMP
# Please see the accompanying LICENSE file for further information.

__docformat__ = 'reStructuredText'

"""A container for dacapo simulations

    This class provides a container for running a dacapo simulation.

    In general the idea is that one first creates a 'Simulation()'
    object. This is more or less just a container for all the
    possible classes one can add:

    'ListOfAtoms' -- For describing the geometry and content of the
	unitcell.

    'PlaneWaveCutoff' -- For setting the plane wave cutoff

    'ElectronicBands' -- For setting the number of electronic bands 

    'NetCDF.Entry' -- For a general variable as described in the 
	'dacapo' manual.

    'NetCDF.Series' -- For a variable with infinite dimension as described in
	the 'dacapo' manual.

    Each of these classes is added as an attribute to the simulation
    instance, and are capable of independently read and write themselves
    to a netcdf file, when asked to do so by the containing object.

    **An example:**

    '>>> mysim=Simulation()'
    '>>> mysim.plancut=PlaneWaveCutoff(340)'
    '>>> mysim.tot=NetCDF.Series("TotalEnergy")'

    having done this setup we can do variuos things:

    '>>> mysim.WriteAsNetCDFFile("myfile.nc")'

    to create a file with the two entries,

    '>>> mysim.UpdateFromNetCDFFile("oldfile.nc")'

    to read in values for the given entries from a previously created file.

    '>>> mysim.Execute("output.nc")'

    to run a dacapo simulation with the given values of the entry, and store
    the result in the file "output.nc". The type of job which is to be 
    executed, serial or parallel, can be specied when the simulation object
    is created:

    '>>> mysim=Simulation("parallel")' 

    serial is default.
"""

import JobTypes
import os
from Dacapo.NetCDF import Collection
from exceptions import Exception
from Dacapo import NetCDF

class DelayedReadNetCDFEntry(NetCDF.Entry):
    """ class for delayed read of data into NetCDF variable.
    data is read from ncfile then WriteToNetCDFFile is called, and
    is not kept in memory after.
    """

    def __init__(self,ncfile,name=None,dimensionnames=None):

        self._ncfile = ncfile
        if not name:
            self.SetName(self.__class__.__name__)
        else:
            self.SetName(name)

        if dimensionnames is not None:
            self.SetDimensionNames(dimensionnames)

    def WriteToNetCDFFile(self,file,indices=None):

        print 'WriteToNetCDFFile, file = ',file,self._ncfile
        # read
        var = self.ReadFromNetCDFFile(self._ncfile)
        value = var.GetValue()
        print 'shape value ',value.shape
        self.SetValue(value)
        
        
        NetCDF.Entry.WriteToNetCDFFile(self,file)
        
        self.SetValue(None) 
        

class DacapoError(Exception):
    """ Class for errors caused by running dacapo

    When and if dacapo implements errorcodes this should be subclassed
    to provide better information on which error occured."""

#uses tempfile
#     Scientific.IO.NetCDF
class Simulation(Collection):
    """A container for dacapo simulations

    This class provides a container for running a dacapo simulation.
    An instance is created by:

    '>>> a=Simulation(jobtype=' *jobtype* ')'

    where jobtype is a string (e.g. "Parallel"), an integer or an instance
    of the 'JobType' class. The default is a serial job. 
    """

    def __init__(self,jobtype="Default"):

        # Collection.__init__(self)
        self.SetJobType(jobtype)

        # this flag is needed by the ExecuteExternalDynamics to signal if 
        # dacapo is currently running
        self._dacapo_is_running = 'false'   

    def __str__(self):
	text=repr(self)+":\n"
	for attr in self.__dict__.keys():
	    text=text+"\n"+str(getattr(self,attr))+"\n"
	return text

    def __repr__(self):
        return "Simulation()"

    def reprWithName(self,name):
   	import string
        s = "#!/usr/bin/env python\n"
	s = s+"#This file is generated using the Simulation.py class \n\n"
    	s = s+"# import the necessary classes etc.\n"
    	s = s+"from Simulations.Dacapo import *	\n\n"
        s=s+name+"="+repr(self)+"\n"
	list = self.__dict__.keys()
        for keys in list:
            t = "\n"
	    if keys[-1]!="_" and keys[0]!="_":#don't take private attr
                try:
                    doc = self.__dict__[keys].__doc__
		    if doc is not None:
			    ret = string.find(doc,'\n')
			    if ret > 0:
			    	doc = doc[0:ret]
			    t =t+ "#"+doc+"\n"
                except AttributeError :
                    pass
                try:
                    t =t+self.__dict__[keys].reprWithName(name+".%s" %keys)
                except AttributeError:
                    t =t+(name+".%s = " %keys)+ repr(self.__dict__[keys])
                t =t+ "\n"
                s=s+t
        return s

    # Partially implement copy protocol hooks :__getinitargs__, __getstate__ and __setstate__. 
    # __getinitargs__(), which should return a tuple containing the arguments to be passed to
    # the class constructor (__init__()).
    
    def __getinitargs__(self):  return (self.GetJobType(),)
    def __getstate__(self):     return self.__dict__
    def __setstate__(self, state):     self.__dict__.update(state)

    def Update(self,subject):
	pass
        # self.Notify()

    def SetJobType(self,jobtype):
	if JobTypes.JobTypes.has_key(jobtype):			
	    self._jobtype_=JobTypes.JobTypes[jobtype]
	else:
	    try: #test integrity of jobtype:
		jobtype.GetExecutable()
		self._jobtype_=jobtype
	    except AttributeError:
		raise "invalid jobtype"
        # self.Notify()

    def GetJobType(self):
	return self._jobtype_

    def UpdateFromNetCDFFile(self,filename,index=None):
	"""Method to read data to attributes from a netcdf file

	This method asks all the attributes of the instance
	to read their data from the file with the given filename.
	Example:

	'>>> mysim=Simulation()'
	'>>> mysim.tot=NetCDF.Series("TotalEnergy")'
	'>>> mysim.UpdateFromNetCDFFile("myfile.nc")'

	will read in all the total energies from "myfile" into
	the object mysim.tot.

	The optional 'index' parameter gives control over which
	step in a dynamic simulation to read in from (default is
	all/last depending on the attribute).
	Example:

	'>>> mysim=Simulation()'
	'>>> mysim.tot=NetCDF.Series("TotalEnergy")'
	'>>> mysim.UpdateFromNetCDFFile("myfile.nc",index=5)'

	will read the total energy in the 5th. dynamic step. 
	"""
	from Scientific.IO.NetCDF import NetCDFFile 
	file=NetCDFFile(filename,'r')
	self.ReadFromNetCDFFile(file,index)
	file.close()

    def WriteAsNetCDFFile(self,filename):
	"""Method to create a netcdf file with data from attributes

	This method asks all the attributes of the instance
	to write them down in the file with the given filename.
	Example:

	'>>> mysim=Simulation()'
	'>>> mysim.plancut=PlaneWaveCutoff(340)'
	'>>> mysim.kpt=NetCDF.Entry("KpointSetup")'
	'>>> mysim.WriteToNetCDFFile("myfile.nc")'

	will create a file with two entries.
	"""
	from Scientific.IO.NetCDF import NetCDFFile 
	file=NetCDFFile(filename,'w')
	self.WriteToNetCDFFile(file)
	file.close()

    def Execute(self,outfile=None,ascii=None,scratch=None,infile=None,stopfile='stop',nonblocking=None):
	"""Method to perform a dacapo run.

	Their are several different ways to run a simulation:

	'>>> mysim.Execute()'

	This simply runs dacapo with the present state of the
	simulation, and updates all the attributes from the
	output afterwards. This can be used in cases where
	only a small fraction of the output is of interest,
	since nothing will be left on the disk afterwards.

	If the output is needed for later inspection the
	following command will save the output in the
	given file:

	'>>> mysim.Execute("output.nc")'

	Using this option the status of the simulation is not
	updated from the output, but this can easily be done
	using:

	'>>> mysim.UpdateFromNetCDFFile("output.nc")'

	If one wants the ascii output redirected into a file
	(instead of on stdin) one can use e.g.:

	'>>> mysim.Execute(outfile="output.nc",ascii="output.txt")'

	There is also the possibility of using a special path for
	the scratch files used by dacapo:

	'>>> mysim.Execute(scratch=".")'

	Mainly interesting for debugging.

	Finally one can decide NOT to use the attributes in
	the object, but rather start a simulation with a
	specific previously created infile for dacapo:

	'>>> mysim.Execute(outfile="output.nc",infile="myfile.nc")'

	again the status of the simulation is not
	updated from the output in this case.

	In all cases where files are specified these are checked
	for existence and a conflict result in an error.
	"""
        import time
	tmp=0
	job=self.GetJobType()
	runstring=self.GetRunScript()
	scratchinfile=job.GetScratchInFile(scratch)
	infile=job.GetInFile(infile)
        if os.system("test -e "+infile)!=0:
            time0 = time.time()
            self.WriteAsNetCDFFile(scratchinfile)
            time1 = time.time()
            os.system("mv "+scratchinfile+" "+infile)
            time2 = time.time()
            # This 2 steps procedure of writing the netcdf-file is only advantageous on computer architectures where schratch-disk and main disk are not the same.
            # print "Timing of writing NetCDF-infile to scratch (seconds): ",time1-time0
            # print "Timing of moving NetCDF-infile from scratch to main disk (seconds): ",time2-time1
	    tmp=1
        runstring=runstring+" " + infile+" "
	outfile=job.GetOutFile(outfile)
	if outfile==None:
	    outfile=infile
	elif os.system("test -e "+outfile)==0:
	    print "Outfile already exists"
            print "Moving outfile to " + outfile + '.bak'
            os.system("mv " + outfile + " " + outfile+".bak") 
        self._outfile = outfile   # needed by the ExecuteExternalDynamics method
	runstring=runstring+outfile+" "	
	ascii=job.GetAsciiFile(ascii)
	if ascii!=None:
            runstring=runstring+"-out "+ascii+" "
            if os.system("test -e "+ascii)==0:
		print "ascii file already exists"
                print "Moving asciifile to " + ascii + '.bak'
                os.system("mv " + ascii + " " + ascii+".bak") 
	scratch=job.GetScratch(scratch)
	if scratch!=None:
	    runstring=runstring+"-scratch "+scratch+" "
	runstring = runstring+"-stop "+stopfile

        # if the call should be non blocking add '&' to the runstring
        if nonblocking!=None:
	    runstring=runstring+' &'

        errorcode= os.system(runstring)

        # if errorcode != 0:
        #     raise DacapoError,errorcode

	# print runstring
	# if no input file existed at call time remove the temporary file
	# if output is to be removed, read it in first.
	if tmp:
	    if outfile==infile:
		self.UpdateFromNetCDFFile(outfile)
            if nonblocking==None:
	        os.remove(infile)  

    def ExecuteExternalDynamics(self,outfile=None,infile=None,ascii=None,scratch=None,stopfile="stop",stopprogram=None):
        """ Method for running dacapo as an external force calculator.

	Using this method dacapo is not restarted but is waiting 
	for python to produce a new ListOfAtoms object contained in 
	the Simulation object. 
	This ListOfAtoms object is added to the netCDF file outfile. 
	The ExecuteExternalDynamics method will wait for dacapo to finish 
	the energy and force calculation, before returning. 
	Communication between dacapo and python is in this implementation using 
	internet socket's. 

	outfile,infile,asci,asciii and scratch is passed to the Execute method. An 
	'&' is added to the runstring produced by the Execute method to make 
	this non-blocking. 

	The script file which are called by dacapo is written by this method, 
	also the 'Dynamics' NetCDF entry is defined in the simulation object, 
	so dacapo knows which script file to call.

	If the parameter stopprogram is 'true', the stop file is written so that 
	dacapo will finish.  

        """
        from socket import socket,AF_INET,SOCK_STREAM
        import os
        from Scientific.IO.NetCDF import NetCDFFile
        from Dacapo import  NetCDF

        if stopprogram=='true': 
                # write stop file
                stfile = open(stopfile,'w') 
                stfile.write('1 \n') 
                stfile.close()

                # signal to dacapo that positions are ready
                # let dacapo continue, it is up to the python mainloop 
                # to allow dacapo enough time to finish properly.
                self._client.send('ok too proceed')        

		# Wait for dacapo to acknowledge that netcdf file has been updated, and analysis part of the code
		# has been terminated. Dacapo sends a signal at the end of call clexit().
                print 'waiting for dacapo to stop ... '
		self._client,self._addr = self.s.accept() # Last mumble before Dacapo dies.
		os.system("sleep 5")                     # Two seconds of silence mourning dacapo.

                # remove the flag _dacapo_is_running so subsequent calls to GetExecuteMethod will
                # be initialized correct
                self._dacapo_is_running=None

                # close the socket s
                self.s.close()
                self._client.close()

                if hasattr(self,'ncDynamics'): 
                	os.system('rm -f ' + self.ncDynamics.ExternalIonMotion_script)

                return

        if (self._dacapo_is_running=='true'): 

            # open netcdf file for append
            file = NetCDFFile(self._outfile,'a')
                
            # find listofatoms object in the Simulation object
            # add one to time-series in netcdf file and write position 
            for attr in self.__dict__.keys():
                try:
                    name = getattr(self,attr).GetName().strip()
                except AttributeError:
                    name = 'xxx'
                    
                if name=='DynamicAtomPositions':
                    getattr(self,attr).WriteToNetCDFFile(file,-1)
                        
            
            # signal to dacapo that positions are ready
            # let dacapo continue
            self._client.send('ok too proceed')        

        else:

            # get process pid that will be used as communication channel 
            pid = os.getpid()

            # setup communication channel to dacapo
            from sys    import version
            from string import split
            effpid = (pid)%(2**16-1025)+1025   # This translate pid [0;99999] to a number in [1025;65535] (the allowed socket numbers)

            self.s = socket(AF_INET,SOCK_STREAM)
	    foundafreesocket = 0
	    while not foundafreesocket:
		try:
            		if split(version)[0] > "2":     # new interface
                		self.s.bind(("",effpid))
            		else:                           # old interface
                		self.s.bind("",effpid)
                	foundafreesocket = 1
		except:
			effpid = effpid + 1

            # write script file that will be used by dacapo
            scriptname = 'script'+`pid`+'.py' 
            scriptfile = open(scriptname,'w') 
            scriptfile.write(
"""#!/usr/bin/env python
from socket import *
from sys    import version
from string import split  
s = socket(AF_INET,SOCK_STREAM)
# tell python that dacapo has finished
if split(version)[0] > "2":     # new interface 
     s.connect(("",""" + `effpid` + """))
else:                           # old interface
     s.connect("","""  + `effpid` + """)
# wait for python main loop
s.recv(14)
""")
            scriptfile.close()
            os.system('chmod +x ' + scriptname) 

            # setup dynamics as external and set the script name
            self.ncDynamics=NetCDF.Entry(name="Dynamics")
            self.ncDynamics.Type = "ExternalIonMotion"
            self.ncDynamics.ExternalIonMotion_script = './'+ scriptname
            
            # dacapo is not running start dacapo non blocking
            self.Execute(outfile=outfile,ascii=ascii,stopfile=stopfile,nonblocking='true') 

            self._dacapo_is_running = 'true'

            self.s.listen(1)      

        # wait for dacapo  
        self._client,self._addr = self.s.accept()

    def RestartDacapo(self):
        """ set a flag telling that dacapo should be restarted on
        next call of Calulate
        """
        self.restart_dacapo = True
            

    def GetExecuteMethod(self,outfile=None,infile=None,ascii=None,scratch=None,stopfile='stop',dynamics='internal',stopprogram=None):
        """ Method for generating a function pointer to the Execute method. 

            The function returned includes a call of the Execute method and an subsequent UpdateFromNetCDFFile. 
            If dynamics='external' the ExecuteExternalDynamics method is used.  

            The optional parameters are copied, except the 'dynamics' argument' to the Execute method, and are used 
            in subsequent use of the function pointer returned from this function.  
        """

        from Scientific.IO.NetCDF import NetCDFFile    

        # Define the function that will be returned from this method
        # This function is basically a call of the Execute method and a subsequent 
        # update, the optional parameters for the function is set at the initial GetExecuteMethod call. 
        def f(simulation=self,outfile=outfile,infile=infile,ascii=ascii,scratch=scratch,stopfile=stopfile,stopprogram=stopprogram,
              dynamics=dynamics):

            if (dynamics=='internal'):
                # remove out and infiles before starting dacapo again
                os.system("rm -f " + outfile) 
                os.system("rm -f " + ascii) 
                simulation.Execute(outfile=outfile,infile=infile,ascii=ascii,scratch=scratch,stopfile=stopfile)
            elif (dynamics=='external'): 
                simulation.ExecuteExternalDynamics(outfile=outfile,infile=infile,ascii=ascii,scratch=scratch,stopfile=stopfile,stopprogram=stopprogram)
            else: 
                raise 'Argument dynamics for GetExecuteMethod must be external or internal' 

            # delete wave function and charge density entry before update
            for attr in ['ChargeDensity','WaveFunction','WaveFunctionFFTindex']:
                simulation.DeleteNetCDFEntry(simulation,attr)
            # update the simulation 
            simulation.UpdateFromNetCDFFile(outfile)

        # return function pointer 
        return f


    def GetTkManipulator(self,parent=None,name=None):
	from Manipulators import TkManipulatorMainFrame
	return TkManipulatorMainFrame.TkManipulatorMainFrame(parent=parent, domain=self)
    
    def GetwxManipulator(self,parent=None,name="sim"):
	from Manipulators.wxPython.SimulationManipulator import SimulationManipulator
	return SimulationManipulator(parent=parent, subject=self,name=name)

    def GetHSMatrix(self):
        """ calculate Hamiltonian matrix and overlab matrix for the
        current set of wavefunctions
        """
        import tempfile
        
        # set non consistent flag
        self.eigsolver=NetCDF.Entry(name="ElectronicMinimization")
        self.eigsolver.Method="NonSelfconsistent"

        inncfile  = tempfile.mktemp()
        outncfile = tempfile.mktemp()

        self.Execute(outfile=outncfile,ascii=asciifile)

        self.hmat=NetCDF.Entry(name="HMatrix")
        self.smat=NetCDF.Entry(name="SMatrix")

        sim.UpdateFromNetCDFFile(outncfile)

        hmat1 = sim.hmat.GetValue()
        smat1 = sim.smat.GetValue()

        N = len(hmat1[:,0,0])
        hmat = num.zeros((N,N),num.Complex)
        smat = num.zeros((N,N),num.Complex)
        hmat.real = hmat1[:,:,0]
        hmat.imag = hmat1[:,:,1]
        smat.real = smat1[:,:,0]
        smat.imag = smat1[:,:,1]

        return hmat,smat
                                                                            
    def GetName(self):
	return "Simulation"
	
# --- end of class Simulation ---

# ------------------------------------------------------
# convenience function for generating default filenames
# ------------------------------------------------------
def ScriptNameRoot():
        # Allow user to generate a systematic root name for outputfiles.
        import os
        import sys
        try:
                rootname = os.path.splitext(os.path.basename(sys.argv[0]))[0]
                if not rootname: raise "VoidName"
                else:            return rootname
        except: return "pid"+`os.getpid()`
# ------------------------------------------------------
def DefaultFileName(ext, IDtoken=''):
	# Generate the string: scriptname[_IDtoken].ext
	if str(IDtoken) == str(''): return ScriptNameRoot() + "." + ext
	else                : return ScriptNameRoot() + "_" + str(IDtoken) + "." + ext
# ------------------------------------------------------
def DefaultOutFileName(IDtoken=''): 
        # Produce a default netCDF outputfilename: scriptname[_IDtoken].nc
	return DefaultFileName("nc", IDtoken)
# ------------------------------------------------------
def DefaultASCIIFileName(IDtoken=''):
	# Produce a default ASCII outputfilename:  scriptname[_IDtoken].txt
	return DefaultFileName("txt", IDtoken)
# ------------------------------------------------------    


    


