    module update_netCDF_history_module 
!=====================================================================
!   Write history information to the netCDF file
!   I/O filename from module run_context
!   The variables depending on the number_ionic_steps dimension :
!
!       UnitCell                  (dirc, transpose(dirc))
!       DynamicAtomPositions      (posion)
!       FermiLevel                (efermi) 
!       TotalEnergy               (entot)
!       DynamicAtomForces         (fortot)
!       DynamicAtomVelocities     (rvelo)
!       EigenValues               (eigen)
!       OccupationNumbers         (occ)
!       EvaluateTotalEnergy       (entot_all_xc)
!       EvaluateExchangeEnergy    (exc_and_corr_all(:,1))
!       EvaluateCorrelationEnergy (exc_and_corr_all(:,2))
!   

!   Previous history for variables not depending on system specific 
!   dimensions has allready been copied using MakeRestrictedCopy
!   (maybe this function should move here?)
!   History for variables depending on 'number_ionic_steps' and on a 
!   system specific dimensions are copied if the dimensions are not changed, 
!   otherwise this data will have netCDF fill values.  
!   Variables in this list is now 
!     EigenValues
!     OccupationNumbers
!   
!=====================================================================
    use netcdfinterface
    use run_context
    use matrix_utilities

!   global data
    integer           :: number_of_steps_in_netcdf=0
    integer,parameter :: NumberSystemDepVariables=2
    character*(nf_max_name)  :: ListOfSystemDepVariables(NumberSystemDepVariables)
    data ListOfSystemDepVariables(1:NumberSystemDepVariables)  / &
                      "EigenValues",  &
                      "OccupationNumbers" /

!##################################################################### 
    contains
!#####################################################################
    subroutine update_netcdf_history(nions,nspec,nkprun,nbands,nionsp,kspin,nspin,dirc,  &
                             posion,efermi,entot,enfree,rvelo,fortot,eigen,occ,entot_all_xc,    & 
                             exc_and_corr_all )
    implicit none
!=====================================================================
     integer,intent(in)  :: nions            ! total number of ions in the structure
     integer,intent(in)  :: nspec            ! total number species in the structure 
     integer,intent(in)  :: nkprun           ! number of k-points in IBZ
     integer,intent(in)  :: nbands           ! number of electronic bands
     integer,intent(in)  :: nionsp(nions)    ! number of ions of a present type, 
     integer,intent(in)  :: kspin(nkprun)    ! gives the spin for the k-points
     integer,intent(in)  :: nspin            ! number of spin , number of k-points
        
     real*8,intent(in)   :: dirc(3,3)              ! unit cell, vectors rowvise, i.e. basis vector j = dirc(j,:)
     real*8,intent(in)   :: posion(3,nions,nspec)  ! scaled atomic coordinates 
     real*8,intent(in)   :: efermi                 ! Fermi energy
     real*8,intent(in)   :: entot                  ! total energy (T=0)
     real*8,intent(in)   :: enfree                 ! total free energy
     real*8,intent(in)   :: fortot(3,nions,nspec)  ! atomic forces
     real*8,intent(in)   :: rvelo(3,nions)         ! atomic velocity
     real*8,intent(in)   :: eigen(nbands,nkprun)   ! eigen values
     real*8,intent(in)   :: occ(nbands,nkprun)     ! occupation numbers
     real*8,intent(in)   :: entot_all_xc(:)        ! non-selfconsistent total energies for all 
                                                   ! exchange-correlation functionals implemented
     real*8,intent(in)   :: exc_and_corr_all(:,:)  ! exchange (:,1) and correlation (:,2) energies 
                                                   ! for all exchange-correlation functionals 
                                                   ! implemented

!--------------------------------------------------------------------- 
!  locals 

   integer             :: ncid_in,ncid_out, status, i,nstep,nkp1,nsp,na,k,ni
   real*8              :: netcdf_pos(3,nions),netcdf_for(3,nions),netcdf_vel(3,nions)  ! netcdf layout of position and force 
   real*8              :: rlatc,dirc_netcdf(3,3)
   character(6)        :: exc_functional_keyword(6)

!-----------------------------------------------------------------------------
!  convert posion, fortot and rvelo -> netCDF layout
   na = 1
   do nsp = 1,nspec
      do ni = 1,nionsp(nsp)
        do k = 1,3
            netcdf_pos(k,na) = posion(k,ni,nsp)
            netcdf_for(k,na) = fortot(k,ni,nsp)
        enddo
        na = na + 1
      enddo
   enddo
   netcdf_vel(:,:) = rvelo(:,:)

!- reorder output arrays corresponding to userprovided atom order ------------

   do i = 1, 3
     call Reorder_atomvector(nconso, netcdf_pos(i,:), "internal_to_netCDF")  
     call Reorder_atomvector(nconso, netcdf_vel(i,:), "internal_to_netCDF")  
     call Reorder_atomvector(nconso, netcdf_for(i,:), "internal_to_netCDF")       
   enddo
!-------------------------------------------------------------------------------

! open netCDF out file 
  status =  nf_open(netCDF_output_filename, NF_WRITE, ncid_out )  
  if (status /= nf_noerr) call abort_calc(nconso,  &
                                "update_netcdf_history -> nf_open : error opening nc-outfile")

! if variables using system dependent dimensions has not been copied do that 
  if (number_of_steps_in_netcdf==0) then 

!============================================================================
!    define EvalFunctionalOfDensity_XC
!    The convention for the NetCDF variable EvalFunctionalOfDensity is that 
!    different tags EvalFunctionalOfDensity_TAGS can be defined, so that they 
!    together describe the total energy obtained from EvalFunctionalOfDensity 
!    For now only implemented for tag = XC, so that EvalFunctionalOfDensity gives 
!    a list of the total energy of each exchange correlatin potential 
!    implemented in the program.  

     exc_functional_keyword(1) = 'PZ'
     exc_functional_keyword(2) = 'VWN'
     exc_functional_keyword(3) = 'PW91'
     exc_functional_keyword(4) = 'PBE'
     exc_functional_keyword(5) = 'revPBE'
     exc_functional_keyword(6) = 'RPBE'

!    EvalFunctionalOfDensity
!    make sure we have the dim6 dimension
     status = nfputglobaldim(ncid_out,'dim6',6)
     if ((status/=nfif_OK ).and.(status/=nfif_dimexist_butOKsize))  then
        write(nconso,*) 'nfputglobaldim: error writing dim6',status
        call clexit(nconso)
     endif                                      
     status = nfdefvar(ncid_out,'EvalFunctionalOfDensity_XC',NF_CHAR,dim_name1='dim6', & 
                   dim_name2= 'dim6' )
     if ((status/=nfif_OK).and.(status/=nfif_varexist_butOKdef)) &
        call abort_calc(nconso,  &
           "update_netcdf_history -> nfdefvar : error defining EvalFunctionalOfDensity_XC")

!    write to EvalFunctionalOfDensity_XC
!     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC", exc_functional_keyword,dim_name1='dim6')
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC", exc_functional_keyword)
!    description
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%Description",  & 
           "Exchange-Correlation functionals used in EvalFunctionalOfDensity" ) 
!    put attributes that describes the keyword
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%PZ",  & 
       'Perdew-Zunger (Rev. Mod. Phys.  89) LDA' )
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%PW91",  & 
       'Perdew,Chevary,Vosko,Jackson,Pederson,Singh, and Fiolhais (Phys.Rev. B 1992) GGA'   )
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%VWN",  & 
       'Vosko, Wilk and Nusair (Can. J. Physics 80) LDA'   )
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%PBE",  & 
       'Perdew, Burke, and Ernzerhof (Phys. Rev. B. 97) GGA'   )
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%revPBE",  & 
       'Zhang and Yang (Phys. Rev. Lett. 98) GGA'   )
     status = nfput(ncid_out,"EvalFunctionalOfDensity_XC%RPBE",  & 
       'Hammer,Hansen,Norskov (Phys. Rev. B 99) GGA'   )

! ============================================================================

     ! open netCDF in file 
     status =  nf_open(netCDF_input_filename, NF_NOWRITE, ncid_in)      
     if (status /= nf_noerr) call abort_calc(nconso, &
                         "update_netcdf_history -> nf_open : error opening nc-infile")

!    Get the unlimited dimension in the SouceSet
     status = nfgetglobaldim(ncid_in,'number_ionic_steps',nstep) 
     if (status /= nfif_OK) call abort_calc(nconso, &
             "update_netcdf_history -> nfgetglobaldim : error reading unlimited dimension in SourceSet")

!    define the system dependent variables in the target set,
!    it is done here so that the call to copy_netCDF_variable does not create
!    the variables with the wrong dimensions
     status = nfdefvar(ncid_out,"EigenValues",NF_DOUBLE,                      &
             dim_name1 = 'number_of_bands',dim_name2 = 'number_of_spin',      &
             dim_name3 = 'number_IBZ_kpoints',dim_name4='number_ionic_steps')
     if ((status/=nfif_OK).and.(status/=nfif_varexist_butOKdef)) &
             call abort_calc(nconso,  &
            "update_netcdf_history -> nfdefvar : error defining EigenValues")

     status = nfdefvar(ncid_out,"OccupationNumbers",NF_DOUBLE,                &
             dim_name1 = 'number_of_bands',dim_name2 = 'number_of_spin',      &
             dim_name3 = 'number_IBZ_kpoints',dim_name4='number_ionic_steps')
     if ((status/=nfif_OK).and.(status/=nfif_varexist_butOKdef)) &
             call abort_calc(nconso,  &
            "update_netcdf_history -> nfdefvar : error defining OccupationNumbers")
                                                                                   
!    loop over list of variables that should be copied
     do i = 1, NumberSystemDepVariables
       
        status = copy_netCDF_variable(ncid_in,ncid_out,&
                       variablename=ListOfSystemDepVariables(i)) !  copy variable 
        if ((status/=nfif_OK).and.(status/=nfif_vardidnotexist)) &
            call abort_calc(nconso,  &
             "update_netcdf_history -> copy_netCDF_variable: unhandled exception")

     enddo ! i=1,NumberOfSystemDepVariables
        
     number_of_steps_in_netcdf = nstep 
  endif

 
  nstep = number_of_steps_in_netcdf

! UnitCell
! scale with UnitCell_ScaleFactor if is exist
  status = nfget(ncid_out, "UnitCell_ScaleFactor", rlatc)
  if (status /= nfif_OK ) then 
     rlatc = 1.0d0  ! set defaults
  endif
  if (status == nfif_OK) then 
    if (rlatc == nf_fill_double) then 
       rlatc = 1.0    ! set default                 
    endif 
  endif
  dirc_netcdf = dirc/rlatc

  status = nfput(ncid_out,'UnitCell',transpose(dirc_netcdf), &
                dim_name1='dim3',dim_name2='dim3',startnf=(/1,1,nstep/),countnf=(/3,3,1/))

! DynamicAtomPositions
  status = nfput(ncid_out,"DynamicAtomPositions", netcdf_pos,startnf=(/1,1,nstep/),countnf=(/3,nions,1/))
  status = nfput(ncid_out,"DynamicAtomPositions%Description","scaled coordinates of dynamic atoms")

! DynamicAtomForces
  status = nfput(ncid_out,"DynamicAtomForces",netcdf_for,                                        &
          dim_name1='dim3',dim_name2='number_of_dynamic_atoms',dim_name3='number_ionic_steps', &
          startnf=(/1,1,nstep/),countnf=(/3,nions,1/)) 
  status= nfput(ncid_out,"DynamicAtomForces%Description","forces on dynamic atoms, ordered as DynamicAtomPositions") 
  status= nfput(ncid_out,"DynamicAtomForces%units","eV/Angstroem") 

! DynamicAtomVelocities
  status = nfput(ncid_out,"DynamicAtomVelocities",netcdf_vel,                                         &
          dim_name1='dim3',dim_name2='number_of_dynamic_atoms',dim_name3='number_ionic_steps', &
          startnf=(/1,1,nstep/),countnf=(/3,nions,1/))
  status = nfput(ncid_out,"DynamicAtomVelocities%Description","velocities of dynamic atom")
  status = nfput(ncid_out,"DynamicAtomVelocities%units","Angstroem/fs") 

! FermiLevel
  status = nfput(ncid_out,"FermiLevel",efermi,dim_name1='number_ionic_steps', & 
                 startnf=(/nstep/),countnf=(/1/)) 
  status = nfput(ncid_out,"FermiLevel%units","eV" )

! TotalEnergy
  status = nfput(ncid_out,"TotalEnergy",entot,dim_name1='number_ionic_steps', & 
                 startnf=(/nstep/),countnf=(/1/)) 
  if (status /= nf_noerr) call abort_calc(nconso, &
                         "update_netcdf_history -> nfput : error writing total-energy")
  status = nfput(ncid_out,"TotalEnergy%units","eV/unitcell" )
  status = nfput(ncid_out,"TotalEnergy%Description","E[rho] extrapolated to T=0" )

! TotalFreeEnergy
  status = nfput(ncid_out,"TotalFreeEnergy",enfree,dim_name1='number_ionic_steps', &
                 startnf=(/nstep/),countnf=(/1/))
  if (status /= nf_noerr) call abort_calc(nconso, &
                         "update_netcdf_history -> nfput : error writing total free-energy")
  status = nfput(ncid_out,"TotalFreeEnergy%units","eV/unitcell" )
  status = nfput(ncid_out,"TotalFreeEnergy%Description","Total Free Energy F[T] " )

! EvaluateTotalEnergy
  status = nfput(ncid_out,"EvaluateTotalEnergy", entot_all_xc,     & 
                  dim_name1='dim6',dim_name2='number_ionic_steps',        & 
                  startnf=(/1,nstep/),countnf=(/6,1/)) 
  status = nfput(ncid_out,"EvaluateTotalEnergy%units","eV/unitcell" )
  status = nfput(ncid_out,"EvaluateTotalEnergy%Description", & 
           "Evaluate total energy of density:&
          & PZ,VWN,PW91,PBE,revPBE,RPBE ")
  
! EvaluateExchangeEnergy
  status = nfput(ncid_out,"EvaluateExchangeEnergy",exc_and_corr_all(:,1),& 
                  dim_name1='dim6',dim_name2='number_ionic_steps',        & 
                  startnf=(/1,nstep/),countnf=(/6,1/)) 
  status = nfput(ncid_out,"EvaluateExchangeEnergy%units","eV/unitcell" )
  status = nfput(ncid_out,"EvaluateExchangeEnergy%Description", & 
           "Evaluate Exchange energy of density:&
           & PZ,VWN,PW91,PBE,revPBE,RPBE ") 

! EvaluateCorrelationEnergy
  status = nfput(ncid_out,"EvaluateCorrelationEnergy",exc_and_corr_all(:,2),&
                  dim_name1='dim6',dim_name2='number_ionic_steps',        &
                  startnf=(/1,nstep/),countnf=(/6,1/))
  status = nfput(ncid_out,"EvaluateCorrelationEnergy%units","eV/unitcell" )
  status = nfput(ncid_out,"EvaluateCorrelationEnergy%Description", &
           "Evaluate Correlation energy of density:&
           & PZ,VWN,PW91,PBE,revPBE,RPBE ")

! EigenValues
! actual number of k-points
  nkp1 = nkprun/nspin
  status = nfput(ncid_out,"EigenValues",eigen,                                &
             dim_name1 = 'number_of_bands',dim_name2 = 'number_of_spin',      &
             dim_name3 = 'number_IBZ_kpoints',dim_name4='number_ionic_steps', &
             startnf=(/1,1,1,nstep/),countnf=(/nbands,nspin,nkp1,1/))
  status = nfput(ncid_out,"EigenValues%Description","One electron eigenvalues")
  status = nfput(ncid_out,"EigenValues%units","eV" )

  status = nfput(ncid_out,"OccupationNumbers",occ,                            &
             dim_name1 = 'number_of_bands',dim_name2 = 'number_of_spin',      &
             dim_name3 = 'number_IBZ_kpoints',dim_name4='number_ionic_steps', &
             startnf=(/1,1,1,nstep/),countnf=(/nbands,nspin,nkp1,1/))
  status = nfput(ncid_out,"OccupationNumbers%Description", "occupation numbers")
  status = nfput(ncid_out,"OccupationNumbers%units", "0..1 (spin-pol)  0..2 (non spin-pol)" )

  status = nf_close(ncid_out)
  if (status /= nf_noerr) call abort_calc(nconso, &
                         "update_netcdf_history -> nf_close : error closing nc-outfile")

! let number_of_steps_in_netcdf points to the next frame
  number_of_steps_in_netcdf = number_of_steps_in_netcdf + 1


end subroutine update_netcdf_history

end module  ! update_netCDF_history_module

