!#####################################################
! This module provides a global interface 
! for accessing contextual information for
! the calculation
!#####################################################
module run_context
! ---------------------------------------------------------------------------
!    conventions: if user specifies (netCDF_input_file == netCDF_output_file) 
!                 (checks identity of names only) then netCDF_output_filename 
!                 is assigned a temporary name
!                 When run is complete, netCDF_output_file is copied to 
!                 netCDF_output_final_filename
!                 If user specifies only an input file, it is attempted to
!                 use it for output as well. This may fail, and it is user
!                 responsibility to assure that it is possible
! ---------------------------------------------------------------------------

   character*999 :: netCDF_input_filename         = "" ! accessed during run
   character*999 :: netCDF_output_filename        = "" ! written during run
   character*999 :: netCDF_output_final_filename  = "" ! final destination for netCDF_output
   logical       :: netCDF_make_restricted_copy        ! set in init_run_context
   logical       :: netCDF_copy_to_final_file          ! set in init_run_context
!
! -- ASCII for runtime monitoring --
!
   integer       :: nconso                       ! global Fortran unit for ASCII out
   logical       :: ASCII_output_filename_provided = .false.
   character*999 :: ASCII_output_filename = ""   ! used as filename root for parallel slaves
   character*(*),parameter :: ASCII_slaveoutput_dump = "/dev/null"

!
! -- faciliate externally controlled ion motion by script; the script is invoked --
! -- call system(ExternalIonMotion_script)                                       --
!
   character*999 :: ExternalIonMotion_script = "" ! set in setuop.F


! file name for swap files

  character*999 :: qrad_swap_filename = ""     ! make sure these strings are
  character*999 :: wf_swap_filename   = ""     ! always well defined, becauese
  character*999 :: vkb_swap_filename  = ""     ! are accessed in clexit()
 
! scratch directory name

  character*999 :: scratch_directory_name = "."       

! stop file name

  character*999 :: stop_file_name = "stop"     ! default name stop

! number of k-points per process
  integer, save :: number_kpoints_per_process

! controls for controlling output 
! initialized in setuop
  logical*4,save  :: output_wavefunction_to_netcdf  = .true.
  logical*4,save  :: output_chargedensity_to_netcdf = .true.
  logical*4,save  :: output_totalstress_to_netcdf = .false.
  integer,save    :: output_effpotential_to_netcdf = 0       ! Effective potential
  integer,save    :: output_elspotential_to_netcdf = 2       ! Electrostatic potential   

! controls for occupation number update 
! initialized in setuop
  logical*4,save  :: update_occupation_numbers  = .true.

! dacapo command line syntax

  character*(*),parameter :: dacapo_command_line_syntax = &
                   " netCDF_input_file [netCDF_output_file]"             // &
                   " [-innc netCDF_input_file] [-outnc netCDF_out_file]" // &
                   " [-out ASCII_output] [-scratch scratch_dir ] [-stop stop_file_name]" 

! command line arguments 
!  for the parallel program the master node initialize the command 
!  line arguments, and send (in msinit) the argument to the slave nodes, 
!  before the slave nodes calls init_run_context

  integer number_of_args
  character*999,allocatable :: str_args(:)

! revision and date string, output to stdout,  maintained by cvs/svn

  character*999 :: revision_date = "$Date: 2009-05-21 11:15:21 +0200 (Thu, 21 May 2009) $"
  character*999 :: revision_number = "$Rev: 472 $"

! the content of the enviroment variable DACAPOPATH 
! used then reading pseudo-potential files

  character*999 :: dacapo_path_string = ""   

contains

!############################################################
subroutine init_run_context(lmastr)
implicit none
!============================================================
! Parse the command line and initialize the data segment in
! this module - this is the only place that reads from stdin.
!============================================================
logical*4,intent(in) :: lmastr  ! only master node read command line arguments 

integer,parameter    :: str_len = 999
character(str_len)   :: str
integer              :: iarg, pid
integer              :: iargc_opsys_interface, getpid_opsys_interface 
character(str_len)   :: getenv_opsys_interface
logical              :: skip_next_arg
logical              :: found_innc, found_outnc

nconso = 6                                  ! bind ASCII output to conventional stdout unit
pid              = getpid_opsys_interface() ! retrieve process ID - used to create temp files

! for master node read all command-line arguments 
if (lmastr) then 

! write revision tag and date to stdout (not assigned yet to a possible out file) 
!  write(nconso,*) trim(revision_date)

  number_of_args   = iargc_opsys_interface()  ! retrieve number of cmd line arguments
! allocate str_args to hold all command line arguments 
  if (.not.allocated(str_args)) then 
    allocate(str_args(0:number_of_args))
  endif
  do iarg = 0,number_of_args 
    call getarg_opsys_interface(iarg,str)
    str_args(iarg) = str 
  enddo 
endif   

if (number_of_args .lt. 1) then
  netCDF_make_restricted_copy = .false.   ! avoid any final copying
  str  = str_args(0) 
  str = "Syntax error: number_of_args < 1; usage:" // trim(str) // dacapo_command_line_syntax
  call abort_calc(nconso, str)
endif 


! -----------------------------------------------------------
!     Scan for command line options, starting with  "-XXX"
! -----------------------------------------------------------

 
!write(nconso,*) 'run_context: number_of_args = ',number_of_args
!test: do iarg = 0, number_of_args               ! sweep command line
!          str = str_args(iarg) 
!          write(nconso,*) 'run_context: iarg= ',iarg,' str = ',trim(str)
!      enddo test    


found_innc    = .false.
found_outnc   = .false.
skip_next_arg = .false.

sweep: do iarg = 1, number_of_args               ! sweep command line (iarg == 0 is the dacapo executable)

         if (skip_next_arg) then                 ! this arg has been handled in conjunction 
             skip_next_arg = .false.             ! previous argument
             cycle
         endif

         str = str_args(iarg)
!        ---------------------------------------------------------------------------
         if (str(1:6) == "-innc ") then          ! input netCDF filename is provided as option
           if (iarg == number_of_args) then      ! bump out, if no filename 
              str = "Syntax error:  -innc specified, but no file found; " //&
                    "usage: "// trim(str_args(0)) // dacapo_command_line_syntax
              call abort_calc(nconso, str)
           endif
           netCDF_input_filename  = str_args(iarg+1) 
           found_innc             = .true.
           skip_next_arg          = .true.
           cycle 
         endif
!        ---------------------------------------------------------------------------
         if (str(1:7) == "-outnc ") then         ! output netCDF filename is provided as option
           if (iarg == number_of_args) then      ! bump out, if no filename 
              str = "Syntax error:  -outnc specified, but no file found; " //&
                    "usage: "// trim(str_args(0)) // dacapo_command_line_syntax
              call abort_calc(nconso, str)
           endif
           netCDF_output_filename  = str_args(iarg+1) 
           found_outnc             = .true.
           skip_next_arg           = .true.
           cycle 
         endif
!        ---------------------------------------------------------------------------
         if (str(1:5) == "-out ") then           ! ASCII output filename is provided
           if (iarg == number_of_args) then      ! bump out, if no filename 
              str = "Syntax error:  -out  specified, but no file found; " //&
                    "usage: "// trim(str_args(0)) // dacapo_command_line_syntax
              call abort_calc(nconso, str)
           endif
           ASCII_output_filename = str_args(iarg+1) 
           ASCII_output_filename_provided = .true.  
           skip_next_arg = .true.
           cycle 
         endif
!        ---------------------------------------------------------------------------
         if (str(1:9) == "-scratch ") then       ! scratch directory is provided
           if (iarg == number_of_args) then      ! bump out, if no dirname 
              str = "Syntax error:  -scratch  specified, but no directory found; " //&
                    "usage: "// trim(str_args(0)) // dacapo_command_line_syntax
              call abort_calc(nconso, str)
           endif
           scratch_directory_name = str_args(iarg+1)
           skip_next_arg = .true.
           cycle 
         endif
!        ---------------------------------------------------------------------------
         if (str(1:6) == "-stop ") then          ! stop file name is provided
           if (iarg == number_of_args) then      ! bump out, if no filename
              str = "Syntax error:  -stop  specified, but no file found; " //&
                     "usage: "// trim(str_args(0)) // dacapo_command_line_syntax
              call abort_calc(nconso, str)
           endif
           stop_file_name = str_args(iarg+1)
           skip_next_arg = .true.
           cycle
         endif

!        ---------------------------------------------------------------------------
!        all non-options (not starting with "-") are considered as file arguments
!        -innc/-outnc precedes implicit file arguments
!        first  encountered non-option -> netCDF_input_filename  (if not already set)
!        second encountered non-option -> netCDF_output_filename (if not already set)
!        do not parse/complain unknown options (MPICH prepends options to the command line)
!        ---------------------------------------------------------------------------

         if (str(1:1) /= "-") then
            
            if (found_innc .eqv. .false.) then   
               netCDF_input_filename = str
               found_innc            = .true.
               cycle             
            endif
            
            if (found_outnc .eqv. .false.) then
               netCDF_output_filename = str
               found_outnc            = .true.
               cycle             
            endif
            
         endif

       enddo sweep


! write(*,*) "netCDF_input_filename  =", netCDF_input_filename
! write(*,*) "netCDF_output_filename =", netCDF_output_filename

! -----------------------------------------------------------
! --- get netCDF I/O filenames; no defaults               ---
!     netCDF_input_filename  = arg no 1
!     netCDF_output_filename = arg no 2, if it isn't a runtime option (starts with "-")
!                                        else netCDF_output_filename = netCDF_input_filename
!     The check of file idendity of these arguments are primitive
!     If found identical, netCDF_output is directed to a temporary file
!     Block filenames starting with character "-"; the user has probably  
!     forgotten to provide 2 files, so echo the command line syntax
! -----------------------------------------------------------

if (.not.(found_innc .or. found_outnc)) then
   str = "Syntax error: no file arguments resolved (need <= 1) ; usage: " &
         // trim(str_args(0)) // dacapo_command_line_syntax
   call abort_calc(nconso, str)
endif 

netCDF_copy_to_final_file = .false.   

if (found_innc .and. found_outnc) then              ! user provided both innc/outnc files

  netCDF_output_final_filename = netCDF_output_filename
  if (netCDF_input_filename == netCDF_output_filename) then   ! use temp. outfile
      write(netCDF_output_filename,'(2a,i10.10)') trim(scratch_directory_name), "/tmp.", pid 
      netCDF_copy_to_final_file = .true.        
  endif
  netCDF_make_restricted_copy   = .true.

elseif  (found_innc.and.(.not.found_outnc)) then    ! use innc as outnc

  netCDF_output_filename       = netCDF_input_filename
  netCDF_output_final_filename = netCDF_input_filename  ! render defined for safety
  netCDF_make_restricted_copy = .false. 

else                           ! do not attempt to use outnc as innc

  str = "Syntax error: no input file resolved; usage: " &
        // trim(str_args(0)) // dacapo_command_line_syntax
  call abort_calc(nconso, str)
  
endif       

! -----------------------------------------------------------
!      Initialize scratch filenames (qrad, wf, vkb arrays)
!      unique filename is created from the PID of the running program
! -----------------------------------------------------------

  write (qrad_swap_filename,'(2a,i10.10)') trim(scratch_directory_name), "/qrad_swap.",pid
  write (wf_swap_filename,  '(2a,i10.10)') trim(scratch_directory_name), "/wf_swap.",  pid
  write (vkb_swap_filename, '(2a,i10.10)') trim(scratch_directory_name), "/vkb_swap.", pid

! -----------------------------------------------------------
!      Get content of enviroment variable DACAPOPATH
!      call getenv directly here
!      temporary because getenv does not seem to work 
!      then called from a subroutine!
! -----------------------------------------------------------
  call getenv("DACAPOPATH",dacapo_path_string)

return
 
end subroutine init_run_context

!#####################################################

subroutine report_FILE_IO()
! ------------------------------------------------------
!  Report module status for file IO.
!  module must be in initialized state
! ------------------------------------------------------
implicit none 

write(nconso, 402) "Reading netCDF formatted input from file:", trim(netCDF_input_filename)

if (netCDF_copy_to_final_file) then
   write(nconso, 402) "During execution, a temporary netCDF output file is used:", &
   trim(netCDF_output_filename)
   write(nconso, 402) "Final file destination for netCDF output:", &
   trim(netCDF_output_final_filename)
else
   write(nconso, 402) "Writing netCDF formatted output to file", trim(netCDF_output_filename)
endif 

if (.not.netCDF_make_restricted_copy) &
   write(nconso, 400) "Attempting to use same file for netCDF input/output"

write(nconso, 402) "Stopfilename:", trim(stop_file_name)

write(nconso,*) ' '

400  format(1x,'FILE_IO: ',a)
402  format(1x,'FILE_IO: ',a,1x,a) 

end subroutine report_FILE_IO     

!#####################################################


end module
