! module for evaluating non-local projectors 
#define TCALBEC1        34
#define THPSI1          33 

module non_local_projectors 

! module data 

! public section

! luse_rs_proj = true if the NetCDF variable USE_RS_PROJ is set
  logical, save :: luse_rs_proj = .false.

! number of grid points around each atom for each projector
! ngridpts
integer, allocatable, save :: ngridpts(:)       ! nions
! from local grid around atom to fft grid
integer,allocatable, save  :: index_rs2fft(:,:) ! max(ngridpts),nions

! real space projector
complex*16, allocatable, save :: vkb_rs(:,:)    ! max(ngridpts),nkb

! reciprocal space projectors 
real*8    , allocatable, save :: vkb(:,:,:,:)     ! nrplwv_global,nkbpmaxatom,nspec,nkpmem 

! phase factor (for use with reciprocal space projectors 
complex*16, allocatable, save :: skb(:,:,:)       ! nrplwv_global,nions,nkpmem


! private section

! number of q-vectors (descrite q-mesh) 
integer, private, save :: number_q_vectors 

! Gmax and gamma,  
real*8, private, save  :: gmax,gamma

! q-vectors 
real*8, allocatable,private, save :: qvectors(:)

! R0/Rc margin (default 1.3) 
real*8,  private, save            :: r02rc

! some constants
real *8, private :: pi,fpi,tpi,bohr
parameter(pi=3.14159265358979d0,tpi=2.0*pi,fpi=4.0*pi,bohr=0.5291772d0) 
real*8,private   :: hsqdtm
parameter(hsqdtm=3.810033d0)         

contains

subroutine init_rs_nl(enmax,betar,r,rab,ndm,nbrx,npsx,nbeta,nkb,nh,lll,kkbeta,rc,nspec,idebug,recc,dirc,volc,ngx,ngy,ngz, & 
                      nions,nionsp,posion)
!
! initialize the real space nonlocal projectors 
! input data
!   enmax                      ! planewave cutoff
!   betar(0:ndm,nbrx,npsx)     ! beta function (r*beta) on real-space radial grid
!   r(0:ndm,npsx)              ! real-space radial grid 
!   rab(0:ndm,npsx)            ! real-space radial grid 
!   nbeta(npsx)                ! number beta functions for each species
!   nkb                        ! total number of projectors
!   nh(nspec)                  ! total number of projectors for each type of atom
!   lll(nbrx,npsx)             ! angular momentum for each beta function   
!   kkbeta(npsx)               ! actual number of real mesh points in in radial grid for betar
!   rc(lmaxx+1,NPSX)           ! cutoff radius (in r0) 
!   idebug                     ! debuging level
!   recc,dirc,volc             ! real and reciprocal unitcell and volume (A^3)
!   ngx,ngy,ngz                ! 3D Fourier grid
!   nspec,nions,nionsp         ! number of different kinds of atoms,total number of atoms and number of each kind of atom
!   posion                     ! position of ions in scaled coordinates

!
  use run_context
  implicit none
  real*8, intent(in) :: enmax
  integer,intent(in) :: ndm,nbrx,npsx
  real*8, intent(in) :: betar(0:ndm,nbrx,npsx) 
  real*8, intent(in) :: r(0:ndm,npsx),rab(0:ndm,npsx)
  integer,intent(in) :: nbeta(npsx),lll(nbrx,npsx),kkbeta(npsx),nspec,nkb,nh(nspec) 
  real*8 ,intent(in) :: rc(3,npsx)  ! lmaxx+1,npsx
  integer,intent(in) :: idebug
  real*8, intent(in) :: recc(3,3),dirc(3,3),volc
  integer,intent(in) :: ngx,ngy,ngz
  integer,intent(in) :: nions,nionsp(nspec)
  real*8 ,intent(in) :: posion(3,nions,nspec)  


  real*8, allocatable :: A(:,:),chi(:),fchi(:,:,:),jl(:),jl2(:),beta(:),fbeta(:)  
  integer nq_gmax,msh
  integer ig,i,np,nb,l,ir
  real*8, parameter  ::  pref = fpi,prefi = pref/(tpi**3)
  real*8             ::  R0(nspec),h,simps ,g1
  integer            ::  nmax(3),nclose(3),n1,n2,n3,npoints ,nsp,ni
  integer            ::  iproj,loop,n11,n22,n33,nion,maxgrid,iproj_start
  real*8             ::  r1,r2,r3,rg(3),dist,ylm(9),rgbeta,gridfactor
  real*8             ::  ra(0:ndm,npsx),raba(0:ndm,npsx),betara(0:ndm,nbrx,npsx)  ! r() converted to A
  complex*16         ::  cfac


  ra(:,:)   = bohr*r(:,:)
  raba(:,:) = bohr*rab(:,:)
  betara(:,:,:) = betar(:,:,:)/bohr**2  ! betar is r*beta(r)

  ! define cutoff radius R0 for real space region around each atom 
  do nsp = 1,nspec 
    R0(nsp) = r02rc* maxval(rc(1:3,nsp))*bohr
  enddo
  gmax  = sqrt(enmax/hsqdtm)
  call getcutoff(recc,ngx,ngy,ngz,g1) 
  gamma = g1 - gmax
  number_q_vectors = gamma * 10
  number_q_vectors = (number_q_vectors/2)*2 + 1

  gridfactor = prefi * sqrt(volc)   ! box normalization

! define q vectors
  if (allocated(qvectors)) deallocate(qvectors) 
  allocate(qvectors(number_q_vectors))
  do i = 1,number_q_vectors
    qvectors(i) = dble(i-1)/dble(number_q_vectors-1)*gamma 
  enddo
  allocate(A(number_q_vectors,number_q_vectors),chi(number_q_vectors))
  allocate(fchi(number_q_vectors,maxval(nbeta(1:nspec)),nspec))
  allocate(jl(ndm),jl2(number_q_vectors),beta(ndm),fbeta(ndm))

! find nq_gmax 
  do nq_gmax = number_q_vectors,1,-1
    if (qvectors(nq_gmax).le.gmax) exit 
  enddo 
  nq_gmax = (nq_gmax/2)*2 + 1

  write(nconso,*) 'init_rs_nl : Fourier filter with gmax,gamma,number_q_vectors,nq_gmax ', & 
                  gmax,gamma,number_q_vectors,nq_gmax


! loop over all species 
  do nsp = 1,nspec 

!  loop over beta functions 
   do nb = 1,nbeta(nsp)
     l = lll(nb,nsp) 
     msh = kkbeta(nsp) 
     write(nconso,*) 'init_rs_nl : Fourier transform betar (nsp,nb,l)',nsp,nb,l

!    do i = 2,kkbeta(nsp)
!      write(nconso,100) 'init_rs_nl: betar',nsp,nb,i,ra(i,nsp),betara(i,nb,nsp)/ra(i,nsp)
!    enddo

     do ig = 1,number_q_vectors
       call uflush(nconso)
       call bess(qvectors(ig),l+1,msh,ra(1,nsp),jl)   ! r(0)? 
       jl(1:msh) = jl(1:msh)*betara(1:msh,nb,nsp)*ra(1:msh,nsp)
       ! call radlg(msh,jl,ra(1,nsp),rlog(nsp),h) 
       call radlg1(msh,jl,raba(1,nsp),h)
       chi(ig) = pref*h 
     enddo
!    do ig = 1,number_q_vectors
!      write(nconso,100) 'init_rs_nl: chi ',nsp,nb,ig,qvectors(ig), chi(ig)
!    enddo

!    setup A matrix
     call calculateA(R0(nsp),l,A)

!    Fourier filter chi 
     fchi(:,nb,nsp) = chi(:)
     call four_filter(nq_gmax,fchi(1,nb,nsp),A) 
!    do ig = 1,number_q_vectors
!      write(nconso,100) 'init_rs_nl: fchi ',nsp,nb,ig,qvectors(ig), fchi(ig,nb,nsp)
!    enddo
100  format(1x,a20,1x,i1,1x,i2,1x,i4,1x,2(f18.9,1x))

!    inverse Fourier transform chi and fchi(debug only) 
     h = qvectors(2)-qvectors(1)
     if (idebug.gt.5) then 
       do ir = 1,msh 
        call bess(ra(ir,nsp),l+1,number_q_vectors,qvectors,jl2)
        jl2(1:number_q_vectors) = jl2(1:number_q_vectors)*chi(1:number_q_vectors)*qvectors(1:number_q_vectors)**2
        beta(ir) = gridfactor*simps(number_q_vectors,jl2,h)
        call bess(ra(ir,nsp),l+1,number_q_vectors,qvectors,jl2)
        jl2(1:number_q_vectors) = jl2(1:number_q_vectors)*fchi(1:number_q_vectors,nb,nsp)*qvectors(1:number_q_vectors)**2
        fbeta(ir) = gridfactor*simps(number_q_vectors,jl2,h)
        write(nconso,110) 'init_rs_nl : rfbeta ',nsp,nb,ir,ra(ir,nsp),beta(ir),fbeta(ir)
       enddo
     endif
110  format(1x,a20,1x,i1,1x,i2,1x,i4,1x,3(f18.9,1x))

   enddo  ! nb 
 enddo    ! nspec


 ! do this loop twice, first time to define ngridpts,
 ! index_rs2fft and vkb_rs is then allocated

 if (allocated(vkb_rs))       deallocate(vkb_rs)
 if (allocated(index_rs2fft)) deallocate(index_rs2fft)

 if (.not. allocated(ngridpts)) allocate(ngridpts(nions))
 ngridpts(:) = 0                                      

 do loop = 1,2

   iproj_start = 1
   nion = 1
   do nsp = 1,nspec  

       ! find number of points to search |b_i|/(2pi) * R0 * ng_i
       nmax(1) = int( sqrt(sum(recc(:,1)**2))* R0(nsp) * ngx + 0.5)
       nmax(2) = int( sqrt(sum(recc(:,2)**2))* R0(nsp) * ngy + 0.5)
       nmax(3) = int( sqrt(sum(recc(:,3)**2))* R0(nsp) * ngz + 0.5)

       do ni = 1,nionsp(nsp)

           ! find the grid point closets to the ions
           nclose(1) = nint(posion(1,ni,nsp)*ngx)
           nclose(2) = nint(posion(2,ni,nsp)*ngy)
           nclose(3) = nint(posion(3,ni,nsp)*ngz)

           np = 0   ! number of grid points found

           ! loop over realspace grid points
           do n3 = nclose(3)-nmax(3),nclose(3)+nmax(3)
             do n2 = nclose(2)-nmax(2),nclose(2)+nmax(2)
               do n1 = nclose(1)-nmax(1),nclose(1)+nmax(1)
                 ! distance from this grid-point to the ion
                 r1 =  dble(n1)/dble(ngx) - posion(1,ni,nsp)
                 r2 =  dble(n2)/dble(ngy) - posion(2,ni,nsp)
                 r3 =  dble(n3)/dble(ngz) - posion(3,ni,nsp)
                 rg(1:3) = dirc(1,:)*r1 + dirc(2,:)*r2 + dirc(3,:)*r3
                 dist = sqrt(sum(rg(:)**2))
                 if (dist<R0(nsp)) then
   
                   np = np + 1
                   ! find grid points, by folding into central unitcell
                   n11 = modulo(n1,ngx)  
                   n22 = modulo(n2,ngy)  
                   n33 = modulo(n3,ngz)
                      
                   if (loop==2) then  
                     ! set index array 
                     index_rs2fft(np,nion) = 1+n11 + n22*ngx + n33*ngx*ngy

                     iproj = iproj_start
                     do nb = 1,nbeta(nsp)

                       l = lll(nb,nsp)   

                       ! get spherical harmonics at this grid point in ylm
                       call ylmg((l+1)**2,rg,dist**2,(l+1)**2, ylm)

                       ! inverse Fourier transform fchi to the fchi(r) at the point (r-r0) (rg)
                       call bess(dist,l+1,number_q_vectors,qvectors,jl2)
                       jl2(1:number_q_vectors) = jl2(1:number_q_vectors)*fchi(1:number_q_vectors,nb,nsp) & 
                                                 *qvectors(1:number_q_vectors)**2
                       rgbeta = gridfactor*simps(number_q_vectors,jl2,h)
                       ! cfac = (0.0d0,-1.0d0)**l
                       cfac = 1.0d0
                  
                       if (l==0) then  
                         vkb_rs(np,iproj) = rgbeta*ylm(1)      *cfac
                         iproj = iproj + 1
                       elseif (l==1) then   
                         vkb_rs(np,iproj  ) = rgbeta*ylm(2)    *cfac
                         vkb_rs(np,iproj+1) = rgbeta*ylm(3)    *cfac
                         vkb_rs(np,iproj+2) = rgbeta*ylm(4)    *cfac
                         iproj = iproj + 3
                       else 
                         vkb_rs(np,iproj  ) = rgbeta*ylm(5)    *cfac
                         vkb_rs(np,iproj+1) = rgbeta*ylm(6)    *cfac
                         vkb_rs(np,iproj+2) = rgbeta*ylm(7)    *cfac
                         vkb_rs(np,iproj+3) = rgbeta*ylm(8)    *cfac
                         vkb_rs(np,iproj+4) = rgbeta*ylm(9)    *cfac
                         iproj = iproj + 5
                       endif
                     enddo   ! nb
                   endif     ! loop==2 
     
                 endif
               enddo
             enddo
           enddo
           if (loop==1) then 
             write(nconso,120) nion,np,R0(nsp)
             120 format('RSPP: number of points around atom ',i3,' : ',i5,' R0 = ',f8.4)
             ngridpts(nion) = np
           endif
           iproj_start = iproj_start + nh(nsp)
           nion = nion + 1
       enddo
   enddo   ! np = 1,nspec
   write(*,*) 'rs check ',nkb,iproj
   if (loop.eq.1) then
      maxgrid = maxval(ngridpts(:))
      allocate(index_rs2fft(maxgrid,nions))
      allocate(vkb_rs(maxgrid,nkb))
   endif
 enddo   ! loop                                                                       

 ! deallocate 
 deallocate(A,chi,fchi,jl2,beta,fbeta) 

       
end subroutine init_rs_nl


subroutine four_filter(nq_gmax,chi,A) 
! --------------------------------------------------------------------
! The Fourier components of chi is ajusted in the interval 
! [nq_gmax:number_q_vectors] following  
! R.D King-Smith, M.C. Payne and J.S Lin, PRB 44, page 13063 (1991), 
! A is the matrix A(q,q') in Eq. 8.   
! The matrix equation (M chi = b) is built so that the integrals 
! [0,Gmax],[Gmax,Gamma] uses Simpson integration (Lennart Bengtson's ffilt)
! nq_gmax should be odd. 
  use run_context
  implicit none
  integer, intent(in)    :: nq_gmax
  real*8,  intent(inout) :: chi(number_q_vectors)
  real*8,  intent(in)    :: A(number_q_vectors,number_q_vectors)

  real*8, parameter :: pi_half = pi/2.0d0
  real*8 b(number_q_vectors),M(number_q_vectors,number_q_vectors)
  real*8 h
  integer q,q1,nq_free,info 
  real*8  ipiv(number_q_vectors),f1(number_q_vectors)
  real *8 simps


! number of free Fourier components in the interval gmax,gamma
  nq_free = number_q_vectors-nq_gmax
  h = qvectors(2)-qvectors(1)

! calculate left had side of Eq. 8 Int_0^Gmax(A(q,q1)chi(q1)dq1
  do q = 1,nq_free

    do q1 = 1, nq_gmax
      f1(q1) = A(q+nq_gmax,q1)*chi(q1)
    enddo 
    b(q) = simps(nq_gmax,f1,h)
  enddo

! setup matrix M 
  M(:,:) = 0.0d0
! first diagonal elements pi/2*q^2 
  do q = 1,nq_free
    M(q,q) = qvectors(q+nq_gmax)**2 * pi_half
  enddo 

! built integrals gmax,gamma using Simpson integration
! first term nq_gmax is fixed so it is added to b
  do q = 1,nq_free
     b(q) = b(q) + A(q+nq_gmax,nq_gmax)*chi(nq_gmax)*(h/3.0d0)
  enddo

! insert even indexes in M (Simpson factor 4/3)
  do q1 = 1,nq_free-1,2
    do q = 1,nq_free 
        M(q,q1) = M(q,q1) - A(nq_gmax+q,nq_gmax+q1)*(h*4.0d0/3.0d0)
    enddo 
  enddo

! insert odd indexes in M (Simpson factor 2/3)
  do q1 = 2,nq_free-2,2
    do q = 1,nq_free
        M(q,q1) = M(q,q1) - A(nq_gmax+q,nq_gmax+q1)*(h*2.0d0/3.0d0)
    enddo
  enddo     
! last index 
  do q = 1,nq_free
     M(q,nq_free) = M(q,nq_free)-A(nq_gmax+q,number_q_vectors)*(h/3.0d0)
  enddo        

! Solve linear equation system
!  call dgesv(nq_free,1,M,number_q_vectors,ipiv,b,number_q_vectors,info) 
! if (info.ne.0) then 
!   write(nconso,*) 'four_filter: dgesv error ',info
!   call clexit(nconso)
! endif
  
  do q = 1,nq_free
   chi(nq_gmax+q) = b(q) 
  enddo

  end subroutine four_filter

  
  subroutine calculateA(R0,l,A)
! -------------------------------------------------------------------
! calculate the matrix A (to be used in Eq. 8)
! A(q,q') = q^2*q'^2 * Int_0^R0 jl(qr) r^2 jl(q'r') dr

  implicit none
  real*8,  intent(in) :: R0
  integer, intent(in) :: l 
  real*8,  intent(out):: A(number_q_vectors,number_q_vectors)

  integer,parameter :: nr = 1001 ! r mesh
  real*8  r(nr),h,jl(nr,number_q_vectors),u(nr) 
  integer i,iq,iq1,ir 
  real*8  simps

  ! setup real space mesh   
  do i = 1,nr 
    r(i) = dble(i-1)/dble(nr-1) * R0 
  enddo 
  h = r(2) - r(1) 

  ! calculate spherical Bessel functions 
  do iq = 1,number_q_vectors 
    call sphbess(qvectors(iq),l,nr,r,jl(1,iq))
  enddo 

  do iq = 1,number_q_vectors
   do iq1 = 1,number_q_vectors
     do ir = 1,nr 
       u(ir) = jl(ir,iq)*jl(ir,iq1)*r(ir)**2
     enddo 
     A(iq,iq1) = qvectors(iq)**2 * qvectors(iq1)**2 * simps(nr,u,h) 
   enddo 
 enddo

 end subroutine calculateA

 subroutine sphbess(q,l,mmax,r,jl)
 !-------------------------------------------------------------------------
 !     calculates spherical bessel functions  j_l(qr)
 !     with approximately 11 digits of precision / Lennart
 
      integer l,mmax
      real*8  q,r(mmax),jl(mmax)
      integer ir
      real*8  qr
 
      if (l.eq.0) then                      !   s  part
         do ir=1,mmax
            qr = q*r(ir)
            if (qr.gt.1d-4) then
               jl(ir) = sin(qr)/qr
            else
               jl(ir) = 1.0d0 - qr**2*(1d0/6d0)
            endif
         enddo
      elseif (l.eq.1) then                   !   p  part
         do ir=1,mmax
            qr = q*r(ir)
            if (qr.gt.0.01d0) then
               jl(ir) = (sin(qr)/qr-cos(qr))/qr
            else
               jl(ir) = &
                  qr*(1d0/3d0)*(1d0-0.1d0*qr**2*(1d0-(1d0/28d0)*qr**2))
            endif
         enddo
      elseif (l.eq.2) then                   !   d  part
         do ir=1,mmax
            qr = q*r(ir)
            if (qr.gt.0.1d0) then
               jl(ir) = ((3d0-qr**2)*sin(qr)-3d0*qr*cos(qr))/qr**3
            else
               jl(ir) = qr**2*(1d0/15d0)*(1d0-(1d0/14d0)*qr**2* &
                  (1d0-(1d0/36d0)*qr**2))
            endif
         enddo
      else
         stop 'Higher l not implemented in sphbess'
      endif

  end subroutine sphbess

 subroutine cal_rs_becp(psi,nrplwv,nbands,nplwkp,ngxs,ngys,ngzs,nkb,nkbmax,nkbtona,nh,ipwpad,nions,nspec,nionsp,becp,timer)
 !-----------------------------------------------------------------------------------
 ! 
 !  calculate overlaps of real space nonlocal projector (vkb_rs) with wavefunctions in 
 !  psi :  becp(1:nkb,1:nbands,nkp) = <beta(1:nkb)|psi(1:nbands,nkp))
 !  
 !  psi(nrplwv,nbands)       ! wavefunctions actual size : nplwkp,nbands
 !  nkptona                  ! proj. number -> atom number
 !  nh(nspec)                ! number of projectors for this type of atom
 !  ipwpad                   ! index array,  wavefunction array -> fft grid
 !  becp(nkbmax,nbands,nkp)  ! output overlab
 !  
 !  module data used: 
 !  vkb_rs
 !  index_rs2fft
 !  ngridpts
 implicit none
 integer  , intent(in)  ::  nrplwv,nbands,nplwkp,ngxs,ngys,ngzs,nkb,nkbmax,nspec 
 complex*16,intent(in)  ::  psi(*)
 integer  , intent(in)  ::  nkbtona(nkbmax),nh(nspec) 
 integer  , intent(in)  ::  ipwpad(nrplwv)
 integer  , intent(in)  ::  nions,nionsp(nspec) 
 complex*16,intent(out) ::  becp(nkbmax,nbands)
 real*8 , intent(inout) ::  timer(*)

 
 complex*16 cwork(ngxs*ngys*ngzs)
 complex*16, allocatable :: psiwork(:,:)    ! max(ngridpts(:),nions)
 complex*16  betapsi(nkbmax)
 integer i,nbnd,na,n,np
 real*8  rinplw
 integer m,l,lda,ldb,ldc,nproj,nsp,ni
 complex*16  alpha,beta
 real*4      time(2)

 allocate(psiwork(maxval(ngridpts(:)),nions))

 rinplw = 1.0d0/(dble(ngxs*ngys*ngzs))

 becp(:,:) = (0.0d0,0.0d0)
 do nbnd = 1,nbands 

   call uttime(time)
   timer(TCALBEC1)=timer(TCALBEC1)-time(1)           

   call uttime(time)
   timer(THPSI1)=timer(THPSI1)-time(1)           

   do na = 1,nions
     do n = 1,ngridpts(na)
       psiwork(n,na) = psi(index_rs2fft(n,na))*rinplw
     enddo
   enddo

   call uttime(time)
   timer(TCALBEC1)=timer(TCALBEC1)+time(1)           

 !   do np = 1,nkb
 !       na = nkbtona(np) 
 !       do n = 1,ngridpts(na)
 !            becp(np,nbnd) = becp(np,nbnd) + conjg(vkb_rs(n,np))*psiwork(n,na)
 !       enddo
 !   enddo

      nproj = 1 
      na    = 1
      do nsp = 1,nspec 
       do ni = 1,nionsp(nsp) 
       
             m   = ngridpts(na)
             lda = size(vkb_rs(:,1))
             l   = nh(nsp)
  
             ldb = size(psiwork(:,1))
             n   = 1
  
             alpha = (1.0d0,0.0d0) 
             beta  = (0.0d0,0.0d0)
             ldc = nkbmax
             betapsi(:) = 0.0d0
             call zgemm('C','N',l,n,m,alpha,vkb_rs(1,nproj),lda,psiwork(1,na),ldb,beta,betapsi,ldc) 
             do i = 1,nh(nsp)
               becp(nproj+i-1,nbnd) = becp(nproj+i-1,nbnd) + betapsi(i)
             enddo
             
             na = na + 1
             nproj = nproj + nh(nsp)
       enddo ! ni 
      enddo  ! nsp

  enddo    ! nbnd

 deallocate(psiwork)

 end subroutine cal_rs_becp
 subroutine set_luse_rs_proj
 !-------------------------------------------------------------------------
 ! set the module variable luse_rs_proj to true if the NetCDF variable 
 ! USE_RS_PROJ is defined
 ! Set margin R0/Rc from integer attribute R02Rc_Margin
 use netcdfinterface 
 use run_context
 implicit none 
 integer status,ncid
 real*8  margin
 character*20 value
 logical,save :: init = .false.

 if (init) return
 init = .true.

!parallel distribution of set_luse_rs_proj  should be done
 luse_rs_proj = .false.
 return

 status =  nf_open(netCDF_input_filename, NF_NOWRITE, ncid)
 if (status /= nf_noerr) call abort_calc(nconso, "set_luse_rs_proj -> nf_open:error opening nc-file")
                                                                                    
 status = nfget(ncid,"USE_RS_PROJ",value) 
 if (status==nf_noerr) then 
    luse_rs_proj = .true. 
    write(nconso,*) 'RSPP: use real-space non-local projectors'
    ! get R0/Rc margin from
    r02rc = 1.3  ! default
    status = nfget(ncid,"USE_RS_PROJ%R02Rc_Margin",margin)
    if (status==nf_noerr) then
      r02rc = margin
    endif                                  
    write(nconso,*) 'RSPP: using R0/Rc = ',r02rc 
 endif 
 status = nf_close(ncid)  

 end subroutine set_luse_rs_proj

 subroutine getcutoff(recc,ngx,ngy,ngz,gamma)
 !----------------------------------------------------------------------
 ! find the non-zero reciprocal lattice vector closets to zero 
 ! (gamma)

      implicit none
      real*8  recc(3,3), gamma
      integer ngx,ngy,ngz

      integer i,j,k,m
      integer grid(3) 
      real*8 q(3,3),qv(3),qq

      grid(:) = (/ngx,ngy,ngz/)
      do j = 1, 3
         do i = 1, 3
            q(i,j) = recc(i,j)*grid(i)
         enddo
      enddo
   ! Find the non-zero reciprocal lattice vector closest to zero
      gamma = 1d10
      do k = -5,5
         do j = -5,5
            do i = -5,5
               if (k.ne.0.or.j.ne.0.or.i.ne.0) then
                  do m = 1, 3
                     qv(m) = i*q(1,m)+j*q(2,m)+k*q(3,m)
                  enddo
                  qq = sqrt(qv(1)**2+qv(2)**2+qv(3)**2)
                  if (qq.lt.gamma) gamma = qq
               endif
            enddo
         enddo
      enddo
      return
   end subroutine                                                                    


 end module non_local_projectors
