      module kp_analysis

      contains 

!=======================================================================
      subroutine kptaid(nconso)
      write(nconso,*) '@(#)kptana.F	1.6 7/1/99'

      end subroutine kptaid
!=======================================================================
      subroutine kptana(dirc,numlat,nlatma,&
                        numspa,nspama,linvad,nkprun,vkpt,wtkpt,&
                        gkp123,nkpnum,nkpibz,nkpunf,lkpinv,nspin,&
                        kspin,idebug,nkpmem,lmastr,nconso)
!=======================================================================
! Analyse the set of k-points
!=======================================================================

      use generate_kpoints
      implicit none
      real*8 dirc(3,3)
      integer numlat,nlatma(3,3,numlat),numspa
      integer nspama(3,3,numspa)
      logical*4 linvad(numspa)
      integer nkprun
      real*8, pointer   :: vkpt(:,:,:), wtkpt(:,:)
      real*8, pointer   :: gkp123(:,:)
      integer,pointer   :: kspin(:,:)
      integer nkpnum
      integer,pointer   ::  nkpibz(:),nkpunf(:,:,:)
      logical*4,pointer ::  lkpinv(:)
      integer nspin
      integer idebug, nkpmem,nconso
      logical*4 lmastr
!=======================================================================
! local work array
!=======================================================================
      integer  nstmax
      parameter(nstmax=100)
      integer nstave(3,nstmax)
      integer nstasc(nstmax)
      real*8 rstasc(nstmax)
      real*8 rsumsc(nstmax)
      logical*4 iany
      logical*4 ioutf
      real*8    acc,epsilo
      parameter(acc=1.0d-5)
      integer   i,j,numsta
!=======================================================================
! nkpnum: The actual number of k-points in the entire 1st BZ
! nkprun: The max AND actual number of k-points in the IBZ
! nkpunf(1..nkpnum): The matrices bringing the k-points to the IBZ
!=======================================================================
! Read in the k-points or generate from a stardard set
! Monkhorst-Pack or Chadi-Cohen
! Also read how many k-points should be in memory
!=======================================================================
      call generate_bz_kpoints(nkpnum,gkp123,nkpmem,idebug,lmastr)
!=======================================================================
! fold in the k-points to the IBZ
!=======================================================================
! output:
! vkpt(1..3,1..nkprun):    The run-time k-points (in the IBZ)
! wtkpt(1..nkprun):        Weights of the run-time k-points
! nkpibz(x=1..nkpnum):     The index of which IBZ k-point corresponds
!                          to the x'th k-point in the 1st Brillouin zone
! nkpunf(3,3,x=1..nkpnum): The matrix bringing the nkpibz(x)'th k-point
!                          from the IBZ to the x'th k-point in the
!                          1st Brillouin zone
!=======================================================================
      allocate(nkpibz(nkpnum))
      allocate(nkpunf(3,3,nkpnum))
      allocate(lkpinv(nkpnum))
      call kptfol(nkprun,vkpt,wtkpt,nkpnum,gkp123,nkpibz,nkpunf,lkpinv,&
                  numspa,nspama,linvad,idebug,nconso,nspin,kspin)
!=======================================================================
! calculate the stars
!=======================================================================
! output:
! nstave(1..3,1..numsta): one vector from the m-th star
! rstasc(1..3,1..numsta): the length of the vectors in the m-th star
!=======================================================================
      call kptsta(dirc,numlat,nlatma,nstmax,numsta,nstave,rstasc)
!=======================================================================
! calculate the sum over k-points of the symmetriced plane waves
!=======================================================================
! output:
! rsumsr: the sum
! nstasc: the number of vectors in each star
!=======================================================================
      call kptsum(numlat,nlatma,numsta,nstave,nstasc,rsumsc,&
                  nkprun,vkpt,wtkpt)
!=======================================================================
! print the non-zero stars in the first 1..numsta stars
!=======================================================================
      if (idebug > 0) then
      write(nconso,*)'KPT:  Star  a1 a2 a3    |R_m|    S_m N_m     N_m'
      do 1000 j=1,numsta
         if (abs(rsumsc(j)).gt.acc.or.idebug.ge.1) then
            write(nconso,1100)j,(nstave(i,j),i=1,3),rstasc(j),rsumsc(j),&
                              nstasc(j)
         endif
 1000 continue 
 1100 format(1x,'KPT: ',i5,' ',3i3,f11.6,f11.6,i5)
      endif
!=======================================================================
! Calculate the Chadi-Cohen [PRB 8, 5747 (1973)] estimate of the error 
!=======================================================================
      epsilo=0.0d0
      iany=.false.
      do 2000 j=1,numsta
         if (abs(rsumsc(j)).gt.acc) then
            iany=.true.
!=======================================================================
! rsumsc(j)/dble(nstasc(j)) is the S_m in formular (20) of the paper
! [The formular expects S=+/-1 as the paper discusses the special sets]
!=======================================================================
            epsilo=epsilo+abs(rsumsc(j)/dble(nstasc(j)))/rstasc(j)**3
         endif
 2000 continue
      if (iany) then
         write(nconso,2100) epsilo
         write(nconso, *  ) ' '
 2100    format(1x,'KPT: Chadi-Cohen asymptotic error estimate:',f16.12&
            ,/, 1x,'KPT: (see PRB 8, 5747 (1973); 13, 5188 (1976))')
      else
         write(nconso,*)'KPT: Cannot calc an error estimate'
         write(nconso,*)'KPT: The realspace grid in finding the stars'
         write(nconso,*)'KPT: - or the max # of stars is too small'
         write(nconso,*)'KPT: At least the first',numsta,' stars are 0!'
      endif
!=======================================================================
!
! write kpoints to netCDF file 
      call write_kpts(nspin,nkpnum,nkprun,gkp123,vkpt,wtkpt,&
                      lmastr)

! check nkpmem: The number of k-points in memory 
      if (nkpmem>nkprun) then 
        nkpmem = nkprun
        write(nconso,*) 'KPT: nkpmem > nkprun : nkpmem = nkprun'
      endif 
      write(nconso,*) 'KPT: nkpmem : ',nkpmem

      end subroutine kptana


!=======================================================================
      subroutine kptfol(nkprun,vkpt,wtkpt,nkpnum,gkp123,nkpibz,nkpunf,&
                 lkpinv,numspa,nspama,linvad,idebug,nconso,nspin,kspin)
!=======================================================================
! fold in the k-points to the IBZ
!=======================================================================
! input:
! gkp123(1..nkpnum):       The k-points in the 1st Brillouin zone
! nspama(3,3,1..numspa):   The numspa space group matrices
! linvad(1..numspa):       True if the inversion has been added to nspama
!=======================================================================
! output:
! vkpt(1..3,1..nkprun):    The run-time k-points (in the IBZ)
! wtkpt(1..nkprun):        Weights of the run-time k-points
! nkpibz(x=1..nkpnum):     The index of which IBZ k-point corresponds
!                          to the x'th k-point in the 1st Brillouin zone
! nkpunf(3,3,x=1..nkpnum): The matrix bringing the nkpibz(x)'th k-point
!                          from the IBZ to the x'th k-point in the
!                          1st Brillouin zone
! lkpinv(1..nkpnum):       True if the inversion has been added to nkpunf
!=======================================================================
      implicit none
      real*8 , pointer :: vkpt(:,:,:)
      real*8 , pointer :: wtkpt(:,:)
      integer, pointer ::  kspin(:,:)

      real*8, pointer  :: gkp123(:,:)
      integer, pointer :: nkpibz(:)
      integer, pointer :: nkpunf(:,:,:)
      logical*4   lkpinv(:)                                

      integer nkprun,numspa,nkpnum
      integer nspama(3,3,numspa)
      logical*4   linvad(numspa)
      integer nspin
      integer idebug,nconso
!=======================================================================
! local arrays
!=======================================================================
      real*8 vres(3)
      logical*4 iirri
      real*8     acc
      parameter (acc=1.0d-5)
      integer   i,j,k,nkp,nspa,nkpr
      real*8    diff1,diff2,diff3,tmp
!=======================================================================
      nkprun = 0
      do 1800 nkp=1,nkpnum
!=======================================================================
! loop over the space group operations and rotate the k-point
! in the first loop we use nkpunf as work array
!=======================================================================

         iirri=.true.
         do 1700 nspa=1,numspa

!=======================================================================
! loop over the k-points already detected in the irriducible Brillouin zone
!=======================================================================
            do 1600 nkpr=1,nkprun

               do 1400 i=1,3
                  vres(i)=0
                  do 1000 k=1,3
                     vres(i)=vres(i)+nspama(i,k,nspa)*&
                             gkp123(k,nkpunf(1,1,nkpr)) 
 1000             continue
 1400          continue

               diff1=vres(1)-gkp123(1,nkp)
               diff2=vres(2)-gkp123(2,nkp)
               diff3=vres(3)-gkp123(3,nkp)
               diff1=abs(nint(diff1)-diff1)
               diff2=abs(nint(diff2)-diff2)
               diff3=abs(nint(diff3)-diff3)
               if (diff1.le.acc.and.diff2.le.acc.and.diff3.le.acc) then
                  iirri=.false.
!=======================================================================
! the weight and the number of the matrix
!=======================================================================
                  nkpunf(2,1,nkpr)=nkpunf(2,1,nkpr)+1
                  nkpunf(3,1,nkp)=nspa
                  nkpibz(nkp)=nkpr

                  if (idebug > 0) then 
                     write(nconso,1601) 'KPT: K-point ',nkp,gkp123(1,nkp),gkp123(2,nkp),gkp123(3,nkp), & 
                                        ' Sym operation ', nspa                  
 1601                format(1x,a14,1x,i4,1x,3f8.4,1x,a16,i2)
                  endif
                
                  go to 1750
               endif
 1600       continue
 1700    continue
 1750    continue
!=======================================================================
! store pointer to this point in the irriducible Brillouin zone
! nkprun              number of k-points in the IBZ
! nkpunf(1,1,nkprun)  index of the nkprun'th irriducible k-point
! nkpunf(2,1,nkp):    counter for the calc. of the weight
! nkpunf(3,1,nkp):    index of the matrix that brough this k-point
!                     into the 1st BZ (does not make sence as the
!                     k-point was already in the 1st BZ - therefore -1)
!=======================================================================
         if (iirri) then
            nkprun=nkprun+1
            nkpunf(1,1,nkprun)=nkp
            nkpunf(2,1,nkprun)=1
            nkpunf(3,1,nkp)=-1
            nkpibz(nkp)=nkprun
         endif
 1800 continue

! ======================================================================
! Now we have the number of irriducible Brillouin zone k-points 
! allocate arrays for coordinates,weights and spin-assign. 
	allocate(vkpt(3,nspin,nkprun)) 
	allocate(wtkpt(nspin,nkprun)) 
	allocate(kspin(nspin,nkprun)) 

!=======================================================================
! Now use the information in the temporary work-array, nkpunf(3,3,n)
! to store information on how the k-points are related
!=======================================================================

!=======================================================================
! First the coordinates
!=======================================================================
      do 2000 nkp=1,nkprun
         do 2010 i=1,nspin
            vkpt(1,i,nkp)=gkp123(1,nkpunf(1,1,nkp))
            vkpt(2,i,nkp)=gkp123(2,nkpunf(1,1,nkp))
            vkpt(3,i,nkp)=gkp123(3,nkpunf(1,1,nkp))
 2010    continue
 2000 continue
!=======================================================================
! Then the weight factors
!=======================================================================
      tmp=0.0d0
      do 2100 nkp=1,nkprun
         tmp=tmp+dble(nkpunf(2,1,nkp))
 2100 continue
      do 2200 nkp=1,nkprun
         do 2210 i=1,nspin
            wtkpt(i,nkp)=dble(nkpunf(2,1,nkp))/tmp
 2210    continue
 2200 continue
!=======================================================================
! Assign spin up/down to each wavefunction
!=======================================================================
      do 7700 nkp=1,nkprun
         do 7710 i=1,nspin
            kspin(i,nkp)=i
 7710    continue
 7700 continue
!=======================================================================
! Finally the matrices
!=======================================================================
      do 2700 nkp=1,nkpnum
         nspa=nkpunf(3,1,nkp)
!=======================================================================
! nspa==-1 if the point already in the IBZ and the matrix is the identity
!=======================================================================
         if (nspa.eq.-1) then
            do 2400 j=1,3
               do 2300 i=1,3
                  nkpunf(i,j,nkp)=0
 2300          continue
               nkpunf(j,j,nkp)=1
 2400       continue
            lkpinv(nkp)=.false.
!=======================================================================
! else the matrix is the nspa'th element of the nspama()
!      (which has been constructed by adding the inversion if linvad true)
!=======================================================================
         else
            do 2600 j=1,3 
               do 2500 i=1,3
                  nkpunf(i,j,nkp)=nspama(i,j,nspa)
 2500          continue
 2600       continue
            lkpinv(nkp)=linvad(nspa)
         endif
 2700 continue
!=======================================================================
      
      write(nconso, 3001)  nkprun
      write(nconso, 3002)
      write(nconso, 3010)
      write(nconso, 3002)
      do nkp = 1,nkprun
        write(nconso, 3000) nkp,vkpt(1,1,nkp),vkpt(2,1,nkp),vkpt(3,1,nkp),wtkpt(1,nkp)
      enddo
      write(nconso, 3002)
      write(nconso,  *  )  'KPT: '


 3001 format(1x,&
      'KPT: k-points in the irriducible Brillouin zone (nkprun) : ',i4)
 3000 format(1x,'KPT: ', i4, 2x, 3f10.6, 2x, f10.6)
 3002 format(1x,'KPT:',1x,58("-"))
 3010 format( ' KPT: k-point      k-point in units of          k-point',&
           /, ' KPT: number      B1        B2       B3         weigth ')

      end subroutine kptfol

!=======================================================================
      subroutine kptsta(dirc,numlat,nlatma,nstmax,numsta,nstave,&
                        rstasc)
!=======================================================================
! calculate the stars
!=======================================================================
      implicit double precision (a,b,d-h,o-z)
      real*8 dirc(3,3)
      integer nlatma(3,3,numlat)
      integer nvec(3)
      integer nres(3)
      integer nstave(3,nstmax)
      real*8 rstasc(nstmax)
      parameter (nmax=7)
      logical*4 inew
!=======================================================================
! Use ncount to find the stars
!=======================================================================
      numsta=0
      ncount=1
      rmax=1000.0d0
!=======================================================================
! loop over a grid in real space
!=======================================================================
      do 1400 n3=nmax,-nmax,-1
         nvec(3)=n3
         do 1300 n2=nmax,-nmax,-1
            nvec(2)=n2
            do 1200 n1=nmax,-nmax,-1
               nvec(1)=n1
!=======================================================================
! calculate the length of the present vector
!=======================================================================
               rlen=0.0d0
               do 1160 i=1,3
                  rtmp=0.0d0
                  do 1130 k=1,3
                     rtmp=rtmp+dirc(k,i)*nvec(k)
 1130             continue
                  rlen=rlen+rtmp**2
 1160          continue
               rlen=sqrt(rlen)
!=======================================================================
! loop over the crystallographic point group to
! see if this vector belongs to a star which is already represented
!=======================================================================
               inew=.true.
               do 1100 nsym=1,numlat
                  do 1050 i=1,3
                     nres(i)=0
                     do 1000 k=1,3
                        nres(i)=nres(i)+nlatma(i,k,nsym)*nvec(k)
 1000                continue
 1050             continue
!=======================================================================
! get the index of the rotated/mirrored vector
!=======================================================================
                  ncounr=-1
                  if (abs(nres(1)).le.nmax.and.&
                      abs(nres(2)).le.nmax.and.&
                      abs(nres(3)).le.nmax) then
                     ncounr=((nmax-nres(3))*(2*nmax+1)&
                             +nmax-nres(2))*(2*nmax+1)&
                             +nmax-nres(1)+1
                  endif
!=======================================================================
! See if the rotated/mirrored vector lies inside the considered grid
!=======================================================================
                  if (ncounr.gt.0) then
!=======================================================================
! If the index of the rotated/mirrored vector is smaller than the index
! of the vector itself, then the vector belongs to a Star which is
! already detected
!=======================================================================
                     if (ncounr.lt.ncount) then
                        inew=.false.
                     endif
                  else
!=======================================================================
! The rotated/mirrored vector lies outside; the length of this vector
! is larger than the length of some vectors outside the grid
!=======================================================================
                     if (rlen.lt.rmax) rmax=rlen
                  endif
 1100          continue
!=======================================================================
! skip the (0,0,0)
!=======================================================================
               if (abs(n1)+abs(n2)+abs(n3).eq.0) inew=.false.
!=======================================================================
! this is a new star [ different from (0,0,0) ]
!=======================================================================
               if (inew) then
!=======================================================================
! find where it fits in the array of stars of increasing length
!=======================================================================
                  nfit=1
                  do 1220 i=1,numsta
                     if (rstasc(i).le.rlen) nfit=i+1
 1220             continue
!=======================================================================
! increase the length of the array of stars
!=======================================================================
                  if (numsta.lt.nstmax) then
                     numsta=numsta+1
                  endif
!=======================================================================
! make room for the new star where it fits
!=======================================================================
                  do 1240 j=numsta,nfit+1,-1
                     do 1230 i=1,3
                        nstave(i,j)=nstave(i,j-1)
 1230                continue
                     rstasc(j)=rstasc(j-1)
 1240             continue
!=======================================================================
! store the new star
!=======================================================================
                  if (nfit.le.nstmax) then
                     do 1250 i=1,3
                        nstave(i,nfit)=nvec(i)
 1250                continue
                     rstasc(nfit)=rlen
                  endif
               endif
               ncount=ncount+1
 1200       continue
 1300    continue
 1400 continue
!=======================================================================
! find out which is the largest relyable star (the search takes place
! on a finite grid => a upper limit on the star-length)
!=======================================================================
      nma=-1
      do 1600 i=1,numsta
         if (rstasc(i).lt.rmax) nma=i
 1600 continue
      numsta=nma

      end subroutine kptsta

!=======================================================================
      subroutine kptsum(numlat,nlatma,numsta,nstave,nstasc,rsumsc,&
                        nkprun,vkpt,wtkpt)
!=======================================================================
! calculate the sum over k-points of the symmetriced plane waves
!=======================================================================
      implicit double precision (a,b,d-h,o-z)
      integer nlatma(3,3,numlat)
      integer nstave(3,numsta)
      integer nstasc(numsta)
      real*8 rsumsc(numsta)
      real*8 vkpt(3,nkprun)
      real*8 wtkpt(nkprun)
!=======================================================================
! local arrays
!=======================================================================
      integer nres(3)
      logical*4 idegen
      integer noth(3,48)
      parameter(tpi=6.2831853072d0)
!=======================================================================
! loop over the stars
!=======================================================================
      do 1500 nsta=1,numsta
         nstasc(nsta)=0
         rsumsc(nsta)=0.0d0
!=======================================================================
! loop over the crystallographic point group
!=======================================================================
         do 1400 nsym=1,numlat
            do 1200 i=1,3
               nres(i)=0
               do 1100 k=1,3
                  nres(i)=nres(i)+nlatma(i,k,nsym)*nstave(k,nsta)
 1100          continue
 1200       continue
!=======================================================================
! find and count the non-degenerate vectors in each star
!=======================================================================
            idegen=.false.
            do 1260 j=1,nstasc(nsta)
               if (nres(1).eq.noth(1,j).and.&
                   nres(2).eq.noth(2,j).and.&
                   nres(3).eq.noth(3,j)) idegen=.true.
 1260       continue
            if (.not.idegen) then
               nstasc(nsta)=nstasc(nsta)+1
               noth(1,nstasc(nsta))=nres(1)
               noth(2,nstasc(nsta))=nres(2)
               noth(3,nstasc(nsta))=nres(3)
!=======================================================================
! loop over k-points and calculate the sum
!=======================================================================
               do 1300 nkp=1,nkprun
                  rang=vkpt(1,nkp)*dble(nres(1))&
                      +vkpt(2,nkp)*dble(nres(2))&
                      +vkpt(3,nkp)*dble(nres(3))
                  rang=tpi*rang
                  rsumsc(nsta)=rsumsc(nsta)+wtkpt(nkp)*cos(rang)
 1300          continue
            endif
 1400    continue
 1500 continue

      end subroutine kptsum


! ======================================================================

      subroutine write_kpts(&
          nspin,nkpnum,nkprun,gkp123,vkpt,wtkpt,lmastr) 

      use netcdfinterface
      use run_context

!     The information for the k-points generated are written to the netCDF 
!     dimensions and variables describing the k-points:
!      dimensions:
!        number_bz_kpoints
!        number_ibz_kpoints
!      variables:
!        BZKpoints(number_BZ_kpoints, dim3)
!        IBZKpoints(number_IBZ_kpoints, dim3) 
!        KpointWeight(number_IBZ_kpoints) 
! 
!      input : 
!          nspin     :    number of spin (for dim the arrays as in main program)
!          nkpnum    :    number of bz k-points
!          nkprun    :    number of ibz k-points
!          gkp123    :    bz kpoints
!          vkpt      :    ibz kpoints
!          wtkpt     :    symmetry weight og ivz k-points
!          lmastr    :    tells if this process should write

      implicit none 
      integer  nspin,nkpnum,nkprun 
      real*8   gkp123(3,nkpnum),vkpt(3,nspin,nkprun)
      real*8   wtkpt(nspin,nkprun)
      logical*4 lmastr

!     locals 
      integer ncid,status
!     make copy of vkpt and wtkpt to remove spin dimension
!     before writing to netcdf file
      real*8 vkpt_netcdf(3,nkprun),wtkpt_netcdf(nkprun)


      if (.not.lmastr) return

      ! open netCDF file
      status = nf_open(netCDF_output_filename,NF_WRITE, ncid )
      if (status /= nf_noerr) stop "nf_open: error write_kpts"                      

!     write nkpnum
      status = nfputglobaldim(ncid,'number_BZ_kpoints',nkpnum) 
      call kpt_netcdf_errhandler(&
          'writing dimension number_BZ_kpoints',status,nconso)  

!     write nkprun
      status = nfputglobaldim(ncid,'number_IBZ_kpoints',nkprun)
      call kpt_netcdf_errhandler(&
          'writing dimension number_IBZ_kpoints',status,nconso)  

!     write gkp123 
      status=nfput(ncid,'BZKpoints',gkp123,dim_name1='dim3',&
                   dim_name2='number_BZ_kpoints') 
      call kpt_netcdf_errhandler('writing BZKpoints',status,nconso)  

!     add attributes
      status = nfput(ncid,'BZKpoints%Description','k-points in the BZ')
      call kpt_netcdf_errhandler(&
          'writing BZKpoints%Description',status,nconso)  
      status = nfput(ncid,'BZKpoints%units',&
          'in units of the reciprocal space basis')
      call kpt_netcdf_errhandler(&
          'writing BZKpoints%unit',status,nconso)  

!     write vktp
      vkpt_netcdf(:,:) = vkpt(:,1,:)
      status=nfput(ncid,'IBZKpoints',vkpt_netcdf,dim_name1='dim3',&
                   dim_name2='number_IBZ_kpoints') 
      call kpt_netcdf_errhandler('writing IBZKpoints',status,nconso)  

!     add attributes
      status=nfput(ncid,'IBZKpoints%Description','k-points in the BZ')
      call kpt_netcdf_errhandler(&
          'writing BZKpoints%Description',status,nconso)
      status = nfput(ncid,'IBZKpoints%units',&
          'in units of the reciprocal space basis')
      call kpt_netcdf_errhandler(&
          'writing IBZKpoints%unit',status,nconso)                 

!     write wtkpt
      wtkpt_netcdf(:) = wtkpt(1,:)
      status = nfput(ncid,'KpointWeight',wtkpt_netcdf,&
                     dim_name1='number_IBZ_kpoints') 
      call kpt_netcdf_errhandler('writing KpointWeight',status,nconso)  

!     add attributes
      status=nfput(ncid,'KpointWeight%Description',&
            'Symmetry weight of k-points(sum of weights=1)')
      call kpt_netcdf_errhandler(&
            'writing KpointWeight%Decsription',status,nconso)  

      status = nf_close(ncid)
      if (status /= nf_noerr) stop "nf_close(kptana): error"                      

      end subroutine write_kpts

! ======================================================================

      subroutine kpt_netcdf_errhandler(text,nferror,nconso) 

      use netcdfinterface

!     Reports errror writing k-points to the netCDF file
!     All errors are fatal. 

      implicit none
      character*(*) text
      integer nferror ,nconso

      if ((nferror/=nfif_OK ).and.&
          (nferror/=nfif_dimexist_butOKsize)) then

        write(nconso,*) 'KPT: nf_error',text,nferror 
        stop

      endif 
 
      end subroutine kpt_netcdf_errhandler


      end module kp_analysis
