#include "definitions.h"
    module  read_structure_netcdf_module
!=====================================================================
!   Read structurel information from netCDF file
!   I/O filename from module run_context
!=====================================================================
    use netcdfinterface
    use run_context
    use matrix_utilities

    private :: get_nuclear_charges    ! map of names -> nuclear_charges
!##################################################################### 
    contains
!#####################################################################
    subroutine read_structure(nions, nspec, atomic_number, nionsp, &
                              dirc, recc, diri, reci, &
                              posion, rvelo, rmove, &
                              sphericalcnstr,icoordsystem,lmastr)

#ifdef PARAL
    use par_functions_module
#endif

    implicit none
!=====================================================================

    integer,intent(out) :: nions            ! total number of ions in the structure
    integer,intent(out) :: nspec            ! total number species in the structure 
    integer, pointer    :: atomic_number(:) ! atom number map of a species present (allocated)
    integer, pointer    :: nionsp(:)        ! number of ions of a present type, 
                                            ! indexed by atomic_number (allocated) 
  
    real*8,intent(out)  :: dirc(3,3)        ! unit cell, vectors rowvise, i.e. basis vector j = dirc(j,:)
    real*8,intent(out)  :: recc(3,3)        ! recipical lattice
    real*8,intent(out)  :: diri(3,3)        !  obsolete - set to dirc for now
    real*8,intent(out)  :: reci(3,3)        !  obsolete - set to recc for now
        
    real*8 , pointer    :: posion(:,:,:)    ! scaled atomic coordinates (allocated)
    real*8 , pointer    :: rvelo(:,:)       ! atomic velocity  (allocated)
    real*8 , pointer    :: rmove(:,:)       ! dynamic factor   (allocated)

    integer,intent(out)  :: sphericalcnstr(2) ! atom pair for which spherical coordsys apply
    integer,intent(out)  :: icoordsystem      ! 0:switch off 1:spherical coordsys for pair
    logical,intent(in)   :: lmastr
!--------------------------------------------------------------------- 
!   locals 

    integer, parameter    :: longstring = 100   ! move to a central file 
    integer, parameter    :: namelength = 6     ! do ; (for atom names)

    integer          :: ncid, status, number_ionic_steps, ion, ipos, ispec
    integer, pointer :: datadim(:)        ! for holding variable dimensions
    integer          :: aux3a(3), aux3b(3)
    integer          :: ng,ni,n,j,nsp,ntyp,i,ipair,ion1,ion2

    integer,allocatable  :: nuclear_charges(:), pair(:)

    character(len=longstring),allocatable ::  DynamicAtomAttributes(:) ! constraints, etc.
    character(len=2),         allocatable ::  DynamicAtomSpecies(:)    ! chem symbols
    character(len=namelength),allocatable ::  DynamicAtomNames(:)     
    real*8,                   allocatable ::  DynamicAtomPositions(:,:)
    real*8,                   allocatable ::  workmatrix(:,:)
    real*8                                ::  rlatc, volc, voli, tmp(6)
    character*3                           ::  yesno(6)
    logical                               ::  foundAtomSpecies,foundAtomNames
    integer                               ::  nOK

#ifdef PARAL 
#include "ms.h"
#endif
   
!=======================================================================       

  if (lmastr) then
  
  status =  nf_open(netCDF_input_filename, NF_NOWRITE, ncid)       
  if (status /= nf_noerr) call abort_calc(nconso, &
                                "read_structure -> nf_open : error opening nc-file")

! ====================================================================
!  retrieve some global dimensions 
! ====================================================================

   status = nfgetglobaldim(ncid, "number_of_dynamic_atoms", nions)            ! must exist
   if (status /= nfif_OK ) call abort_calc(nconso, &
              "read_structure -> nfgetglobaldim : error retrieving dimension (number_of_dynamic_atoms)")
    
   status = nfgetglobaldim(ncid, "number_ionic_steps", number_ionic_steps)    ! must exist
   if (status /= nfif_OK ) call abort_calc(nconso, &
              "read_structure -> nfgetglobaldim : error retrieving dimension (number_ionic_steps)")
   

! ====================================================================
!   Get the unit cell
!   NB: vectors are expected rowvise in the netCDF set
! ====================================================================
   
    status = nfget(ncid, "UnitCell_ScaleFactor", rlatc)     
    if (status /= nfif_OK ) then
       rlatc = 1.0d0                           ! set defaults
    else
      if (rlatc == nf_fill_double) rlatc = 1.0 ! set default
    endif
    
    aux3a = (/1, 1, number_ionic_steps/)
    aux3b = (/3, 3, 1                 /)
    status = nfget(ncid, "UnitCell", dirc, startnf = aux3a, countnf =aux3b)
    if (status /= nfif_OK )   call abort_calc(nconso, &
                                  "read_structure -> nfget(UnitCell) : load error")
    if (dirc(1,1) == nf_fill_double)    call abort_calc(nconso, &
                                            "read_structure -> found empty UnitCell entry")

    dirc      = transpose(dirc)                   ! convert to internal representation
    dirc(:,:) = dirc(:,:) * rlatc                 ! rescale basis
    diri(:,:) = dirc(:,:)                         ! maintain temporary  
    
    call bastr(dirc,recc,volc)                    ! generate recipical basis
    call bastr(diri,reci,voli)                    ! maintain temporary

! ====================================================================
!   Get the ion types + coordinates + attributes
!         for now :  don't consider the possibility of static atom types
! ====================================================================

    allocate ( DynamicAtomSpecies(nions)      )
    allocate ( DynamicAtomNames(nions)        ) 
    allocate ( DynamicAtomAttributes(nions)   ) 

    allocate ( DynamicAtomPositions(3, nions) )
    allocate ( nuclear_charges(nions)         ) 
    allocate ( pair(nions)                    ) 

    allocate ( workmatrix(nions, 12) )      ! contains Z,position,rvelo,rmove,pair

    allocate ( rvelo(3, nions) )            ! outbound variable
    allocate ( rmove(3, nions) )            ! outbound variable


    if (associated(Internal_to_netCDF_order))  call abort_calc(nconso, &
          "read_structure: Internal_to_netCDF_order(:) initialized elsewhere") 
    allocate ( Internal_to_netCDF_order(nions) )  ! defined in netcdfinterface module
     


! --- Form nuclear charge map from either DynamicAtomNames/Species ---


    foundAtomSpecies = .true. 
    DynamicAtomSpecies = ""      ! clear buffers
    status = nfget(ncid, "DynamicAtomSpecies", DynamicAtomSpecies)   
    if (status /= nfif_OK ) then 
        foundAtomSpecies = .false.
    else 
       if (DynamicAtomSpecies(1)(1:1) == achar(nf_fill_char))  & 
          foundAtomSpecies = .false.
    endif 

    foundAtomNames = .true. 
    DynamicAtomNames = ""        ! clear buffers 
    status = nfget(ncid, "DynamicAtomNames", DynamicAtomNames)
    if (status /= nfif_OK ) then 
        foundAtomNames = .false.
    else
        if (DynamicAtomNames(1)(1:1) == achar(nf_fill_char)) & 
           foundAtomNames = .false. 
    endif


    if (foundAtomNames) then     
        call get_nuclear_charges(nconso, DynamicAtomNames,   nuclear_charges)
    elseif (foundAtomSpecies) then 
        call get_nuclear_charges(nconso, DynamicAtomSpecies, nuclear_charges)
    else
        call abort_calc(nconso, &
          "read_structure -> found neither DynamicAtomSpecies nor DynamicAtomNames")      
    endif
 
! --- Check, whether attributes has been provided ---
! --- for now, only 0/1 constraints are supported ---

    rmove(:,:) = 1.0                                 ! default : all degrees of freedom are open
    ipair      = 0                                   ! default : no spherical pairs are defined
    pair(:)   = 0
    icoordsystem = 0

    DynamicAtomAttributes = "" ! clear buffers
    status = nfget(ncid, "DynamicAtomAttributes", DynamicAtomAttributes)
    if ((status == nfif_OK ) .and. &
        (DynamicAtomAttributes(1)(1:1) /= achar(nf_fill_char))) then  ! Attributes are set
!      --------------------
       do ion = 1, nions
          ipos = index(DynamicAtomAttributes(ion), "fix:")         ! inspect for fixed coord.
          if (ipos /= 0) then
             if (scan(DynamicAtomAttributes(ion)(ipos+4:ipos+6), "1") /= 0) rmove(1,ion) = 0.0
             if (scan(DynamicAtomAttributes(ion)(ipos+4:ipos+6), "2") /= 0) rmove(2,ion) = 0.0
             if (scan(DynamicAtomAttributes(ion)(ipos+4:ipos+6), "3") /= 0) rmove(3,ion) = 0.0            
          endif  ! ipos > 0

          ipos = index(DynamicAtomAttributes(ion), "pair")         ! inspect for pair keyword
          if (ipos /= 0) then
             ipair      = ipair + 1
             pair(ion)  = ipair              ! set pairmember (1,2)
             sphericalcnstr(ipair) = ion     ! temporary storage
          endif

       enddo     ! ion = ...
!      --------------------
       if     (ipair == 0) then           ! no pairs found
          icoordsystem = 0 
       elseif (ipair == 2) then           ! scan for spherical constraints (additive)
          icoordsystem = 1
          ion1 = sphericalcnstr(1)        !                                -> MMP coor
          ion2 = sphericalcnstr(2)        !                                -> r,theta,phi coor

          ipos = index(DynamicAtomAttributes(ion1), "fix_MMP:") 
          if (ipos /= 0) then
             if (scan(DynamicAtomAttributes(ion1)(ipos+8:ipos+10), "1") /= 0) rmove(1,ion1) = 0.0
             if (scan(DynamicAtomAttributes(ion1)(ipos+8:ipos+10), "2") /= 0) rmove(2,ion1) = 0.0
             if (scan(DynamicAtomAttributes(ion1)(ipos+8:ipos+10), "3") /= 0) rmove(3,ion1) = 0.0            
          endif  ! ipos > 0

          ipos = index(DynamicAtomAttributes(ion2), "fix_MMP:") 
          if (ipos /= 0) then
             if (scan(DynamicAtomAttributes(ion2)(ipos+8:ipos+10), "1") /= 0) rmove(1,ion1) = 0.0
             if (scan(DynamicAtomAttributes(ion2)(ipos+8:ipos+10), "2") /= 0) rmove(2,ion1) = 0.0
             if (scan(DynamicAtomAttributes(ion2)(ipos+8:ipos+10), "3") /= 0) rmove(3,ion1) = 0.0            
          endif  ! ipos > 0
          
          ipos = index(DynamicAtomAttributes(ion1), "fix_spherical:") 
          if (ipos /= 0) then
             if (scan(DynamicAtomAttributes(ion1)(ipos+14:ipos+16), "r") /= 0) rmove(1,ion2) = 0.0
             if (scan(DynamicAtomAttributes(ion1)(ipos+14:ipos+16), "t") /= 0) rmove(2,ion2) = 0.0
             if (scan(DynamicAtomAttributes(ion1)(ipos+14:ipos+16), "p") /= 0) rmove(3,ion2) = 0.0            
          endif  ! ipos > 0

          ipos = index(DynamicAtomAttributes(ion2), "fix_spherical:") 
          if (ipos /= 0) then
             if (scan(DynamicAtomAttributes(ion2)(ipos+14:ipos+16), "r") /= 0) rmove(1,ion2) = 0.0
             if (scan(DynamicAtomAttributes(ion2)(ipos+14:ipos+16), "t") /= 0) rmove(2,ion2) = 0.0
             if (scan(DynamicAtomAttributes(ion2)(ipos+14:ipos+16), "p") /= 0) rmove(3,ion2) = 0.0            
          endif  ! ipos > 0

       else                               ! exception
          call abort_calc(nconso, "read_structure: DynamicAtomPositions_Attribute : error setting pair")
       endif
!      --------------------
    endif        ! if  (Attributes are set) ...

   
    rvelo(:,:) = 0.0                                 ! default : zero velocities

    aux3a = (/1, 1,     number_ionic_steps/)
    aux3b = (/3, nions, 1                 /)
    status = nfget(ncid, "DynamicAtomVelocities", rvelo, startnf = aux3a, countnf =aux3b)

    if (status == nfif_OK) then
       if ((rvelo(1,1) == nf_fill_double)) rvelo(:,:) = 0.0   ! delete fill
    endif
    
    DynamicAtomPositions = 0.0    ! render variable defined in all cases, before if statesment below

    status = nfget(ncid, "DynamicAtomPositions", DynamicAtomPositions, startnf = aux3a, countnf =aux3b)
    if ((status /= nfif_OK ) .or. &
        (DynamicAtomPositions(1,1) == nf_fill_double)) &
           call abort_calc(nconso, "read_structure -> nfget: found no DynamicAtomPositions")

    where (abs(rmove) < 1.0d-6)                      ! make  rmove and rvelo consistent
       rvelo = 0.0
    end where


! --- end of reading: close netCDF set ---


    status = nf_close(ncid)
    if (status /= nf_noerr) call abort_calc(nconso, &
                         "read_structure -> nf_close : error closing nc-outfile")   

! ====================================================================      
! Setup compound matrix and sort it according to nuclear charges 
! ====================================================================

     workmatrix(:, 1)    = nuclear_charges(:)    ! implicit double type cast


! -- workmatrix(:, 1) below: fix to ensure internal ion ordering for same species
!    is as provided by user; OK for nions < 500000 ...

     do ion = 1, nions                                        
       workmatrix(ion, 1) = workmatrix(ion, 1) + ion * 1.d-8  
       Internal_to_netCDF_order(ion) = ion                        ! initialize
     enddo

     workmatrix(:, 2:4)  = reshape(source = DynamicAtomPositions, order = (/2,1/), shape = (/nions,3/) )
     workmatrix(:, 5:7)  = reshape(source = rvelo,                order = (/2,1/), shape = (/nions,3/) )
     workmatrix(:, 8:10) = reshape(source = rmove,                order = (/2,1/), shape = (/nions,3/) )
     workmatrix(:, 11)   = pair(:)                      ! implicit double type cast
     workmatrix(:, 12)   = Internal_to_netCDF_order(:)  ! implicit double type cast 

     call hpsort_matrix(workmatrix)              !  def+interface in matrix_utilities
  
     nuclear_charges(:)   = nint( workmatrix(:, 1) )            ! explicit cast
     DynamicAtomPositions = reshape(source = workmatrix(:, 2:4) , order = (/2,1/), shape = (/3, nions/) ) 
     rvelo                = reshape(source = workmatrix(:, 5:7) , order = (/2,1/), shape = (/3, nions/) )
     rmove                = reshape(source = workmatrix(:, 8:10), order = (/2,1/), shape = (/3, nions/) )
     pair(:)              = nint( workmatrix(:, 11) )         ! explicit cast
     Internal_to_netCDF_order(:) = nint( workmatrix(:, 12) )  ! explicit cast


! ====================================================================
! Generate derived quantities + arrays
! ====================================================================     

     sphericalcnstr(:) = 0                    ! extract sphericalcnstr(:) from sorted pair(:)
     do ion = 1, nions                                        
       if (pair(ion) > 0) sphericalcnstr(pair(ion)) = ion
     enddo

     nspec = 1                                ! count species - now nuclear_charges
     do ion = 2, nions                        ! are an ordered array
       if (nuclear_charges(ion-1) /= nuclear_charges(ion)) &
       nspec = nspec + 1
     enddo

     allocate( posion(3,nions,nspec))
     allocate( atomic_number(nspec))
     allocate( nionsp(nspec))

     atomic_number(:) = 0                       ! initialise  target arrays
     nionsp(:) = 0
     posion(:,:,:) = 0.0
     ispec = 1                                 
     atomic_number(1) = nuclear_charges(1)
     posion(:, 1, 1) = DynamicAtomPositions(:, 1) 
     nionsp(1) = 1

     do ion = 2, nions                        
       if (nuclear_charges(ion) /= nuclear_charges(ion-1)) then
          ispec = ispec + 1                                 ! a new species
          atomic_number(ispec) = nuclear_charges(ion)
       endif
       nionsp(ispec) =  nionsp(ispec) + 1
       posion(:, nionsp(ispec), ispec) = DynamicAtomPositions(:, ion) 
     enddo

! ====================================================================
!   Write structure report 
! ====================================================================

      write(nconso,'(1x,a,f12.8)') "Structure: lattice scaling factor = ", rlatc
      write(nconso,'(1x,a)')       "Structure:-----------------------------------------------------------"
      write(nconso,'(1x,a)')       "Structure: unit cell A            (lattice vectors coloumnwise) :"
      write(nconso,'(1x,a)')       "Structure:             A1           A2            A3"
      write(nconso, 100)           "Structure:  x", dirc(:,1), "Angstroem"
      write(nconso, 100)           "Structure:  y", dirc(:,2), "Angstroem"
      write(nconso, 100)           "Structure:  z", dirc(:,3), "Angstroem"
      write(nconso,'(1x,a)')       "Structure:-----------------------------------------------------------"
      write(nconso,'(1x,a)')       "Structure: recipical unit cell B  (vectors coloumnwise, A B^t = 2 Pi)"
      write(nconso,'(1x,a)')       "Structure:             B1           B2            B3"
      write(nconso, 100)           "Structure: kx", recc(:,1), "1/Angstroem"
      write(nconso, 100)           "Structure: ky", recc(:,2), "1/Angstroem"
      write(nconso, 100)           "Structure: kz", recc(:,3), "1/Angstroem"
      write(nconso,'(1x,a)')       "Structure:-----------------------------------------------------------"    
      call uflush(nconso)
 100  format(1x,a,1x,3f14.8,1x,a)

      write(nconso,'(1x,a,i5)') "Structure: Number of atoms (nions)           : ", nions
      write(nconso,'(1x,a,i5)') "Structure: Number of different atoms (nspec) : ", nspec
      write(nconso,'(1x,a)')    "Structure: Internal atom mapping: &
                                &-> ALL INFO IN THIS FILE REFERS TO THIS ATOM ORDER <-"
      write(nconso,'(1x,a)')    "Structure:   atom#  <->  (species, number)"
      ng=1                                     
      do nsp=1,nspec
        do ni=1,nionsp(nsp)
          write(nconso,'(1x,a,i5,8x,2i7)')  "Structure: ", ng, nsp, ni
          ng=ng+1
        enddo
      enddo            
      call uflush(nconso)

      

! ---  write out ionic positions (real/scaled) etc ---

      write(nconso, *)
      write(nconso, 220)  
      write(nconso,'(1x,a)')  "Structure:  >>         Ionic positions in scaled coordinates (u,v,w)             <<" 
      write(nconso,'(1x,a)')  "Structure:  atom#   Z        u            v            w         motion constraints"   
      write(nconso, 220)
      ng=1                                    
      do nsp=1,nspec
        do ni=1,nionsp(nsp)
          write(nconso,200) ng, atomic_number(nsp),posion(1,ni,nsp),posion(2,ni,nsp),posion(3,ni,nsp), & 
                                                   rmove(1,ng),rmove(2,ng),rmove(3,ng) 
          ng=ng+1
        enddo
      enddo
      write(nconso, 220) 
      call uflush(nconso)
     

 200  format(1x, "Structure:",i5,2x,i4,1x,3f13.8,1x,3f7.3)
 220  format(1x, "Structure:", 73("-"))

      write(nconso, 220)
      write(nconso,'(1x,a)')  "Structure:  >>         Ionic positions/velocities in cartesian coordinates       <<"
      write(nconso,'(1x,a)')  "Structure:  atom#   Z        x            y            z          initial velocity "  
      write(nconso,'(1x,a)')  "Structure:                           [Angstroem]                  [Angstroem/fs]   " 
      write(nconso, 220)

      ng=1                                    
      do nsp=1,nspec
        do ni=1,nionsp(nsp)
          write(nconso,200) ng, atomic_number(nsp),                      &
                            (sum( dirc(:,j)*posion(:,ni,nsp) ), j=1,3),  &
                            rvelo(1,ng),rvelo(2,ng),rvelo(3,ng)
          ng=ng+1
        enddo
      enddo
      write(nconso, 220) 
      call uflush(nconso)

! --- check atomic positions and write a small report ---
      call check_atomic_positions(nions, nspec, nionsp, dirc,posion)
      
! --- report spherical constraints ---

    write(nconso, *)
    if (icoordsystem == 0) then
       write(nconso,'(1x,a)') "Structure: no spherical constrained pair was found"
    else
       write(nconso,'(1x,a)') "Structure: found 1 spherical constrained pair:"

       do i=1,2
          ng=1                                     
          do nsp=1,nspec
             do ni=1,nionsp(nsp)
                if     (ng == sphericalcnstr(i)) then
                   write(nconso,'(1x,a,i1,a,i4,3x,3f6.3)') &
                         "Structure: atom#",ng," is paired - (Z,u,v,w) = ", &
                          atomic_number(nsp),posion(:,ni,nsp)
                   tmp( 3*i-2 : 3*i ) = rmove(:,ng)
                endif 
                ng=ng+1
             enddo
          enddo
       enddo
       
       where (abs(tmp) < 1.0d-6)      
         yesno = "yes"
       elsewhere
         yesno = " no"
       end where

       write(nconso,'(1x,a,3a4,a)') "Structure: pair constraints fix MMP(1,2,3)    : (", &
                                     yesno(1),yesno(2),yesno(3),")"
       write(nconso,'(1x,a,3a4,a)') "Structure: pair constraints fix (r,theta,phi) : (", &
                                     yesno(4),yesno(5),yesno(6),")"
    endif


! --------------------------------------------------------------
! Clean up 
! --------------------------------------------------------------

    deallocate ( DynamicAtomSpecies    )
    deallocate ( DynamicAtomNames      ) 
    deallocate ( DynamicAtomAttributes ) 
    deallocate ( DynamicAtomPositions  )
    deallocate ( nuclear_charges       ) 
    deallocate ( pair                  ) 
    deallocate ( workmatrix            ) 


    write(nconso, *) ' '
    write(nconso, *) ' '
    call uflush(nconso)


    endif ! if lmastr 

#ifdef PARAL 
    ! send dimensions xxxxxxx XXXX
    call mspack_integer_scalar(nconso, ANY, MSG_SETUP,&
                   INTEGER4, nions,  1, nOK)
    call mspack_integer_scalar(nconso, ANY, MSG_SETUP,&
                   INTEGER4, nspec,  1, nOK)

    ! allocate arrays on slave nodes 
    if (.not.lmastr) then 
       allocate( posion(3,nions,nspec))
       allocate( atomic_number(nspec))
       allocate( nionsp(nspec))

       allocate ( rvelo(3, nions) )            
       allocate ( rmove(3, nions) ) 
       rvelo = 0          
       rmove = 0
       sphericalcnstr(:) = 0
       icoordsystem=0

       allocate (Internal_to_netCDF_order(nions) )  ! defined in netcdfinterface module

    endif

    call mspack_double_array (nconso, ANY, MSG_SETUP,&
                     REAL8, posion,nspec*nions*3,nOK)
    call mspack_integer_array (nconso, ANY,MSG_SETUP,&
                     INTEGER4, nionsp,  nspec, nOK)
    call mspack_integer_array (nconso, ANY,MSG_SETUP,&
                     INTEGER4, atomic_number, nspec, nOK)

    call mspack_double_array (nconso, ANY, MSG_SETUP,&
                     REAL8, dirc,3*3,nOK)
    call mspack_double_array (nconso, ANY, MSG_SETUP,&
                     REAL8, recc,3*3,nOK)
    call mspack_double_array (nconso, ANY, MSG_SETUP,&
                     REAL8, diri,3*3,nOK)
    call mspack_double_array (nconso, ANY, MSG_SETUP,&
                     REAL8, reci,3*3,nOK)
    call mspack_integer_array (nconso, ANY,MSG_SETUP,&
            INTEGER4,Internal_to_netCDF_order,nions, nOK)

#endif



    end subroutine read_structure


!######################################################################      
      subroutine get_nuclear_charges(nconso, Names, nuclear_charges)
! -------------------------------------------------------
!     Map  names -> nuclear_charges ( = Z, not valence)
!     Convention : 1) try first to match a double letter specie to atom name
!                  2) then try to match a single letter specie to atom name
!                  3) otherwise assignment unsuccessful
!     For now    : first letter capital, second lower case
!
!     Thanks to Hans Skrives for a nice periodic table
!     awk '{if (length($2)==2) printf("%s %2d\n", $2,$1)  }' /home/zeise/skriver/bulk/period/atomic.ptb | sort -k1 
!     awk '{if (length($2)==1) printf("%s %2d\n", $2,$1)  }' /home/zeise/skriver/bulk/period/atomic.ptb | sort -k1
! -------------------------------------------------------
      implicit none
      integer                   ::  nconso              ! i/o unit
      character*(*)             ::  Names(:)            ! array of atom names
      integer                   ::  nuclear_charges(:)  ! assigned total nuclear charge

      integer       :: nions, ion
      character*2   :: namehead
      character*500 :: message
! ------------------------------------------------------
      nions = size(nuclear_charges)

      if (nions<1) &
         call abort_calc(nconso,"read_structure -> get_nuclear_charges : nions<1")

      do ion = 1, nions
        namehead = "  "
        namehead = adjustl(Names(ion))

! --------------------------------------------------------------------------------
!  Double letter species
! --------------------------------------------------------------------------------

        if (namehead(1:2) == "Ac") then ; nuclear_charges(ion) = 89 ; cycle ; endif 
        if (namehead(1:2) == "Ag") then ; nuclear_charges(ion) = 47 ; cycle ; endif 
        if (namehead(1:2) == "Al") then ; nuclear_charges(ion) = 13 ; cycle ; endif 
        if (namehead(1:2) == "Am") then ; nuclear_charges(ion) = 95 ; cycle ; endif 
        if (namehead(1:2) == "Ar") then ; nuclear_charges(ion) = 18 ; cycle ; endif 
        if (namehead(1:2) == "As") then ; nuclear_charges(ion) = 33 ; cycle ; endif 
        if (namehead(1:2) == "At") then ; nuclear_charges(ion) = 85 ; cycle ; endif 
        if (namehead(1:2) == "Au") then ; nuclear_charges(ion) = 79 ; cycle ; endif 
        if (namehead(1:2) == "Ba") then ; nuclear_charges(ion) = 56 ; cycle ; endif 
        if (namehead(1:2) == "Be") then ; nuclear_charges(ion) =  4 ; cycle ; endif 
        if (namehead(1:2) == "Bi") then ; nuclear_charges(ion) = 83 ; cycle ; endif 
        if (namehead(1:2) == "Bk") then ; nuclear_charges(ion) = 97 ; cycle ; endif 
        if (namehead(1:2) == "Br") then ; nuclear_charges(ion) = 35 ; cycle ; endif 
        if (namehead(1:2) == "Ca") then ; nuclear_charges(ion) = 20 ; cycle ; endif 
        if (namehead(1:2) == "Cd") then ; nuclear_charges(ion) = 48 ; cycle ; endif 
        if (namehead(1:2) == "Ce") then ; nuclear_charges(ion) = 58 ; cycle ; endif 
        if (namehead(1:2) == "Cf") then ; nuclear_charges(ion) = 98 ; cycle ; endif 
        if (namehead(1:2) == "Cl") then ; nuclear_charges(ion) = 17 ; cycle ; endif 
        if (namehead(1:2) == "Cm") then ; nuclear_charges(ion) = 96 ; cycle ; endif 
        if (namehead(1:2) == "Co") then ; nuclear_charges(ion) = 27 ; cycle ; endif 
        if (namehead(1:2) == "Cr") then ; nuclear_charges(ion) = 24 ; cycle ; endif 
        if (namehead(1:2) == "Cs") then ; nuclear_charges(ion) = 55 ; cycle ; endif 
        if (namehead(1:2) == "Cu") then ; nuclear_charges(ion) = 29 ; cycle ; endif 
        if (namehead(1:2) == "Dy") then ; nuclear_charges(ion) = 66 ; cycle ; endif 
        if (namehead(1:2) == "Er") then ; nuclear_charges(ion) = 68 ; cycle ; endif 
        if (namehead(1:2) == "Eu") then ; nuclear_charges(ion) = 63 ; cycle ; endif 
        if (namehead(1:2) == "Fe") then ; nuclear_charges(ion) = 26 ; cycle ; endif 
        if (namehead(1:2) == "Fr") then ; nuclear_charges(ion) = 87 ; cycle ; endif 
        if (namehead(1:2) == "Ga") then ; nuclear_charges(ion) = 31 ; cycle ; endif 
        if (namehead(1:2) == "Gd") then ; nuclear_charges(ion) = 64 ; cycle ; endif 
        if (namehead(1:2) == "Ge") then ; nuclear_charges(ion) = 32 ; cycle ; endif 
        if (namehead(1:2) == "He") then ; nuclear_charges(ion) =  2 ; cycle ; endif 
        if (namehead(1:2) == "Hf") then ; nuclear_charges(ion) = 72 ; cycle ; endif 
        if (namehead(1:2) == "Hg") then ; nuclear_charges(ion) = 80 ; cycle ; endif 
        if (namehead(1:2) == "Ho") then ; nuclear_charges(ion) = 67 ; cycle ; endif 
        if (namehead(1:2) == "In") then ; nuclear_charges(ion) = 49 ; cycle ; endif 
        if (namehead(1:2) == "Ir") then ; nuclear_charges(ion) = 77 ; cycle ; endif 
        if (namehead(1:2) == "Kr") then ; nuclear_charges(ion) = 36 ; cycle ; endif 
        if (namehead(1:2) == "La") then ; nuclear_charges(ion) = 57 ; cycle ; endif 
        if (namehead(1:2) == "Li") then ; nuclear_charges(ion) =  3 ; cycle ; endif 
        if (namehead(1:2) == "Lu") then ; nuclear_charges(ion) = 71 ; cycle ; endif 
        if (namehead(1:2) == "Mg") then ; nuclear_charges(ion) = 12 ; cycle ; endif 
        if (namehead(1:2) == "Mn") then ; nuclear_charges(ion) = 25 ; cycle ; endif 
        if (namehead(1:2) == "Mo") then ; nuclear_charges(ion) = 42 ; cycle ; endif 
        if (namehead(1:2) == "Na") then ; nuclear_charges(ion) = 11 ; cycle ; endif 
        if (namehead(1:2) == "Nb") then ; nuclear_charges(ion) = 41 ; cycle ; endif 
        if (namehead(1:2) == "Nd") then ; nuclear_charges(ion) = 60 ; cycle ; endif 
        if (namehead(1:2) == "Ne") then ; nuclear_charges(ion) = 10 ; cycle ; endif 
        if (namehead(1:2) == "Ni") then ; nuclear_charges(ion) = 28 ; cycle ; endif 
        if (namehead(1:2) == "Np") then ; nuclear_charges(ion) = 93 ; cycle ; endif 
        if (namehead(1:2) == "Os") then ; nuclear_charges(ion) = 76 ; cycle ; endif 
        if (namehead(1:2) == "Pa") then ; nuclear_charges(ion) = 91 ; cycle ; endif 
        if (namehead(1:2) == "Pb") then ; nuclear_charges(ion) = 82 ; cycle ; endif 
        if (namehead(1:2) == "Pd") then ; nuclear_charges(ion) = 46 ; cycle ; endif 
        if (namehead(1:2) == "Pm") then ; nuclear_charges(ion) = 61 ; cycle ; endif 
        if (namehead(1:2) == "Po") then ; nuclear_charges(ion) = 84 ; cycle ; endif 
        if (namehead(1:2) == "Pr") then ; nuclear_charges(ion) = 59 ; cycle ; endif 
        if (namehead(1:2) == "Pt") then ; nuclear_charges(ion) = 78 ; cycle ; endif 
        if (namehead(1:2) == "Pu") then ; nuclear_charges(ion) = 94 ; cycle ; endif 
        if (namehead(1:2) == "Ra") then ; nuclear_charges(ion) = 88 ; cycle ; endif 
        if (namehead(1:2) == "Rb") then ; nuclear_charges(ion) = 37 ; cycle ; endif 
        if (namehead(1:2) == "Re") then ; nuclear_charges(ion) = 75 ; cycle ; endif 
        if (namehead(1:2) == "Rh") then ; nuclear_charges(ion) = 45 ; cycle ; endif 
        if (namehead(1:2) == "Rn") then ; nuclear_charges(ion) = 86 ; cycle ; endif 
        if (namehead(1:2) == "Ru") then ; nuclear_charges(ion) = 44 ; cycle ; endif 
        if (namehead(1:2) == "Sb") then ; nuclear_charges(ion) = 51 ; cycle ; endif 
        if (namehead(1:2) == "Sc") then ; nuclear_charges(ion) = 21 ; cycle ; endif 
        if (namehead(1:2) == "Se") then ; nuclear_charges(ion) = 34 ; cycle ; endif 
        if (namehead(1:2) == "Si") then ; nuclear_charges(ion) = 14 ; cycle ; endif 
        if (namehead(1:2) == "Sm") then ; nuclear_charges(ion) = 62 ; cycle ; endif 
        if (namehead(1:2) == "Sn") then ; nuclear_charges(ion) = 50 ; cycle ; endif 
        if (namehead(1:2) == "Sr") then ; nuclear_charges(ion) = 38 ; cycle ; endif 
        if (namehead(1:2) == "Ta") then ; nuclear_charges(ion) = 73 ; cycle ; endif 
        if (namehead(1:2) == "Tb") then ; nuclear_charges(ion) = 65 ; cycle ; endif 
        if (namehead(1:2) == "Tc") then ; nuclear_charges(ion) = 43 ; cycle ; endif 
        if (namehead(1:2) == "Te") then ; nuclear_charges(ion) = 52 ; cycle ; endif 
        if (namehead(1:2) == "Th") then ; nuclear_charges(ion) = 90 ; cycle ; endif 
        if (namehead(1:2) == "Ti") then ; nuclear_charges(ion) = 22 ; cycle ; endif 
        if (namehead(1:2) == "Tl") then ; nuclear_charges(ion) = 81 ; cycle ; endif 
        if (namehead(1:2) == "Tm") then ; nuclear_charges(ion) = 69 ; cycle ; endif 
        if (namehead(1:2) == "Xe") then ; nuclear_charges(ion) = 54 ; cycle ; endif 
        if (namehead(1:2) == "Yb") then ; nuclear_charges(ion) = 70 ; cycle ; endif 
        if (namehead(1:2) == "Zn") then ; nuclear_charges(ion) = 30 ; cycle ; endif 
        if (namehead(1:2) == "Zr") then ; nuclear_charges(ion) = 40 ; cycle ; endif 

! --------------------------------------------------------------------------------
!  Single letter species
! --------------------------------------------------------------------------------

        if (namehead(1:1) == "B") then ; nuclear_charges(ion) =  5 ; cycle ; endif  
        if (namehead(1:1) == "C") then ; nuclear_charges(ion) =  6 ; cycle ; endif 
        if (namehead(1:1) == "F") then ; nuclear_charges(ion) =  9 ; cycle ; endif 
        if (namehead(1:1) == "H") then ; nuclear_charges(ion) =  1 ; cycle ; endif 
        if (namehead(1:1) == "I") then ; nuclear_charges(ion) = 53 ; cycle ; endif 
        if (namehead(1:1) == "K") then ; nuclear_charges(ion) = 19 ; cycle ; endif 
        if (namehead(1:1) == "N") then ; nuclear_charges(ion) =  7 ; cycle ; endif 
        if (namehead(1:1) == "O") then ; nuclear_charges(ion) =  8 ; cycle ; endif 
        if (namehead(1:1) == "P") then ; nuclear_charges(ion) = 15 ; cycle ; endif 
        if (namehead(1:1) == "S") then ; nuclear_charges(ion) = 16 ; cycle ; endif 
        if (namehead(1:1) == "U") then ; nuclear_charges(ion) = 92 ; cycle ; endif 
        if (namehead(1:1) == "V") then ; nuclear_charges(ion) = 23 ; cycle ; endif 
        if (namehead(1:1) == "W") then ; nuclear_charges(ion) = 74 ; cycle ; endif 
        if (namehead(1:1) == "Y") then ; nuclear_charges(ion) = 39 ; cycle ; endif 

! --- assignment failed : report+abort

        write(message,*) "read_structure -> get_nuclear_charges : unable to map ion=",&
                          ion, "  Name=", trim(Names(ion))
        call abort_calc(nconso, message)

     enddo

     end subroutine get_nuclear_charges


     subroutine check_atomic_positions(nions, nspec, nionsp, dirc,posion) 
!=====================================================================
!    check_atomic_positions performs a simple check of the 
!    mimimum distance between any two atoms. 
!=====================================================================    
     use run_context
     implicit none
!=====================================================================
 
     integer,intent(in)  :: nions            ! total number of ions in the structure
     integer,intent(in)  :: nspec            ! total number species in the structure
     integer, pointer    :: nionsp(:)        ! number of ions of a present type,
 
     real*8,intent(in)   :: dirc(3,3)        ! unit cell, vectors rowvise,
                                             ! i.e. basis vector j = dirc(j,:)
     real*8 , pointer    :: posion(:,:,:)    ! scaled atomic coordinates (allocated)
!---------------------------------------------------------------------
!    locals
     integer n1,n2,nsp1,nsp2,ni1,ni2,box_x,box_y,box_z
     integer j
     integer atom_pair(2),min_box(3)
     real*8  box(3),minimum_distance,r1(3),r2(3),distance
!---------------------------------------------------------------------

     minimum_distance = 1000.0d0

     n1 = 1
     do nsp1 = 1,nspec 
       do ni1 = 1,nionsp(nsp1) 

         do box_x = -2,2
          do box_y = -2,2
           do box_z = -2,2

               n2 = 1
               do nsp2 = 1,nspec 
                 do ni2 = 1,nionsp(nsp2) 
            
                    box(1) = dble(box_x) 
                    box(2) = dble(box_y) 
                    box(3) = dble(box_z) 
                    do j = 1,3 
                      r1(j) =  sum( dirc(:,j)*(posion(:,ni1,nsp1)))
                      r2(j) =  sum( dirc(:,j)*(posion(:,ni2,nsp2)+box(:)))
                    enddo
                    box(1) = dble(box_x) 
                    box(2) = dble(box_y) 
                    box(3) = dble(box_z) 

                    if (.not.((n1.eq.n2).and.(box_x.eq.0).and.(box_y.eq.0).and.(box_z.eq.0))) then 
                      distance = sqrt(sum((r1(:)-r2(:))**2))

                      if (distance<minimum_distance) then 
                         minimum_distance = distance 
                         atom_pair(1)     = n1
                         atom_pair(2)     = n2
                         min_box(:)       = (/box_x,box_y,box_z/)
                      endif
                    endif
                      

                    n2 = n2 + 1
                 enddo  ! ni2 
               enddo    ! nsp2

           enddo ! box_x
          enddo  ! box_y
         enddo   ! box_z  

         n1 = n1 + 1
      enddo ! ni1 
     enddo  ! ni2

     write(nconso,'(1x,a)') "Structure: "
     write(nconso,'(1x,a,f6.3)') "Structure: Minimum distance between any two atoms (Angstroem) : ",minimum_distance

     write(nconso,'(1x,a,i3,a,i3,a,1x,2(i2,1x),i2,a)') &
                "Structure: Mimimum distance found between atom",atom_pair(1),' and atom',atom_pair(2) , & 
                " (for box repetition ",min_box,")"

     write(nconso,'(1x,a)') "Structure: "

     if (minimum_distance<0.1d0) then    
        call abort_calc(nconso, &
                   "read_structure -> error : A too short distance between atoms found") 
     endif

     end subroutine check_atomic_positions

     
!###################################################################### 
      end module  ! read_structure_netcdf_module






