
! Appending and deleting columns from a QR factorization
! of a m by n matrix A=QR where Q is a m by n orthogonal
! matrix and R is n by n upper triangular.
!
! Q and R are in complex*16 precision


! The following subroutines ZDELC, ZINSC, ZORTHO, ZADFGR,ZAPLRC
! and ZAPLRR are slightly modified versions of the ones found in
! algorithm 686, Collected algorithms from ACM.
!
! A description is given in
!
!    L.Reichel and W.B. Gragg, FORTRAN subroutines for updating
!    the QR decomposition, ACM, Trans. Math. Software, 16(1990),
!    pp. 369-377
!
! The code has been modified so it uses standard BLAS and works
! with complex numbers (complex*16)
!

      SUBROUTINE ZDELC(Q,LDQ,M,N,R,LDR,K,V,INFO)

      IMPLICIT NONE
      INTEGER LDQ,M,N,LDR,K,INFO
      COMPLEX*16 Q(LDQ,N), R(LDR,N), V(M)
!                
!     LET THE MATRICES Q AND R BE A QR DECOMPOSITION OF A:=Q*R.
!     GIVEN Q AND R, ZDELC UPDATES THIS QR DECOMPOSITION WHEN THE
!     KTH COLUMN OF MATRIX A IS DELETED.
!        
!     ON ENTRY
!
!        Q       COMPLEX*16 (LDQ,N)
!                THE COLUMNS OF Q ARE ASSUMED ORTHONORMAL.
!                
!        LDQ     INTEGER
!                LDQ IS THE LEADING DIMENSION OF THE ARRAY Q.
!                
!        M       INTEGER
!                M IS THE NUMBER OF ROWS OF THE MATRIX Q.
!
!        N       INTEGER
!                N IS THE NUMBER OF COLUMNS OF THE MATRICES Q AND R,
!                AND THE NUMBER OF ROWS OF THE MATRIX R. N<=M REQUIRED.
!
!        R       COMPLEX*16 (LDR,N)
!                R IS AN UPPER TRIANGULAR N BY N MATRIX. A:=Q*R IS THE
!                MATRIX WHOSE KTH COLUMN IS DELETED.
!                
!        LDR     INTEGER
!                LDR IS THE LEADING DIMENSION OF THE ARRAY R.
!        
!        K       INTEGER
!                COLUMN K OF MATRIX A:=Q*R IS DELETED.
!
!        INFO    INTEGER
!                
!                INFO=1    V CONTAINS THE DELETED COLUMN ON RETURN.
!                
!                INFO=0    THE CONTENT OF V ON RETURN IS THE SAME AS
!                          ON ENTRY. V IS NOT REFERRED TO DURING THE
!                          COMPUTATIONS.
! 
!     ON RETURN
! 
!        Q       Q IS THE M BY N-1 MATRIX WITH ORTHONORMAL COLUMNS IN  
!                A QR DECOMPOSITION OF THE MATRIX OBTAINED BY DELETING
!                COLUMN K OF MATRIX A.
!                
!        R       R IS THE UPPER TRIANGULAR N-1 BY N-1 MATRIX IN A QR
!                DECOMPOSITION OF THE MATRIX OBTAINED BY DELETING
!                COLUMN K OF MATRIX A.
!                
!        V       COMPLEX*16 (N)
!                V CONTAINS THE DELETED COLUMN IF INFO=1 ON ENTRY.
! 
!        INFO    INFO=0     SUCCESSFUL TERMINATION.
!        
!                INFO=9     ERROR EXIT. K<1 OR K>N OR M<N OR LDQ<M
!                           OR LDR<N. NO COMPUTATIONS HAVE BEEN
!                           CARRIED OUT. ON RETURN Q AND R ARE THE
!                           SAME AS ON ENTRY.
!                          
!     INTERNAL VARIABLES
!
      INTEGER L,N1
      COMPLEX*16 CS,SN, zone, zzero
      DATA zone, zzero /(1.0d0,0.0d0), (0.0d0,0.0d0)/

!
!     ERROR EXIT. IF K<1 OR K>N OR M<N OR LDQ<M OR LDR<N, SET FLAG AND
!     RETURN.
!                
      IF(K.LT.1.OR.K.GT.N.OR.M.LT.N.OR.LDQ.LT.M.OR.LDR.LT.N)THEN
        INFO=9
        RETURN
      ENDIF
!        
!     COMPUTE V IF INFO=1.
!                           
      IF(INFO.EQ.1)THEN
         CALL ZSCAL(M,zzero,V,1)
        DO 10 L=1,K        
           call ZAXPY(M,R(L,K),Q(1,L),1,V,1)
 10     CONTINUE
      ENDIF
! 
!     UPDATE R AND Q.   
!
      INFO=0
      N1=N-1
      DO 20 L=K,N1
        CALL ZADFGR(R(L,L+1),R(L+1,L+1),CS,SN)
        CALL ZAPLRR(CS,SN,R,LDR,N,L,L+2)
        CALL ZAPLRC(CS,SN,M,Q(1,L),Q(1,L+1))
   20 CONTINUE
!
!     MOVE COLUMNS OF R.
!
      DO 30 L=K,N1
         call ZCOPY(L,R(1,L+1),1,R(1,L),1)
   30 CONTINUE              
!
!     SET NTH COLUMNS TO ZERO.
!
      call ZSCAL(N,zzero,R(1,N),1)
      call ZSCAL(M,zzero,Q(1,N),1)
      RETURN
      END

!---------------------------------------------------------------------
      SUBROUTINE ZINSC(Q,LDQ,M,N,R,LDR,K,V,RCOND,WK,INFO)

      IMPLICIT NONE
      INTEGER K,LDQ,M,N,LDR,INFO

      COMPLEX*16 Q(LDQ,N),R(LDR,N),V(M),WK(2*M+2*N)
      REAL*8 RCOND
!
!     LET THE MATRICES Q AND R BE A QR DECOMPOSITION OF A:=Q*R.
!     GIVEN Q AND R, ZINSC UPDATES THIS QR DECOMPOSITION WHEN THE
!     VECTOR V IS INSERTED BETWEEN COLUMNS K-1 AND K OF MATRIX A.
!
!     ON ENTRY
!
!        Q       COMPLEX*16 (LDQ,N)
!                THE COLUMNS OF Q ARE ASSUMED ORTHONORMAL.
!
!        LDQ     INTEGER
!                LDQ IS THE LEADING DIMENSION OF THE ARRAY Q.
!
!        M       INTEGER
!                M IS THE NUMBER OF ROWS OF THE MATRIX Q.
!
!        N       INTEGER
!                N-1 IS THE NUMBER OF COLUMNS OF THE MATRICES Q AND R,
!                AND THE NUMBER OF ROWS AND COLUMNS OF MATRIX R.
!                N<=M REQUIRED.
!
!        R       COMPLEX*16 (LDR,N)
!                R IS AN UPPER TRIANGULAR N-1 BY N-1 MATRIX. A:=Q*R IS
!                THE MATRIX TO BE UPDATED.
!     
!        LDR     INTEGER
!                LDR IS THE LEADING DIMENSION OF THE ARRAY R.
!     
!        K       INTEGER
!                VECTOR V IS INSERTED BETWEEN COLUMNS K-1 AND K OF
!                MATRIX A:=Q*R, 1<=K<=N.
!
!        V       COMPLEX*16 (M)
!                VECTOR V IS INSERTED BETWEEN COLUMNS K-1 AND K OF
!                MATRIX A:=Q*R.
!        
!        RCOND   REAL*8 
!                DO NOT INSERT V IF THE RECIPROCAL CONDITION NUMBER OF
!                THE MATRIX OBTAINED BY APPENDING COLUMN V/LENGHT(V) TO
!                MATRIX Q IS LESS THAN RCOND.
!                
!        WK      COMPLEX*16 (2*M+2*N)
!                WK IS A SCRATCH ARRAY.
!        
!     ON RETURN
!                
!        Q       Q IS THE M BY N MATRIX WITH ORTHONORMAL COLUMNS IN A
!                QR DECOMPOSITION OF THE MATRIX OBTAINED BY INSERTING
!                COLUMN V INTO MATRIX A.
!     
!        R       R IS THE UPPER TRIANGULAR N BY N MATRIX IN A QR
!                M BY N MATRIX WHOSE COLUMNS ARE V/LENGHT(V) AND THOSE
!                OF Q WHERE V AND Q ARE AS DEFINED ON ENTRY.
!        
!        INFO    INTEGER
!
!                INFO=0    SUCCESSFUL TERMINATION.
!                
!                INFO=1    ERROR EXIT. V WAS FOUND TO LIE IN THE RANGE
!                          OF Q NUMERICALLY. SUBROUTINE ZORTHO WAS
!                          UNABLE TO DETERMINE A VECTOR ORTHOGONAL TO
!                          THE COLUMNS OF Q. Q, R AND V ARE THE SAME
!                          AS ON ENTRY.
!                
!                INFO=2    THE M BY N MATRIX WITH COLUMNS Q AND
!                          V/LENGTH(V) WITH Q AND V AS GIVEN ON ENTRY
!                          HAS RECIPROCAL CONDITION NUMBER SMALLER
!                          THAN RCOND AS GIVEN ON ENTRY. Q, R AND V
!                          ARE THE SAME AS ON ENTRY.
!                
!                INFO=9    ERROR EXIT. K>0 OR K<N OR M<N OR LDQ<M OR
!                          LDR<N. Q, R, AND V ARE THE SAME AS ON ENTRY.
!
!
!     INTERNAL VARIABLES
!                
      INTEGER L,N1
!     DOUBLE PRECISION CS,SN
      complex*16 SN,CS

!                          
!     ERROR EXIT. IF K<1 OR K>N OR M<N OR LDQ<M OR LDR<N, SET FLAG AND
!     RETURN.
!                
      IF(K.LT.1.OR.K.GT.N.OR.M.LT.N.OR.LDQ.LT.M.OR.LDR.LT.N)THEN
        write(*,*) 'K=',K
        write(*,*) 'N=',N
        write(*,*) 'M=',M
        write(*,*) 'LDQ=',LDQ
        write(*,*) 'LDR=',LDR
        INFO=9
        RETURN
      ENDIF
!                          
!     ORTHOGONALIZE V AND INSERT ORTHOGONALIZED VECTOR INTO Q.
!
      N1=N-1  
      CALL ZORTHO(Q,LDQ,M,N1,V,WK,WK(N+1),RCOND,WK(M+N+1),INFO)

!     
!     GENERALLY IT IS NOT MEANINGFUL TO INSERT A COLUMN THAT IS
!     A LINEAR COMBINATION OF ALREADY EXISTING COLUMNS. EXIT IF
!     INFO IS NONVANISHING.
!
      IF(INFO.NE.0)RETURN
!
!     LET VECTOR WK(N+1:M+N+1), WHICH IS ORTHOGONAL TO Q, BE A NEW
!     LAST COLUMN OF Q. MOVE COLUMNS N-1, N-2, ..., K OF R.
!

      call ZCOPY(M,WK(N+1),1,Q(1,N),1)
      DO 10 L=N1,K,-1
         call ZCOPY(L,R(1,L),1,R(1,L+1),1)
   10 CONTINUE

      DO 20 L=K+1,N
        R(L,L)=0D0
   20 CONTINUE

!     ELEMENTS WK(K+1:N) ARE ANNIHILATED BY GIVENS REFLECTORS SO THAT
!     R BECOMES UPPER TRIANGULAR.
!     
      DO 30 L=N1,K,-1
        CALL ZADFGR(WK(L),WK(L+1),CS,SN)
        CALL ZAPLRR(CS,SN,R,LDR,N,L,L+1)
        CALL ZAPLRC(CS,SN,M,Q(1,L),Q(1,L+1))
   30 CONTINUE
      call ZCOPY(K,WK,1,R(1,K),1)
      RETURN
      END

!---------------------------------------------------------------------
      SUBROUTINE ZORTHO(Q,LDQ,M,N,W,S,V,RCOND,WK,INFO)

      IMPLICIT NONE
      INTEGER LDQ,M,N,INFO
      COMPLEX*16 Q(LDQ,N),S(N+1),W(M),V(M),WK(M+N)
      REAL*8 RCOND
!
!     THE MATRIX Q(1:M,1:N), M>=N, IS ASSUMED TO HAVE ORTHONORMAL
!     COLUMNS. ZORTHO ORTHOGONALIZES W AGAINST THE COLUMNS OF Q AND
!     RETURNS THE VECTOR SO OBTAINED IN V. IF M>N THEN V IS OF UNIT
!     LENGTH. IF M=N THEN V=0. S(1:N) IS A VECTOR OF FOURIER
!     COEFFICIENTS, I.E. S(1:N)=Q'*W WHERE THE PRIME DENOTES
!     TRANSPOSITION. S(N+1) IS THE DISTANCE FROM W TO THE RANGE OF Q.
!     IF M=N THEN S(N+1)=0.
!
!     ON ENTRY
!
!        Q       COMPLEX*16 (LDQ,N)
!                Q CONTAINS ORTHONORMAL COLUMNS.
!
!        LDQ     INTEGER
!                LDQ IS THE LEADING DIMENSION OF THE ARRAY Q.
!
!        M       INTEGER
!                M IS THE NUMBER OF ROWS OF THE MATRIX Q.
!
!        N       INTEGER
!                N IS THE NUMBER OF COLUMNS OF THE MATRIX Q. N MUST
!                NOT BE LARGER THAN M.
!
!        W       COMPLEX*16 (M)
!                W IS ORTHOGONALIZED AGAINST THE COLUMNS OF Q.
!
!        RCOND   REAL*8
!                RCOND IS A BOUND FOR THE RECIPROCAL CONDITION NUMBER
!                RKAPPA OF THE M BY N+1 MATRIX OBTAINED BY APPENDING
!                THE COLUMN W/LENGHT(W) TO Q. THE RECIPROCAL CONDITION
!                NUMBER IS WITH RESPECT TO THE LEAST SQUARES NORM. THE
!                COMPUTATIONS ARE TERMINATED AND W IS NOT
!                ORTHOGONALIZED AGAINST Q, IF RCOND>RKAPPA. RCOND=0
!                PREVENTS EARLY TERMINATION.
!
!        WK      COMPLEX*16 (M+N)
!                WK IS A SCRATCH ARRAY.
!
!     ON RETURN
!
!        V       COMPLEX*16 (M)
!                IF W DOES NOT LIE IN THE RANGE OF Q NUMERICALLY,
!                THEN V IS A UNIT VECTOR SUCH THAT RANGE(Q,V)=
!                RANGE(Q,W), AND V IS ORTHOGONAL TO THE COLUMNS OF Q.
!                IF W LIES IN THE RANGE OF Q NUMERICALLY, THEN V=0.
!
!        S       COMPLEX*16 (N+1)
!                S(1:N) CONTAINS THE FOURIER COEFFICIENTS OF W WITH
!                RESPECT TO THE COLUMNS OF Q, I.E. S(1:N)=Q'*W.
!                S(N+1) IS THE DISTANCE FROM W TO THE RANGE OF Q.
!
!        W       W IS THE SAME AS ON ENTRY.
!
!        RCOND   RCOND IS THE RECIPROCAL CONDITION NUMBER OF THE M BY
!                N+1 MATRIX OBTAINED BY APPENDING COLUMN W/LENGTH(W)
!                TO Q. THE CONDITION NUMBER IS WITH RESPECT TO THE
!                LEAST SQUARES NORM.
!
!        INFO    INTEGER
!
!                INFO=0    W DOES NOT LIE IN THE RANGE OF Q
!                          NUMERICALLY.
!
!                INFO=1    W WAS FOUND TO LIE IN THE RANGE OF Q
!                          NUMERICALLY. ON RETURN S(1:N)=Q'*W,
!                          S(N+1)=0, AND V=0.
!
!                INFO=2    THE RECIPROCAL CONDITION NUMBER OF THE M BY
!                          N+1 MATRIX OBTAINED BY APPENDING THE COLUMN
!                          W/LENGTH(W) TO Q IS SMALLER THAN THE ENTRY
!                          VALUE OF RCOND. ON RETURN RCOND CONTAINS
!                          THIS RECIPROCAL CONDITION NUMBER. S AND V
!                          ARE UNDEFINED.
!
!     INTERNAL VARIABLES
!
      DOUBLE PRECISION DZNRM2,SIGMAV,SIGMAW,SIGOLD,SIMAX,SIMIN,TMP

      COMPLEX*16 zone, zzero, ztemp
      DATA zone, zzero /(1.0d0,0.0d0), (0.0d0,0.0d0)/

!
!     IF M=N, SET FLAG, S(1:N)=Q'*W, S(N+1)=0, V=0 AND RETURN.
!
      IF(M.EQ.N)THEN
        INFO=1
        call ZGEMV('C',M,N,zone,Q,LDQ,W,1,zzero,S,1)
        S(N+1)=0D0
        call ZSCAL(M,zzero,V,1)
        RCOND=0D0
        RETURN
      ENDIF
!
!     THE LENGTH OF W IS STORED IN SIGMAW. THE LENGTH OF THE CURRENT
!     VECTOR V IS STORED IN SIGMAV. AFTER UPDATING V, THE PREVIOUS
!     VALUE OF SIGMAV IS STORED IN SIGOLD.
!
      SIGMAW=DZNRM2(M,W,1)
!
!     IF W=0, SET FLAG, S=V=0 AND RETURN.
!
      IF(SIGMAW.EQ.0D0)THEN
        INFO=1
        call ZSCAL(N+1,zzero,S,1)
        call ZSCAL(M,zzero,V,1)
        RCOND=0D0
        RETURN
      ENDIF
!
!     STORE SCALED W IN V.
!
      call ZCOPY(M,W,1,V,1)
      ztemp=dcmplx(1.0d0/SIGMAW,0.0d0)
      call ZSCAL(M,ztemp,V,1)
!
!     ORTHOGONALIZATION OF V.
!
      call ZGEMV('C',M,N,zone,Q,LDQ,V,1,zzero,S,1)
      SIMAX=DSQRT(1D0+DZNRM2(N,S,1))
      CALL ZGEMV('N',M,N,zone,Q,LDQ,S,1,zzero,WK(N+1),1)

      call ZAXPY(M,-zone,WK(N+1),1,V,1)
      SIGMAV=DZNRM2(M,V,1)
      SIMIN=SIGMAV/SIMAX
!
!     SIMAX = LARGEST SINGULAR VALUE OF MATRIX OBTAINED BY APPENDING
!     THE COLUMN W/LENGHT(W) TO MATRIX Q.
!     SIMIM = SMALLEST SINGULAR VALUE OF MATRIX OBTAINED BY APPENDING
!     THE COLUMN W/LENGHT(W) TO MATRIX Q.
!     RETURN IF PROBLEM TOO ILL-CONDITIONED.
!
      IF(SIMIN/SIMAX.LT.RCOND)THEN
        INFO=2
        RCOND=SIMIN/SIMAX
        RETURN
      ENDIF
      RCOND=SIMIN/SIMAX
!
!     CHECK IF V IS ORTHOGONAL TO THE COLUMNS OF Q. THE TEST USED
!     IS DISCUSSED IN PARLETT, THE SYMMETRIC EIGENVALUE PROBLEM,
!     PRENTICE-HALL, ENGLEWOOD CLIFFS, N.J., 1980, P. 107.
!
      IF(SIGMAV.GT.0.717D0)THEN
         ztemp=dcmplx(1.0d0/SIGMAV,0.0d0)
         call ZSCAL(M,ztemp,V,1)
         ztemp=dcmplx(SIGMAW,0.0d0)
         call ZSCAL(N,ztemp,S,1)

        S(N+1)=SIGMAV*SIGMAW
        INFO=0
        RETURN
      ENDIF
!
!     ONE REORTHOGONALIZATION.
!
      SIGOLD=SIGMAV
      call ZGEMV('C',M,N,zone,Q,LDQ,V,1,zzero,WK,1)
      call ZAXPY(N,zone,WK,1,S,1)
      call ZGEMV('N',M,N,zone,Q,LDQ,WK,1,zzero,WK(N+1),1)
      call ZAXPY(M,-zone,WK(N+1),1,V,1)
      SIGMAV=DZNRM2(M,V,1)
!
!     CHECK IF V IS ORTHOGONAL AGAINST THE COLUMNS OF Q.
!
      IF(SIGMAV.GT.0.717D0*SIGOLD)THEN
        ztemp=dcmplx(1.0d0/SIGMAV,0.0d0)
        call ZSCAL(M,ztemp,V,1)
        ztemp=dcmplx(SIGMAW,0.0d0)
        call ZSCAL(N,ztemp,S,1)
        S(N+1)=SIGMAV*SIGMAW
        INFO=0
        RETURN
      ENDIF
!
!     W LIES IN SPAN(Q) NUMERICALLY.
!
      INFO=1
      ztemp=dcmplx(SIGMAW,0.0d0)
      call ZSCAL(N,ztemp,S,1)
      S(N+1)=0D0
      call ZSCAL(M,zzero,V,1)
      RETURN
      END

!---------------------------------------------------------------------
      SUBROUTINE ZADFGR(X,Y,C,S)

      IMPLICIT NONE
!
!     ZADFGR DEFINES AND APPLIES THE SYMMETRIC GIVENS REFLECTOR
!
!
!             ( C   S )
!         G = (       )            C*C+S*S=1,
!             ( S  -C ),
!
!
!     FOR WHICH ( X, Y ) * G = ( Z, 0 ). REPLACES ( X, Y ) BY ( Z, 0 ).
!
!

      COMPLEX*16 C,S,T,X,X1,Y,Y1
      REAL*8 MU
      COMPLEX*16 zone, zzero
      data zone, zzero /(1.0d0,0.0d0), (0.0d0,0.0d0)/
!
      IF(Y.EQ.0D0)THEN
        C=zone
        S=zzero
        RETURN
      ENDIF
!
      MU=DMAX1(CDABS(X),CDABS(Y))
      X1=X/(dcmplx(MU,0.0d0))
      Y1=Y/(dcmplx(MU,0.0d0))
      t=MU*CDSQRT(X1*X1+Y1*Y1)
      if (real(x)<0) then
         t=-t;
      end if

      C=X/T
      S=Y/T
      X=T
      Y=zzero
      RETURN
      END


!---------------------------------------------------------------------
      SUBROUTINE ZAPLRC(C,S,N,X,Y)

      IMPLICIT NONE
!
!     ZAPLRC APPLIES A GIVENS REFLECTOR G DEFINED BY C AND S, WHERE
!     G HAS BEEN DETERMINED BY SUBROUTINE ZADFGR. WHEN CALLED ZAPLRC
!     REPLACES THE TWO COLUMN MATRIX ( X, Y )  BY ( X, Y ) * G.
!
      INTEGER I,N
      COMPLEX*16 S,C,TMP,X(N),Y(N)
!
      DO 10 I=1,N
        TMP=C*X(I)+S*Y(I)
        Y(I)=S*X(I)-C*Y(I)
        X(I)=TMP
   10 CONTINUE
      RETURN
      END

!---------------------------------------------------------------------
      SUBROUTINE ZAPLRR(C,S,A,LDA,N,IRW,L)

      IMPLICIT NONE
!
!     ZAPLRR APPLIES A GIVENS REFLECTOR G DEFINED BY C AND S, WHERE
!     G HAS BEEN DETERMINED BY SUBROUTINE ZADFGR. WHEN CALLED ZAPLRR
!     REPLACES THE TWO ROW MATRIX ( A(IRW,J), A(IRW+1,J) )', L<=J<=N,
!     BY G * ( A(IRW,J), A(IRW+1,J) )', L<=J<=N.
!
!     ON ENTRY
!
!        C, S    COMPLEX*16
!                C AND S DEFINE A GIVENS REFLECTOR G.
!
!        A       COMPLEX*16 (LDA,N)
!                G IS APPLIED TO (PARTS OF) TWO ROWS OF THE MATRIX A.
!
!        LDA     INTEGER
!                LDA IS THE LEADING DIMENSION OF THE ARRAY A.
!
!        N       INTEGER
!                N IS THE NUMBER OF ROWS AS WELL AS THE NUMBER OF
!                COLUMNS OF THE MATRIX A.
!
!        IRW     INTEGER
!                IRW AND IRW+1 ARE THE ROW INDICES OF THE ROWS
!                OF THE MATRIX A TO WHICH G IS APPLIED.
!
!        L       INTEGER
!                G IS APPLIED TO THE 2 BY N-L+1 MATRIX
!                ( A(IRW,J), A(IRW+1,J) ), J=L,L+1,...,N.
!
!     ON RETURN
!
!        A       ( A(IRW,J), A(IRW+1,J) ), J=L,L+1,...,N, IS OBTAINED
!                BY MULTIPLYING ROWS IRW AND IRW+1, AND COLUMNS
!                L,L+1,...,N OF THE MATRIX A GIVEN ON ENTRY BY THE
!                MATRIX G.
!
      INTEGER I,IRW,L,LDA,N
      COMPLEX*16 S,C,TMP,A(LDA,N)
!
      DO 10 I=L,N
        TMP=C*A(IRW,I)+S*A(IRW+1,I)
        A(IRW+1,I)=S*A(IRW,I)-C*A(IRW+1,I)
        A(IRW,I)=TMP
   10 CONTINUE
      RETURN
      END
