!##############################################################
!  This module simplifies the use of netCDF I/O
!  by defining generic I/O operations using overloaded
!  I/O symbols.
!  Only the var, vara, var1 families is presently resolved
!  Avoid defining symbols: nf_*
!  local (=module parameters) : nfif_*
!  This interface relies on the type conversion applied by 
!  the netCDF library
!  Subroutines that accept "ncid" input arguments require
!  that the data set is open
!  No assumptions are made about the data set mode (define/data):
!  each subroutine invokes nfsetmode, just before it is needed,
!  to enable external efficiency blocking of netCDF calls
!---------------------------------------------- 
!  entry   : is_a_variable_name
!  syntax  : function is_a_variablename(name)
!  return  : logical
!  purpose : determine whether name is a variable, based on
!            the syntax [var[%att]]
!----------------------------------------------
!  entry   : is_an_attribute_name
!  syntax  : function is_an_attribute(name)
!  return  : logical
!  purpose : determine whether name is an attribute, based on
!            the syntax [var[%att]]
!----------------------------------------------
!  entry   : get_variablename
!  syntax  : function get_variablename(name,errorstring)
!  return  : character(len=nf_max_name)
!  purpose : Derive variable name A from the syntax A[%B]
!            A == "" is allowed (this indicates a global attribute)
!----------------------------------------------
!  entry   : get_attributename
!  syntax  : function get_attributename(name,errorstring)
!  return  : character(len=nf_max_name)
!  purpose : Derive attribute name B from the syntax A[%B]
!            "%" is present and B == ""  its considered a fatal exception
!            "%" is not present, B is returned as an empty string
!----------------------------------------------
!  entry   : as_string4 
!  syntax  : function as_string4 (integer :: i) 
!  return  : character(len = 4)
!  purpose : Cast integer i to a string(len = 4)
! 
!----------------------------------------------
!  entry   : nfsetmode
!  syntax  : subroutine nfsetmode(ncid, target_mode)
!  return  : none
!  purpose : Set the mode of an open netCDF data set ncid to targetmode 
!            without flagging the situation that the set is already in 
!            targetmode, if this is the case.
!----------------------------------------------
!  entry   : nfget
!  syntax  : integer nfget(ncid, variablename, variable, OPTIONALS)
!  return  : error code (see table below)
!  purpose : get a dataelement/attribute from a netCDF data set
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) variablename : for looking after in the netCDF data set [in]
!                                    If variablename has the form "A%B" the attribute B
!                                    to variable A is attemted loaded.                   
!        <...>  variable           : where data is loaded into [out]
!                                    Type/rank optional with restrictions
!        integer OPTIONALS         : index(*), start(*), count(*)
!                                    used for partial data load  [in]
!---------------------------------------------- 
!  entry   : nfput
!  syntax  : integer nfput(ncid, variablename, variable, OPTIONALS)
!  return  : error code (see table below)
!  purpose : put a dataelement/attribute to a netCDF data set
!            If the variable with name 'variablename' do not exists in the data set
!            it is created, with the list of dimensions given as optional variables 
!            (see below) 
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) variablename : name of the variable/attribute in the netCDF data set [in]
!                                    If variablename has the form "A%B" the attribute B
!                                    to variable A is added to the dataset.
!        <...>  variable           : where data is taken from [in]
!                                    Type/rank optional with restrictions
!        OPTIONALS
!        integer                   : index(*), start(*), count(*)
!                                    used for partial data write  [in]
!        character                 : dim_name1 = "dim_name1",..,dim_name6="dim_name6" [in]
!                                    list of dimensions specified then creating a new variable
!                        
!----------------------------------------------                                 
!  entry   : nfgetvardim
!  syntax  : integer nfgetvardim(ncid, name, datadim) 
!  return  : error code (see table below)
!  purpose : get dimensions for a dataelement/attribute from a netCDF data set
!
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) variablename : for looking after in the netCDF data set [in]
!                                    Variablename of the form "A%B" refers to
!                                    attribute B of variable A                
!        integer, pointer :: datadim(:) var/att dimensions [out]
!---------------------------------------------- 
!  entry   : nfgetglobaldim
!  syntax  : integer nfgetglobaldim(ncid, name, dimlen) 
!  return  : error code (see table below)
!  purpose : get dimensions for a dataelement/attribute from a netCDF data set
!
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) name         : for looking after in the netCDF data set [in]  
!        integer dimlen            : length of the sought dimension in data set [out]                     
!---------------------------------------------- 
!  entry   : nfputglobaldim
!  syntax  : integer nfputglobaldim(ncid, name, dimlen)
!  return  : error code (see table below)
!  purpose : put a new dimension for a netCDF data set
!
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) name         : name of new dimension in the netCDF data set [in]
!        integer dimlen            : length of the new dimension [in]
!----------------------------------------------
!  entry   : nfdefvar
!  syntax  : integer nfdefvar(ncid, variablename, TYPE, [dim_name1 = "dim_name1",..,dim_name6="dim_name6"])
!  return  : error code (see table below)
!  purpose : define a new variable im a netCDF data set
!            This is needed, when a varaible is written segment wise (so that
!            an nfput() call are not able to compute the total size of the variable
!            from a given variable segment
!
!        integer ncid              : netCDF data set ID, obtained from nf_open [in]
!        character(*) variablename : name of new variable in the netCDF data set [in]
!        integer TYPE              : type of the new variable 
!                                       NF_CHAR(character), NF_INT(integer),NF_REAL(real),
!                                       NF_DOUBLE(double)
!        list of dimensions        : optional names of dimensions 
!                                    (from 0 to 6 dimensions are allowed) [in]
!  example : to define a 2 dimensional double precision variable 'newvar' using the two netCDF
!            dimensions 'dim1','dim2' : call nfdefvar(ncid,'newvar',NF_DOUBLE,'dim1','dim2') 
!            (note keyword are not neccesary for the optional parameters).  
!  example : to define a integer scalar : call nfdefvar(ncid,'newvar',NF_INT)
!
!----------------------------------------------
!  entry   : copy_netCDF_variable
!  syntax  : function copy_netCDF_variable(ncid_in, ncid_out,variableid,variablename)
!  return  : integer nfif_* error code
!  purpose : copy one netCDF variable from source to target netCDF file
!            Thin wrapper to transfer_netcdf_variable/copy_netcdf_vardef                     
!================================================================= 
!  entry   : copy_netcdf_vardef
!  syntax  : subroutine copy_netcdf_varattributes(ncid_in, ncid_out, varid_in, varid_out)
!  return  : none
!  purpose : copy all attributes of variable with ID varid_in  in open netcdf set ncid_in
!                                to variable with ID varid_out in open netcdf set ncid_out
!            additive, i.e. existing attributes of varid_out is not deleted
!================================================================= 
!  entry   : transfer_netcdf_variable
!  syntax  : subroutine transfer_netcdf_variable(ncid_in, ncid_out, varid_in, varid_out)
!  return  : none
!  purpose : Raw transfer of variable varid_in  in data set ncid_in
!            to variable varid_out in data set ncid_out   
!================================================================= 
!    
! REMARKS : IBM xlf90 (4.1) has a bug :
!  write(*,*)  nfget(...)                     => coredump
!  status = nfget(...) ; write(*,*) status    is OK
!
! 
!  Asbjorn.08Jun1999
!##############################################################
module netcdfinterface
implicit none
#include  "netcdf.inc"

!-------------------------------------------------
interface nfget        ! branch on type/rank of variable

  module procedure  nfget_rank0_text       
  module procedure  nfget_rank1_text       

  module procedure  nfget_rank0_int
  module procedure  nfget_rank1_int        
  module procedure  nfget_rank2_int
  module procedure  nfget_rank3_int
  module procedure  nfget_rank4_int
  module procedure  nfget_rank5_int
  module procedure  nfget_rank6_int

  module procedure  nfget_rank0_real
  module procedure  nfget_rank1_real        
  module procedure  nfget_rank2_real
  module procedure  nfget_rank3_real
  module procedure  nfget_rank4_real
  module procedure  nfget_rank5_real
  module procedure  nfget_rank6_real

  module procedure  nfget_rank0_double
  module procedure  nfget_rank1_double        
  module procedure  nfget_rank2_double
  module procedure  nfget_rank3_double
  module procedure  nfget_rank4_double
  module procedure  nfget_rank5_double
  module procedure  nfget_rank6_double

end interface
!-------------------------------------------------
interface nfput        ! branch on type/rank of variable

  module procedure  nfput_rank0_text                                       
  module procedure  nfput_rank1_text                                       

  module procedure  nfput_rank0_int        
  module procedure  nfput_rank1_int        
  module procedure  nfput_rank2_int
  module procedure  nfput_rank3_int
  module procedure  nfput_rank4_int
  module procedure  nfput_rank5_int          
  module procedure  nfput_rank6_int          

  module procedure  nfput_rank0_real
  module procedure  nfput_rank1_real        
  module procedure  nfput_rank2_real
  module procedure  nfput_rank3_real
  module procedure  nfput_rank4_real
  module procedure  nfput_rank5_real
  module procedure  nfput_rank6_real

  module procedure  nfput_rank0_double
  module procedure  nfput_rank1_double       
  module procedure  nfput_rank2_double
  module procedure  nfput_rank3_double
  module procedure  nfput_rank4_double
  module procedure  nfput_rank5_double                 
  module procedure  nfput_rank6_double                 
end interface 

interface Reorder_atomvector
  module procedure  Reorder_atomvector_integer
  module procedure  Reorder_atomvector_real8
end interface

!-------------------------------------------------
integer, parameter :: nfif_len_errorstring      = 2000   ! internal error tracing
!-------------------------------------------------
integer, parameter          :: nfif_define_mode = 82345534   ! for nfsetmode
integer, parameter          :: nfif_data_mode   = 37712399   ! for nfsetmode
!-------------------------------------------------
!  return code  :  0  : read  : variable/attribute existed in data set + successful data load
!                       write : sucessfull data write 
!                       define: variable/dimension/attribute defined sucessfully
!   
!                  100: define: dimension did allready exist in data set, but size is OK
!                  101: define: variable  did allready exist in data set, but definition is OK
!
!                 -1  : variable/attribute existed in data set, but unsuccessful data load
!                 -2  : variablename did not exist in data set
!                 -3  : attribute did not exist in data set
!                 -5  : dimension allready exist in data set 
!                 -6  : data set in read-only mode (nf_eperm)
!                 -7  : variable allready used
 
integer, parameter :: nfif_OK              =  0       !  note : parameter (nf_noerr = 0)

integer, parameter :: nfif_vardidnotexist  = -1
integer, parameter :: nfif_attdidnotexist  = -2
integer, parameter :: nfif_dimdidnotexist  = -3
integer, parameter :: nfif_foundscalarvariable = -4
integer, parameter :: nfif_dimexist = -5
integer, parameter :: nfif_readonly = -6
integer, parameter :: nfif_varinuse = -7

! --- non fatal circumstances ---

integer, parameter :: nfif_dimexist_butOKsize = 100
integer, parameter :: nfif_varexist_butOKdef  = 101
!-------------------------------------------------
! this array contains the mapping between user specified 
! atom order in the netCDF file and the internal 
! (by species sorted) atom order in dacapo.
!------------------------------------------------
integer, pointer   :: Internal_to_netCDF_order(:)  
       
                                         
!================================================
!  >>>>>>>  Visibility control section   <<<<<<<
!================================================

private   :: local_error_handler
private   :: get_nfvar_info

!##############################################################
!==============================================================
contains
!==============================================================
!#############################################################

!#############################################################
logical function is_an_attribute_name(name)
! --------------------------------------------------
!  determine whether name is an attribute, based on
!  the syntax [var[%att]]
! --------------------------------------------------
character*(*),intent(in)    :: name

  is_an_attribute_name = .false.
  if (len(name) == 0) return
  if (index(name,"%") > 0) is_an_attribute_name = .true.
 
end function is_an_attribute_name

!#############################################################
logical function is_a_variable_name(name)
! --------------------------------------------------
!  determine whether name is a variable, based on
!  the syntax [var[%att]]
! --------------------------------------------------
character*(*),intent(in)    :: name

  is_a_variable_name = .false.
  if (len(name) == 0) return
  if (index(name,"%") == 0) is_a_variable_name = .true.
  
end function is_a_variable_name


!#############################################################
function get_variablename(name,errorstring)
!--------------------------------------------------------------
!  Derive variable name A from the syntax A[%B]
!  A == "" is allowed (this indicates a global attribute)
!--------------------------------------------------------------
implicit none 
character(len=nf_max_name)  :: get_variablename
character*(*),intent(in)    :: name
character*(*),intent(inout) :: errorstring ! history
!-----------------------------
character(*),parameter      :: myname     = "get_variablename"    
integer                     :: idx
character(len=len(name))    :: namebuffer
! ===========================================================
errorstring = trim(errorstring) // "\n" // trim(myname) // ": "  ! report entry
errorstring = trim(errorstring) // " inputname = " // trim(name)

namebuffer = adjustl(name)
get_variablename = ""
idx = index(namebuffer,"%")

if (idx == 0) then                   ! name is not an attribute
   get_variablename = namebuffer  
elseif (idx == 1) then               ! name is a global attribute
   get_variablename = ""
else                                 ! name is an attribute like A%B
   get_variablename = namebuffer(:idx-1) 
endif 

errorstring = trim(errorstring) // "   resolved variablename = " //  trim(get_variablename)
end function get_variablename

!##############################################################
function get_attributename(name,errorstring)
!--------------------------------------------------------------
!  Derive attribute name B from the syntax A[%B]
!  "%" is present and B == ""  its considered a fatal exception
!  "%" is not present, B is returned as an empty string
!--------------------------------------------------------------
implicit none 
character(len=nf_max_name)  :: get_attributename
character*(*),intent(in)    :: name
character*(*),intent(inout) :: errorstring ! history
!-----------------------------
character(*),parameter      :: myname     = "get_attributename"    
integer                     :: idx
character(len=len(name))    :: namebuffer
! ===========================================================
errorstring = trim(errorstring) // "\n" // trim(myname) // ": "  ! report entry
errorstring = trim(errorstring) // " inputname = " // trim(name)

namebuffer = adjustl(name)
get_attributename = ""
idx = index(namebuffer,"%")
if (idx == 0) then                    ! name is not an attribute  
   get_attributename = ""  
elseif (idx == 1) then                ! name is a global attribute
   get_attributename = namebuffer(2:) 
elseif (idx == len_trim(namebuffer)) then   ! name is like A% - abort
   call local_error_handler(nf_noerr, errorstring, "variablename like var% invalid")
else                                  ! name is an attribute like A%B
   get_attributename = namebuffer(idx+1:) 
endif 

errorstring = trim(errorstring) // "   resolved attributename = " &
                                // trim(get_attributename)
end function get_attributename
!#############################################################
subroutine local_error_handler(status,errorstring,msg)
!-------------------------------------------------------------
!  This local error handler is called to resolve fatal
!  error conditions (program execution is halted)
!  The stopping sequence may be costumized at the end
!  of local_error_handler by calling my_stopping_sequence()
!-------------------------------------------------------------
implicit none 
integer,  intent(in)        :: status      ! netCDF recent error token
character*(*),intent(in)    :: errorstring ! history
character*(*),intent(in)    :: msg         ! last error message

if (status /= nf_noerr)  then
  write(*,*) "netCDF library flagged error"
  write(*,*) " * netCDF error number:", status
  write(*,*) " * netCDF error message = ", nf_strerror(status) 
else
  write(*,*) "netCDF library did not flag error" 
endif

write(*,*) "Error trace :\n", trim(errorstring)
write(*,*) "last error message =", trim(msg)

! --------------------------------------------------
!  abort via abort_calc(); other applications
!  of the netCDF interface may want to use another
!  stopping sequence at this point
!  For the time being: provide UNIT=6 explicitly here 
! --------------------------------------------------

call abort_calc(6, "stopped by netCDFinterface -> local_error_handler")


end subroutine local_error_handler


!###################################################################
function as_string4 (i) result (the_string)
!=================================================================
! Cast integer i to a string(len = 4)
!=================================================================
integer             :: i
character(len = 4)  :: the_string
write(the_string, '(i4)') i

end function as_string4


!#################################################################
subroutine nfsetmode(ncid, targetmode)
!=================================================================
! Set the mode of an open netCDF data set ncid to targetmode 
! without flagging the situation that the set is already in 
! targetmode, if this is the case. All other netCDF exceptions 
! are considered fatal
! The netCDF library does not offer functionality to intercept
! cases where the set ncid in READ-ONLY
! targetmode may be either nfif_define_mode/nfif_data_mode
! (exported module data). All other values for targetmode are 
! considered exceptions.
! This is the only place, where nf_redef/nf_enddef is called
! explicitly, to enable external efficiency blocking of netCDF calls
!=================================================================
integer,   intent(in)      :: ncid          ! netCDF fileset ID
integer,   intent(in)      :: targetmode    ! 
! --- locals ---
integer                             :: status
character(len=nfif_len_errorstring) :: errorstring = "subroutine nfsetmode"
! ----------------------------------------------------------------
if     (targetmode == nfif_define_mode) then

   status = nf_redef(ncid)
   if ((status /= nf_noerr).and.(status /= nf_eindefine)) &
      call local_error_handler(status, errorstring, "NFerror at nf_redef")

elseif (targetmode == nfif_data_mode)   then

   status = nf_enddef(ncid)
   if ((status /= nf_noerr).and.(status /= nf_enotindefine)) &
      call local_error_handler(status, errorstring, "NFerror at nf_enddef")

else 

   call local_error_handler(nf_noerr, errorstring, ": unknown targetmode")

endif     
end subroutine nfsetmode
   
                                          
!#############################################################
function nfgetglobaldim(ncid, name, dimlen) result(return_code)
implicit none
integer,      intent(in)      :: ncid          ! netCDF fileset ID
character*(*),intent(in)      :: name          ! dimension name
integer                       :: dimlen        ! length of dimension
!--------------------------------------------------------------
integer                             :: return_code                   
character(*),parameter              :: myname      = "nfgetglobaldim"
character(len=nfif_len_errorstring) :: errorstring = "", tmpstring
integer                             :: status, dimid
!============================================================== 
return_code = nfif_OK
 
errorstring = "\n" // trim(myname) // ": "  ! report entry
write(tmpstring, *) "\n argument ncid     = ", ncid, &
                    "\n argument name     = ", name

errorstring =  trim(errorstring) // trim(tmpstring)

!---- get the dimension ID ---

status = nf_inq_dimid(ncid, name, dimid)
if (status == nf_ebaddim) then
   return_code = nfif_dimdidnotexist
   return
endif
if (status /= nf_noerr) &
     call local_error_handler(status,errorstring,"NFerror at nf_inq_dimid")

!---- get the dimension length --

status = nf_inq_dimlen(ncid, dimid, dimlen)
if (status /= nf_noerr) &
     call local_error_handler(status,errorstring,"NFerror at nf_inq_dimlen")

end function nfgetglobaldim


!#############################################################
function nfputglobaldim(ncid, name, dimlen) result(return_code)
implicit none
integer,      intent(in)      :: ncid          ! netCDF fileset ID
character*(*),intent(in)      :: name          ! dimension name
integer, intent(in)           :: dimlen        ! length of dimension
!--------------------------------------------------------------
integer                             :: return_code
character(*),parameter              :: myname      = "nfputglobaldim"
character(len=nfif_len_errorstring) :: errorstring = "", tmpstring
integer                             :: status, dimid, existing_dimlen 
!==============================================================
errorstring = "\n" // trim(myname) // ": "  ! report entry
write(tmpstring, *) "\n argument ncid     = ", ncid, &
                    "\n argument name     = ", name

errorstring =  trim(errorstring) // trim(tmpstring)
 
! ------------------------------------------
!   check if the dimension is already set
!   if it is set, check its size, else define it
! ------------------------------------------
status = nfgetglobaldim(ncid, name, existing_dimlen)

if (status == nfif_dimdidnotexist) then

    call nfsetmode(ncid, nfif_define_mode)    ! must be in define mode

    !---- create the dimension ---
    status = nf_def_dim(ncid, name, dimlen, dimid) 
    if (status==nf_noerr) then 
       return_code = nfif_ok
    elseif (status == nf_enameinuse) then 
       call local_error_handler(status,errorstring,"nfputglobaldim: ??") 
    else
       call local_error_handler(status,errorstring,"NFerror at nf_def_dim") 
    endif
     
else      ! the requested dimension did already exist - check size == dimlen

    return_code = nfif_dimexist
    if (dimlen == existing_dimlen) return_code = nfif_dimexist_butOKsize 
    
endif     ! if (status == nfif_dimdidnotexist ...

end function nfputglobaldim


!#############################################################
function nfdefvar(ncid, name_var, type, dim_name1,dim_name2,dim_name3, & 
                  dim_name4,dim_name5,dim_name6,varidpar) result(return_code)
!============================================================================
!  Check, if a variable: name_var in netCDF set with ID ncid (and in open state)
!  is defined. If defined, check that types/dimensions are as required
!  by type/dim_name*. If so, just return varid if present(varidpar) - else dump
!  
!  If not defined, define name_var according to type/dim_name.
!  Succesful variable definition:                   return_code = nf_noerr 
!  Variable already defined, but definition is OK : return_code = nfif_varexist_butOKdef
!============================================================================
implicit none
integer,      intent(in)      :: ncid          ! netCDF fileset ID
character*(*),intent(in)      :: name_var      ! netCDF name of variable to define
integer, intent(in)           :: type          ! netCDF type (FLOAT/DOUBLE ..)  
character*(*), optional, intent(in) & 
                              :: dim_name1,dim_name2,dim_name3,dim_name4,dim_name5,dim_name6 
                                               ! name of dimensions (0 to 6 dimensions are allowed)
integer, optional, intent(out):: varidpar      ! optional variable for returning the id of the 
!                                                created variable 
!--------------------------------------------------------------
integer                             :: return_code
character(*),parameter              :: myname      = "nfdefvar"
character(len=nfif_len_errorstring) :: errorstring = "", tmpstring
integer                             :: status,dimid(6),varid,rank,ranknr
integer                             :: natts, xtype, nvdims, dimIDset(100)
character(40)                       :: dim_names(6), dummy_name
character(99)                       :: msg
!==============================================================
return_code = nfif_OK

errorstring = "\n" // trim(myname) // ": "  ! report entry
write(tmpstring, *) "\n argument ncid     = ", ncid, &
                    "\n argument name     = ", name_var

errorstring =  trim(errorstring) // trim(tmpstring)

! --------------------------------------------------
! resolve dim_names input and variable rank
! -------------------------------------------------- 

! --- find the rank of the new variable
rank = 0
if (present(dim_name1)) then 
   dim_names(1) = dim_name1;  rank = 1
endif
if (present(dim_name2)) then 
   dim_names(2) = dim_name2;  rank = 2
endif
if (present(dim_name3)) then 
   dim_names(3) = dim_name3;  rank = 3
endif
if (present(dim_name4)) then 
   dim_names(4) = dim_name4;  rank = 4
endif
if (present(dim_name5)) then 
   dim_names(5) = dim_name5;  rank = 5
endif
if (present(dim_name6)) then 
   dim_names(6) = dim_name6;  rank = 6
endif

! --- get dimensions id
do ranknr = 1,rank
  status = nf_inq_dimid(ncid,dim_names(ranknr),dimid(ranknr)) 
  if (status/=nf_noerr) then 
     if (status == nf_ebaddim) then
        return_code = nfif_dimdidnotexist
        return
     else
        call local_error_handler(status,errorstring,"NFerror at nf_ing_dimid")
     endif 
  endif
enddo


status = nf_inq_varid(ncid,name_var,varid) 

if     (status == nf_noerr)   then   !!!==== Is defined; check that the definition is OK

      status = nf_inq_var(ncid, varid, dummy_name, xtype, nvdims, dimIDset, natts)
      if (status/=nf_noerr) &
            call local_error_handler(status,errorstring,"NFerror at nf_inq_var")
      if (xtype /= type) &
            call local_error_handler(status,errorstring,"nfdefvar: type redef not possible")
      if (nvdims /= rank) &  
            call local_error_handler(status,errorstring,"nfdefvar: rank redef not possible")

      do ranknr = 1, rank      ! OK for rank == 0
         if (dimIDset(ranknr) /= dimid(ranknr)) then
            write(msg,'(a,1x,i3)') "nfdefvar: dimension mismatch for dimension", ranknr
            call local_error_handler(nf_noerr, errorstring, msg)
         endif
      enddo

      return_code = nfif_varexist_butOKdef      ! flag defined, but check was OK

elseif (status == nf_enotvar) then   !!!==== name_var not defined - define it ======== !!!

      call nfsetmode(ncid, nfif_define_mode)

! ---- create the variable ---

      status = nf_def_var(ncid, name_var, type, rank, dimid,varid)
      if (status==nf_noerr) then
         return_code = nf_noerr
      elseif (status == nf_enameinuse) then
         call local_error_handler(status,errorstring,"unexpected, unhandled error in nfdefvar") 
      else
         call local_error_handler(status,errorstring,"NFerror at nf_def_var")
      endif

else
      call local_error_handler(status,errorstring,"Unhandled NFerror at nfdefvar")

endif  !!! ========  name_var defined ...      ============ !!!


if (present(varidpar)) varidpar = varid    ! optionally return id of the new variable

end function nfdefvar



!#############################################################
function nfgetvardim(ncid, name, datadim)   result(return_code)
implicit none
integer,      intent(in)      :: ncid          ! netCDF fileset ID
character*(*),intent(in)      :: name          ! var/att name
integer, pointer              :: datadim(:)    ! var/att dimensions (allocated here)
!--------------------------------------------------------------
integer                             :: return_code                   
character(*),parameter              :: myname      = "nfgetvardim"
character(len=nfif_len_errorstring) :: errorstring = "", tmpstring
character(len=nf_max_name)          :: varname, attribname
integer                             :: status, varid, attlen, varndims
integer                             :: idim, istat, dimid
!==============================================================  
errorstring = "\n" // trim(myname) // ": "  ! report entry
write(tmpstring, *) "\n argument ncid     = ", ncid, &
                    "\n argument name     = ", name
errorstring =  trim(errorstring) // trim(tmpstring)


! --- inquire about the variable ---

varname = get_variablename(name,errorstring)

if (len_trim(varname) == 0) then              ! asking for a global attribute?
  if (is_an_attribute_name(name)) then        ! set the global dummy variable
     varid = nf_global                        ! else flag name error
  else
     call local_error_handler(status,errorstring, name//": is invalid")
  endif
else
  status = nf_inq_varid(ncid,varname,varid)   ! inquire about the variable
  if (status == nf_enotvar) then              ! variable non existent - return
     return_code = nfif_vardidnotexist        
     return              
  elseif (status /= nf_noerr) then            ! other errors considered fatal
     call local_error_handler(status,errorstring,"NFerror at nf_inq_varid")
  endif
endif

! --- get dimensions, either variable/attribute  ---

if (is_an_attribute_name(name)) then      
                                   
!  attributes always have rank = 1, possibly len=1

  attribname = get_attributename(name,errorstring)
  status = nf_inq_attlen(ncid,varid,attribname,attlen)        
  if (status == nf_enotatt) then              ! attribute non existent - return
     return_code = nfif_attdidnotexist
     return  
  elseif (status /= nf_noerr) then
     call local_error_handler(status,errorstring,"NFerror at nf_inq_attlen") 
  else
     if (associated(datadim)) nullify(datadim)         ! clean up
     allocate (datadim(1), STAT=status )               ! attributes always have rank 1
     if (status /= 0) &
	call local_error_handler(nf_noerr,errorstring,"error allocating datadim(:) for attribute")
     datadim(1)  = attlen
     return_code = nfif_OK
     return
  endif

elseif  (is_a_variable_name(name)) then 
                                          
  status = nf_inq_varndims(ncid, varid, varndims)      ! 1) get #dimensions for this var
  if (status   /= nf_noerr)  &                         !    do not test on existence 
    call local_error_handler(status,errorstring,"NFerror at nf_inq_varndims") 
  write(tmpstring, *) "\n number of dimensions =    = ", varndims
  errorstring =  trim(errorstring) // trim(tmpstring)  

  if (associated(datadim)) nullify(datadim)         ! clean up

  if (varndims > 0) then
    allocate (datadim(varndims), STAT=istat)        ! allocate dimension vector
    if (istat  /= 0) &
      call local_error_handler(nf_noerr,errorstring,"error allocating datadim(:) for variable")
  elseif (varndims == 0) then
    allocate (datadim(1), STAT=istat)               ! this is a scalar, varndims==0
    if (istat  /= 0) &
      call local_error_handler(nf_noerr,errorstring,"error allocating datadim(1) for scalar variable")  
    datadim(1) = 1
    return_code = nfif_foundscalarvariable
    return
  else
    call local_error_handler(nf_noerr,errorstring,"number of dimensions (varndims) < 0 for variable")  
  endif

  status = nf_inq_vardimid(ncid, varid, datadim)          ! 2) get dimension IDs
  if (status  /= nf_noerr) &
     call local_error_handler(status,errorstring,"NFerror at nf_inq_vardimid") 
  write(tmpstring, *) "\n dimension ID's=    = ", datadim(:)
  errorstring =  trim(errorstring) // trim(tmpstring)  

  do idim = 1, varndims                                   ! loop over dimensions          
     dimid = datadim(idim)
     status = nf_inq_dimlen(ncid, dimid, datadim(idim))   ! 3) get dimension length
     if (status  /= nf_noerr) then
        write(tmpstring,*) "NFerror at nf_inq_dimlen, dimension # ", idim
        call local_error_handler(status,errorstring,trim(tmpstring)) 
     endif
  enddo
  write(tmpstring, *) "\n variable dimensions = ", datadim(:)
  errorstring =  trim(errorstring) // trim(tmpstring)  

  return_code = nfif_OK
  return 

else

  call local_error_handler(nf_noerr, errorstring, &
                           name//": is neither attribute/variable")

endif        ! if attribute/variable


end function nfgetvardim

!##############################################################
!#                Expand nfget template                       #
!##############################################################

#define   NFGET_TEMPLATE     nfget_int
#define   TRACENAME         "nfget_int"
#define   NFGET_VARDECLAR    integer :: variable(*)    
#define   NFGET_ATT          nf_get_att_int
#define   NFGET_VAR1         nf_get_var1_int
#define   NFGET_VARA         nf_get_vara_int
#define   NFGET_VAR          nf_get_var_int
#include "nfget.template"
! ======= Dummy resolvents for :: variable = integer, rank==0 special  =========
#define  RANK_0                             
#define  NFGET_WRAPPER                      nfget_rank0_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable, aux(1)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank1_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"                                                  
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank2_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:,:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank3_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:,:,:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank4_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:,:,:,:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank5_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:,:,:,:,:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank6_int
#define  NFGET_WRAPPER_VARDECLAR integer :: variable(:,:,:,:,:,:)
#define  NFGET_CORE                         nfget_int
#include "nfget_wrapper.template"
!-----------------------------------------------------------                  
!##############################################################

#define   NFGET_TEMPLATE     nfget_real
#define   TRACENAME         "nfget_real"
#define   NFGET_VARDECLAR    real :: variable(*)    
#define   NFGET_ATT          nf_get_att_real
#define   NFGET_VAR1         nf_get_var1_real
#define   NFGET_VARA         nf_get_vara_real
#define   NFGET_VAR          nf_get_var_real
#include "nfget.template"
! ======= Dummy resolvents for :: variable = real, rank==0 special  =========
#define  RANK_0                             
#define  NFGET_WRAPPER                      nfget_rank0_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable, aux(1)
#define  NFGET_CORE                         nfget_rank1_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank1_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"                                                      
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank2_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:,:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank3_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:,:,:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank4_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:,:,:,:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank5_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:,:,:,:,:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank6_real
#define  NFGET_WRAPPER_VARDECLAR real :: variable(:,:,:,:,:,:)
#define  NFGET_CORE                         nfget_real
#include "nfget_wrapper.template"
!-----------------------------------------------------------             
!##############################################################

#define   NFGET_TEMPLATE     nfget_double
#define   TRACENAME         "nfget_double"
#define   NFGET_VARDECLAR    double precision :: variable(*)
#define   NFGET_ATT          nf_get_att_double
#define   NFGET_VAR1         nf_get_var1_double
#define   NFGET_VARA         nf_get_vara_double
#define   NFGET_VAR          nf_get_var_double
#include "nfget.template"
! ======= Dummy resolvents for :: variable = double precision, rank==0 special  =========
#define  RANK_0                             
#define  NFGET_WRAPPER                      nfget_rank0_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable, aux(1)
#define  NFGET_CORE                         nfget_rank1_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank1_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank2_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:,:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank3_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:,:,:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------                           
#define  NFGET_WRAPPER                      nfget_rank4_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank5_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:,:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------
#define  NFGET_WRAPPER                      nfget_rank6_double
#define  NFGET_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:,:,:)
#define  NFGET_CORE                         nfget_double
#include "nfget_wrapper.template"
!-----------------------------------------------------------                      
!##############################################################
#define   NFGET_TEMPLATE     nfget_rank0_text
#define   TRACENAME         "nfget_rank0_text"
#define   NFGET_VARDECLAR    character*(*) :: variable
#define   NFGET_ATT          nf_get_att_text
#define   NFGET_VAR1         nf_get_var1_text
#define   NFGET_VARA         nf_get_vara_text
#define   NFGET_VAR          nf_get_var_text
#include "nfget.template"
! ========== Dummy resolvents for :: variable = char  ================

function nfget_rank1_text(ncid, name, variable, indexnf, startnf, countnf) result(return_code)
! -------------------------------------------------------------
! rank1 wrapper for character
! -------------------------------------------------------------
implicit none 

integer,      intent(in)       :: ncid          ! netCDF fileset ID
character*(*),intent(in)       :: name         
character*(*)                  :: variable(:)  !  <<<
integer,optional,intent(in)    :: indexnf(2)   ! nf_get_var1_ subarray designation
integer,optional,intent(in)    :: startnf(2)   ! nf_get_vara_ subarray designation
integer,optional,intent(in)    :: countnf(2)   ! nf_get_vara_ subarray designation
!
integer                             :: return_code
!--------------------------------------
integer :: i
integer, pointer :: iaux(:)
!--------------------------------------

return_code = nfgetvardim(ncid, name, iaux) 
if (return_code /= nfif_OK) return
         
! --- check for inconsistent optional ---

if (present(startnf) .neqv. present(countnf)) &       
    stop "nfget_rank1_text: inconsistent optionals"
if (present(indexnf) .and. (present(startnf) .or. present(countnf))) &
     stop "nfget_rank1_text: inconsistent optionals"


if (present(indexnf)) then

   return_code = nfget_rank0_text(ncid, name, variable(indexnf(2)), indexnf=indexnf)

elseif (present(startnf) .and. present(countnf)) then
 
  do i = startnf(2), startnf(2)+countnf(2)-1
    return_code = nfget_rank0_text(ncid, name, variable(i), &
                       startnf=(/startnf(1),i/),  countnf=(/countnf(1),1/))

  enddo
 
else
 
  do i=1, iaux(2)
    return_code = nfget_rank0_text(ncid, name, variable(i), &
                       startnf=(/1,i/),  countnf=(/iaux(1),1/))
  enddo

endif

if (associated(iaux)) deallocate(iaux)

end function nfget_rank1_text
!############################################################################


!=============================================================

!##############################################################
!#                Expand nfput template                       #
!##############################################################

#define   NFPUT_TEMPLATE     nfput_int
#define   TRACENAME         "nfput_int"
#define   NFPUT_VARDECLAR    integer :: variable(*)
#define   XTYPE              NF_INT
#define   NFPUT_ATT          nf_put_att_int
#define   NFPUT_VAR1         nf_put_var1_int
#define   NFPUT_VARA         nf_put_vara_int
#define   NFPUT_VAR          nf_put_var_int
#include "nfput.template"
! ======= Dummy resolvents for :: variable = integer, rank==0 special  =========
#define  RANK_0
#define  NFPUT_WRAPPER                      nfput_rank0_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable, aux(1)
#define  NFPUT_CORE                         nfput_rank1_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank1_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank2_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:,:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"                                             
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank3_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:,:,:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank4_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:,:,:,:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank5_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:,:,:,:,:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank6_int
#define  NFPUT_WRAPPER_VARDECLAR integer :: variable(:,:,:,:,:,:)
#define  NFPUT_CORE                         nfput_int
#include "nfput_wrapper.template"
!-----------------------------------------------------------


!##############################################################
#define   NFPUT_TEMPLATE     nfput_text
#define   TRACENAME         "nfput_text"
#define   NFPUT_VARDECLAR    character*(*) :: variable
#define   XTYPE              NF_CHAR
#define   TEXTATTRIBUTE                         ! special syntax for nf_put_att_text
#define   NFPUT_ATT          nf_put_att_text
#define   NFPUT_VAR1         nf_put_var1_text
#define   NFPUT_VARA         nf_put_vara_text
#define   NFPUT_VAR          nf_put_var_text
#include "nfput.template"
#undef TEXTATTRIBUTE
! ========== Dummy resolvents for :: variable = char  ================

function nfput_rank0_text(ncid, name, variable, indexnf, startnf, countnf) result(return_code)
! -------------------------------------------------------------
! rank1 wrapper for character
! -------------------------------------------------------------
implicit none

integer,      intent(in)       :: ncid          ! netCDF fileset ID
character*(*),intent(in)       :: name
character*(*)                  :: variable     !  <<<
integer,optional,intent(in)    :: indexnf(2)   ! nf_put_var1_ subarray designation
integer,optional,intent(in)    :: startnf(2)   ! nf_put_vara_ subarray designation
integer,optional,intent(in)    :: countnf(2)   ! nf_put_vara_ subarray designation
!
integer                             :: return_code
!--------------------------------------
character(len = len(variable)) :: aux
integer :: i, finger
!--------------------------------------

aux(1 : len(variable)) = variable

return_code = nfput_text(ncid, name, aux, len(variable),  &
                         indexnf=indexnf, startnf=startnf, countnf=countnf)

end function nfput_rank0_text                               

! =========================================================================

function nfput_rank1_text(ncid, name, variable, indexnf, startnf, countnf) result(return_code)
! -------------------------------------------------------------
! rank1 wrapper for character
! -------------------------------------------------------------
implicit none

integer,      intent(in)       :: ncid         ! netCDF fileset ID
character*(*),intent(in)       :: name         ! 
character(6)                   :: variable(:)  !  <<<  Temporary fix (Lars)
integer,optional,intent(in)    :: indexnf(2)   ! nf_put_var1_ subarray designation
integer,optional,intent(in)    :: startnf(2)   ! nf_put_vara_ subarray designation
integer,optional,intent(in)    :: countnf(2)   ! nf_put_vara_ subarray designation
!
integer                             :: return_code
!--------------------------------------
character(len = size(variable)*len(variable)) :: aux
integer :: i, finger
!--------------------------------------
do i=1, size(variable)                ! load variable-> aux
 finger = 1 + (i-1)*len(variable) ! load variable -> aux
 aux(finger : finger+len(variable)-1) = variable(i)
enddo

return_code = nfput_text(ncid, name, aux, len(variable), &
                         indexnf=indexnf, startnf=startnf, countnf=countnf)


end function nfput_rank1_text
                                                                                      

!##############################################################

#define   NFPUT_TEMPLATE     nfput_real
#define   TRACENAME         "nfput_real"
#define   NFPUT_VARDECLAR    real :: variable(*)
#define   XTYPE              NF_FLOAT
#define   NFPUT_ATT          nf_put_att_real
#define   NFPUT_VAR1         nf_put_var1_real
#define   NFPUT_VARA         nf_put_vara_real
#define   NFPUT_VAR          nf_put_var_real
#include "nfput.template"
! ======= Dummy resolvents for :: variable = real, rank==0 special  =========
#define  RANK_0
#define  NFPUT_WRAPPER                      nfput_rank0_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable, aux(1)
#define  NFPUT_CORE                         nfput_rank1_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank1_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank2_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:,:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"                                               
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank3_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:,:,:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank4_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:,:,:,:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank5_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:,:,:,:,:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank6_real
#define  NFPUT_WRAPPER_VARDECLAR real :: variable(:,:,:,:,:,:)
#define  NFPUT_CORE                         nfput_real
#include "nfput_wrapper.template"
!-----------------------------------------------------------

!##############################################################

#define   NFPUT_TEMPLATE     nfput_double
#define   TRACENAME         "nfput_double"
#define   NFPUT_VARDECLAR    double precision :: variable(*)
#define   XTYPE              NF_DOUBLE
#define   NFPUT_ATT          nf_put_att_double
#define   NFPUT_VAR1         nf_put_var1_double
#define   NFPUT_VARA         nf_put_vara_double
#define   NFPUT_VAR          nf_put_var_double
#include "nfput.template"
! ======= Dummy resolvents for :: variable = double precision, rank==0 special  =========
#define  RANK_0
#define  NFPUT_WRAPPER                      nfput_rank0_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable, aux(1)
#define  NFPUT_CORE                         nfput_rank1_double
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank1_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank2_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:,:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"                                          
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank3_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:,:,:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank4_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"
!-----------------------------------------------------------
#define  NFPUT_WRAPPER                      nfput_rank5_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:,:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"
!----------------------------------------------------------- 
#define  NFPUT_WRAPPER                      nfput_rank6_double
#define  NFPUT_WRAPPER_VARDECLAR double precision :: variable(:,:,:,:,:,:)
#define  NFPUT_CORE                         nfput_double
#include "nfput_wrapper.template"
!-----------------------------------------------------------                            

!###########################################################################

function copy_netCDF_variable(ncid_in, ncid_out,variableid,variablename) &
         result(return_code)
!=================================================================
! copy one netCDF variable from source to target netCDF file
!    ncid_in(in)     :   internal netCDF "unit #" for SourceSet
!    ncid_out(in)    :   internal netCDF "unit #" for TargetSet
!    varid(in)       :   netCDF id for varable to be copied
! The variable is defined in the TargetSet (ncid_out) if it does 
! not allready exists. If the variable is not found in the SourceSet
! no action is taken (only for variablename argument).
! Limitations: The dimensions for the variables must exists in the 
!              TargetSet.
!
! IMPORTANT NOTICE: copy_netCDF_variable should be used ONLY for
!                   small datasets with limited number of entries, 
!                   as each variable definition involves
!                   a file copy for (file corresponding to) ncid_out
!=================================================================
integer, intent(in)                           :: ncid_in,ncid_out
integer, intent(in),optional                  :: variableid
integer                                       :: return_code
character*(nf_max_name), intent(in), optional :: variablename
! --- locals ---
integer              :: varid_in, varid_out, status
!=================================================================
! --- resolve varid_in ---

return_code = nfif_OK

if     (present(variableid)) then 

   varid_in = variableid

elseif (present(variablename)) then 
 
   status = nf_inq_varid(ncid_in, variablename, varid_in) 
   if (status==nf_enotvar) then
      return_code = nfif_vardidnotexist
      return
   elseif (status/=nf_noerr) then
       call local_error_handler(status,&
         "copy_netCDF_variable:nf_inq_varid failed for variablename="//variablename,"")
   endif 

else

   call local_error_handler(nf_noerr,"","copy_netCDF_variable: not able to get varid_in")

endif

! --- copy_netcdf_vardef will set varid_out ---
! --- copy_netcdf_vardef also copies the attributes of varid_in

call copy_netcdf_vardef(ncid_in, ncid_out, varid_out, variableid=varid_in)
call transfer_netcdf_variable(ncid_in, ncid_out, varid_in, varid_out)

end function copy_netCDF_variable

 
!###########################################################################

subroutine copy_netcdf_vardef(ncid_in, ncid_out, varid_out, variableid, variablename)
!=================================================================
! Copy the definition of a netCDF variable from source to target netCDF file
! This includes the attributes of a netCDF variable in source set
! ----------------------------------
!    ncid_in(in)                   :   internal netCDF "unit #" for SourceSet
!    ncid_out(in)                  :   internal netCDF "unit #" for TargetSet
!    variableid, variablename(in)  :   netCDF id/name for varable to be copied
!                                      presence of variableid is superseding
!    varid_out(out)                :   netCDF ID of new variable in TargetSet
!
! Limitations: The dimensions for the variables must exists in the 
!              TargetSet.
!=================================================================
integer, intent(in)                           :: ncid_in,ncid_out
integer, intent(out)                          :: varid_out  
integer, intent(in),optional                  :: variableid
character*(nf_max_name), intent(in), optional :: variablename

integer              :: natts,status,datatype 
integer, allocatable :: dim_IDset_in(:)  ! varying subset of {ID_in}  for dimensions
integer, allocatable :: dim_IDset_out(:) ! varying subset of {ID_out} for dimensions        
integer, pointer     :: datadim_in(:),datadim_out(:)

character*(nf_max_name)   :: varname, dimname     

integer                   :: i,idummy,nvdims,varid,offset

! --------------------------------------------------------------------------                 
! find variableid (varid) and variablename (varname) from optional arguments
! presence of variableid precedes presence of variablename
! -------------------------------------------------------------------------- 
  if (present(variableid)) then 

    varid = variableid
    status = nf_inq_varname(ncid_in, varid, varname )   ! get varname
    if (status/=nf_noerr) call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_inq_varname (optional varid=", "")
                                                   
  elseif (present(variablename)) then 

    varname = variablename
    status = nf_inq_varid(ncid_in, varname, varid ) 
    if (status==nf_noerr) then 
 
    elseif (status==nf_enotvar) then 
       return 
    else
      call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_inq_var (varname=" // &
            trim(varname) // ")\nnetCDF error message = ", "")                                         
    endif
  else

    call local_error_handler(nf_noerr,"", &
         "copy_netCDF_vardef: copy_netCDF_vardef must have variablename or variableid" )

  endif

! --- now: the pair (varid, varname) is defined


! ------------------------------------------------------
! check,  if the variable exists in the TargetSet
! if not, define it in TargetSet
! ------------------------------------------------------

  status = nf_inq_varid(ncid_out, varname, varid_out)  ! may set varid_out

  if (status==nf_enotvar)  then                        ! variable not defined - define it

   status = nf_inq_varndims(ncid_in, varid, nvdims)
   if (status /= nf_noerr) call local_error_handler(status, &
         "copy_netCDF_vardef: nf_inq_varndims(varid=" // &
          as_string4(varid) // ")\nnetCDF error message = ", "")
 
   if (nvdims > 0) then
      allocate ( dim_IDset_in (nvdims) )                             ! (nvdims==0) => dim_IDset = []
      allocate ( dim_IDset_out(nvdims) )                             
   else
      allocate ( dim_IDset_in (1) )                                  ! DEC safety device
      allocate ( dim_IDset_out(1) )                                  ! DEC safety device
      dim_IDset_in(1)  = -9999                                       ! DEC gives Segmentation fault if not
      dim_IDset_out(1) = -9999                                       
   endif
 
   status = nf_inq_var(ncid_in, varid, varname, datatype, nvdims, dim_IDset_in, natts)
   if (status /= nf_noerr) call local_error_handler(status, &
            "copy_netCDF_vardef:  nf_inq_var (varname=" // trim(varname), "")                                    

   ! --- get list of dimensions for this variable in dim_IDset_out
   ! --- i.e. map the set dim_IDset_in -> dim_IDset_out

   do i = 1, nvdims                                                 ! loop over dimensions
   
     status = nf_inq_dimname(ncid_in, dim_IDset_in(i),dimname)      ! get dim name from dim id

     if (status /= nf_noerr) call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_inq_dimid(idimid =" // &
            as_string4(dim_IDset_in(i)), "")

     status = nf_inq_dimid(ncid_out, dimname, idummy)               ! get dimid in new set
     if (status /= nf_noerr) call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_inq_dimid (dimname=" // trim(dimname), "")  
     dim_IDset_out(i) = idummy                                      ! defined ID set for TargetSet

   enddo    !  i = 1, nvdims                    

                                                

   !  --- now dim_IDset_out is defined and the variable can be defined in the TargetSet
   !  --- also set varid_out

   status = nf_def_var(ncid_out, varname, datatype, nvdims, dim_IDset_out, varid_out)
   if (status /= nf_noerr) call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_def_var (varname=" // trim(varname), "") 
                                          

  elseif (status==nf_noerr) then 

  ! ------------------------------------------------------------
  ! nf_inq_varid(ncid_out ...) told that the variable did
  ! exist in the TargetSet; now check that the variable has the same 
  ! data layout as in SourceSet
  ! ------------------------------------------------------------

    status = nfgetvardim(ncid_in,varname,datadim_in)
    if (.not.((status==nfif_ok) .or. (status==nfif_foundscalarvariable)))      &
       call local_error_handler(nf_noerr, &
         "copy_netCDF_vardef:  nfgetvardim ncid_in (varname=" // &
           trim(varname), "") 

    status =  nfgetvardim(ncid_out,varname,datadim_out)
    if (.not.((status==nfif_ok) .or. (status==nfif_foundscalarvariable)))      &
       call local_error_handler(nf_noerr, &
         "copy_netCDF_vardef:  nfgetvardim ncid_out (varname=" // &
           trim(varname), "") 
  
! --- all mismatches are considered fatal ---

    if (size(datadim_in)/=size(datadim_out)) &
       call local_error_handler(nf_noerr,    &
             "copy_netCDF_vardef: size(datadim_in)/=size(datadim_out)", "")

    ! check each dimension
    offset = 0
    if (has_unlimited_dimension(ncid_in,varid)) offset = 1
 
    do i = 1,size(datadim_in)-offset
       if (datadim_in(i)/=datadim_out(i)) &
          call local_error_handler(nf_noerr, &
               "copy_netcdf_vardef: datadim_in /= &
               &datadim_out, datadim="//as_string4(i), "")     
    enddo 

 

  else   ! status of nf_inq_varid(ncid_out ...)

  ! ------------------------------------------------------------
  ! all other return states of nf_inq_varid(ncid_out ...)
  ! are exceptions 
  ! ------------------------------------------------------------

      call local_error_handler(status, &
           "copy_netCDF_vardef:  nf_inq_var (varname=" // &
            trim(varname), "")            
     
  endif  ! status of nf_inq_varid(ncid_out ...)

  ! --- now: varid_out is defined; copy variable attributes

  call copy_netcdf_varattributes(ncid_in, ncid_out, varid, varid_out)


  ! --- general clean up ---

  if (allocated(dim_IDset_in))  deallocate(dim_IDset_in)
  if (allocated(dim_IDset_out)) deallocate(dim_IDset_out)
  if (associated(datadim_in))   deallocate(datadim_in)   ! was allocated in nfgetvardim
  if (associated(datadim_out))  deallocate(datadim_out)  ! was allocated in nfgetvardim

end subroutine copy_netcdf_vardef

!###########################################################################

subroutine copy_netcdf_varattributes(ncid_in, ncid_out, varid_in, varid_out)
! ==============================================================================
! copy all attributes of variable with ID varid_in  in open netcdf set ncid_in
!                     to variable with ID varid_out in open netcdf set ncid_out
! additive, i.e. existing attributes of varid_out is not deleted
! OK if variable varid_in has no attributes
! Especially, varid_in, varid_out may be nf_global, i.e. the global dummy
! so that global attributes may be copied
! ==============================================================================
integer, intent(in)        :: ncid_in, ncid_out, varid_in, varid_out
integer                    :: status, iattid, natts
character(len=nf_max_name) :: attname
! ==============================================================================

 

 status = nf_inq_varnatts(ncid_in, varid_in, natts)                ! get transfer info

 if (status /= nf_noerr) call local_error_handler(status, &
        "copy_netCDF_varattributes:  nf_inq_varnatt (varid=" // &
         as_string4(varid_in), "")   
 
 if (natts > 0) call nfsetmode(ncid_out, nfif_define_mode)  ! set ncid_in may be either 
                                                            ! in data/define mode
 
 do iattid = 1, natts                                            ! transfer attributes
                                                                 ! get attribute name
    status = nf_inq_attname(ncid_in, varid_in, iattid, attname)
    if (status /= nf_noerr) call local_error_handler(status, &
         "copy_netCDF_varattributes:  nf_inq_attname (varid,iattid=" // &
          as_string4(varid_in) // as_string4(iattid), "") 
 
    status = nf_copy_att(ncid_in, varid_in, attname, ncid_out, varid_out)
    if (status /= nf_noerr) call local_error_handler(status, &
         "copy_netCDF_varattributes:  nf_copy_att (varid,iattid=" // &
          as_string4(varid_in) // as_string4(iattid), "") 

 enddo

end subroutine copy_netcdf_varattributes

! ###############################################################

subroutine transfer_netcdf_variable(ncid_in, ncid_out, varid_in, varid_out)
!=========================================================================
! Raw transfer of variable varid_in  in data set ncid_in
!              to variable varid_out in data set ncid_out
! variable varid_out must have been defined before invoking 
! transfer_netcdf_variable
! only data is transferred, not variable attributes
! uses direct call to specific netCDF nf_get_var_/nf_put_var_
!=========================================================================
integer, intent(in) :: ncid_in, ncid_out, varid_in, varid_out
integer             :: status,datasize, datatype
integer,pointer     :: dim_vector(:),unit_vector(:) ! allocated and initialized in nf_get_var_info

! --- AUXVAR in transfer_netCDF_var.template is expanded 
!     to these variables

integer*1,        allocatable :: aux_int1(:)   
integer*2,        allocatable :: aux_int2(:)   
integer,          allocatable :: aux_int(:)
real,             allocatable :: aux_real(:)
double precision, allocatable :: aux_double(:)
character,        allocatable :: aux_char(:)
!=========================================================================
 
 call nfsetmode(ncid_out, nfif_data_mode)    ! must be in write (data) mode 

 call get_nfvar_info(ncid_in, varid_in, datasize, datatype,dim_vector,unit_vector)


 select case (datatype)                                          ! transfer variables

      case ( nf_int1 )   ! ----------------------------------------------

#define  AUXVAR aux_int1
#define  NFGET  nf_get_var_int1
#define  NFPUT  nf_put_vara_int1
#include "transfer_netCDF_var.template"

      case ( nf_int2 )   ! ----------------------------------------------

#define  AUXVAR aux_int2
#define  NFGET  nf_get_var_int2
#define  NFPUT  nf_put_vara_int2
#include "transfer_netCDF_var.template"

      case ( nf_int  )   ! ----------------------------------------------

#define  AUXVAR aux_int
#define  NFGET  nf_get_var_int
#define  NFPUT  nf_put_vara_int
#include "transfer_netCDF_var.template"

      case ( nf_real  )   ! ----------------------------------------------

#define  AUXVAR aux_real
#define  NFGET  nf_get_var_real
#define  NFPUT  nf_put_vara_real
#include "transfer_netCDF_var.template"

      case ( nf_double  ) ! ----------------------------------------------

#define  AUXVAR aux_double
#define  NFGET  nf_get_var_double
#define  NFPUT  nf_put_vara_double
#include "transfer_netCDF_var.template"

      case ( nf_char  )   ! ----------------------------------------------

#define  AUXVAR aux_char
#define  NFGET  nf_get_var_text
#define  NFPUT  nf_put_vara_text
#include "transfer_netCDF_var.template"

      case default        ! ----------------------------------------------

         call local_error_handler(nf_noerr, &
           "transfer_netcdf_variable:  do not know how to handle datatype" // &
           "\n(internal netCDF error for nf_TYPE)", "")

      end select

end subroutine transfer_netcdf_variable
! ###############################################################


function has_unlimited_dimension(ncid,varid)
!=================================================================
! returns true if the variable with id 'varid' has an unlimited 
! dimension
!=================================================================
implicit none

integer, intent(in)  :: ncid,varid 
logical              :: has_unlimited_dimension

! locals
integer, allocatable :: dimids(:)  
integer              :: recid,nvdims,status

!=================================================================

 status = nf_inq_varndims(ncid, varid, nvdims)
 if (nvdims>0) then 
   allocate(dimids(nvdims))
   status = nf_inq_vardimid(ncid, varid, dimids) 
   status = nf_inq_unlimdim(ncid,recid) 
   if (status/=nf_noerr) then 
     has_unlimited_dimension = .false.
   else
     if (dimids(nvdims) == recid) then 
       has_unlimited_dimension = .true.
     else  
       has_unlimited_dimension = .false.
     endif 
   endif
 else
   has_unlimited_dimension = .false.
 endif
     
 return 

end function has_unlimited_dimension



subroutine get_nfvar_info(ncid, varid, datasize, datatype, dim_vector, unit_vector)
!=================================================================
! wrapper for some  nf_inq_  calls
! Inquire for netCDF set ID = ncid, for variable with ID varid the following :
!   datasize    (= product of dimensions, string length if type==character, 1 if scalar ) 
!   datatype    (return as a netCDF integer symbol nf_<TYPE>)
!   dim_vector  (len1,len2,...len_rak)  = data layout     (allocated here, if argument is present)
!                                         scalars: return dim_vector(1:1)  = 1
!   unit_vector (   1,   1,...       1) = "startnf-dummy" (allocated here, if argument is present)
!                                         scalars: return unit_vector(1:1) = 1                          
!   convenience function for (multi dimensional) slice writing
!
!   NB: user MUST deallocate  dim_vector/unit_vector after use !
!       any existing data in dim_vector/unit_vector prior to invoking 
!       get_nfvar_info() is lost
!=================================================================
integer, intent(in)           :: ncid,varid
integer, intent(out)          :: datasize, datatype
integer, pointer,optional     :: dim_vector(:), unit_vector(:)
integer              :: status, nvdims, idim, len
integer, allocatable :: dimids(:)                    
! ------------------------

! --- stipulate datasize ---

status = nf_inq_varndims(ncid, varid, nvdims) 
if (status /= nf_noerr) call local_error_handler(status, &
        "get_nfvar_info: nf_inq_varndims(varid=" // & 
         as_string4(varid), "")

if (nvdims > 0) then                        ! do allocations
   allocate ( dimids (nvdims) )             ! local         
   if (present(dim_vector))  allocate ( dim_vector(nvdims)  ) ! optionally exported
   if (present(unit_vector)) then
       allocate ( unit_vector(nvdims) )                       ! optionally exported
       unit_vector(:) = 1
   endif
                    
   status = nf_inq_vardimid(ncid, varid, dimids) 
   if (status /= nf_noerr) call local_error_handler(status, &
           "get_nfvar_info: nf_inq_vardimid(varid=" // & 
            as_string4(varid), "")

else          ! --- scalar variable: consider x = x(1:1) ---
              ! --- no alloca of dimids(:)

   if (present(dim_vector)) then
      allocate ( dim_vector(1)  )              ! optionally exported 
      dim_vector(1)  = 1      
   endif   
   if (present(unit_vector)) then
      allocate ( unit_vector(1) )              ! optionally exported                              
      unit_vector(1) = 1
   endif

endif

datasize = 1            
do idim  = 1, nvdims   ! now accumulate datasize      ! no looping for nvdims==0

  status = nf_inq_dimlen(ncid, dimids(idim), len)
  if (status /= nf_noerr) call local_error_handler(status, &
           "get_nfvar_info: nf_inq_dimlen(idim=" // & 
            as_string4(idim), "")

  datasize  = datasize * len
  if (present(dim_vector)) dim_vector(idim) = len

enddo

! --- inquire datatype ---

status = nf_inq_vartype(ncid, varid, datatype)
if (status /= nf_noerr) call local_error_handler(status, &
           "get_nfvar_info: nf_inq_vartype(varid=" // & 
            as_string4(varid), "") 

! --- clean up ---

if (allocated(dimids)) deallocate(dimids)

end subroutine get_nfvar_info

!==============================================================
! specific interface methods for DACAPO <-> netCDF user space
!==============================================================

!-----------------------------------------------------------
#define  REORDER_ATOMVECTOR_TEMPLATE Reorder_atomvector_integer
#define  VARTYPE                     integer 
#include "Reorder_atomvector.template"
!-----------------------------------------------------------
#define  REORDER_ATOMVECTOR_TEMPLATE Reorder_atomvector_real8
#define  VARTYPE                     real*8
#include "Reorder_atomvector.template"
!-----------------------------------------------------------


end module netcdfinterface


