
subroutine MakeRestrictedSetCopy(nconso, SourceSet, TargetSet)
!##########################################################################
! Make a restricted copy of one netCDF set 
! <SourceSet> to another <TargetSet>
! netCDF objects, which depends on a dimension name
! in RestrictedDimensionsList(:) will not be copied
! Jan 18 2001: blocked define/data operations, for efficiency (Asbjorn)
! -------------------------------------------------------------------------
! Tech : - netCDF ID's are (apparently) organized as ID=1..MAXID
!          where MAXID can be obtained by a nf_inq<>  call
!        - RestrictedDimensionsList(:) is later going to be initialized 
!          at runtime - currently hardcoded
! ------------------------------------------
! tmp :  xlf90 MakeRestrictedSetCopy.F -d -L. -lnetcdf_ibm_power2_thin
!          f90 -free MakeRestrictedSetCopy.F -lnetcdf
!          echo simpletest.nc | a.out ;  ../ncdump newset.nc 
!##########################################################################
use netcdfinterface,only : as_string4, &
                           copy_netcdf_varattributes, &
                           transfer_netcdf_variable,  & 
                           copy_netcdf_variable
implicit none

#include "netcdf.inc"     ! not visible via netcdfinterface: only...
integer, intent(in)       :: nconso    
character*(*), intent(in) :: SourceSet
character*(*), intent(in) :: TargetSet


! ---- locals ----

integer       :: ncid_in         ! internal netCDF "unit #" for SourceSet
integer       :: ncid_out        ! internal netCDF "unit #" for TargetSet
integer       :: ndims           ! number of dimensions in SourceSet
integer       :: nvars           ! number of variables in SourceSet
integer       :: ngatts          ! number of global attributes in SourceSet
integer       :: unlim_dimid     ! dimID for unlimited dimension - if none, then -1

integer, pointer :: restricted_dimid(:)    ! list holding IDs on restricted dimensions
integer, pointer :: variables_to_copy(:)   ! list variable IDs(_in) without  restricted dimension
integer, pointer :: dim_ID_map(:)          ! map for dimensions : (ID_in -> ID_out) 
integer, pointer :: var_ID_map(:)          ! map for variables  : (ID_in -> ID_out)
integer, pointer :: dim_IDset_in(:)        ! varying subset of {ID_in}  for dimensions 
integer, pointer :: dim_IDset_out(:)       ! varying subset of {ID_out} for dimensions

integer          :: status, dimid, dimlen, natts, datasize, datatype
integer          :: iattid, idimid, idummy, n, ivarid, nvdims 
 
character*(nf_max_name)  :: varname, attname, dimname

integer,parameter        :: NumberOfRestrictedDimensions = 13
character*(nf_max_name)  :: RestrictedDimensionsList(NumberOfRestrictedDimensions) 

! =========================================================================

data RestrictedDimensionsList(1:NumberOfRestrictedDimensions) / &
            "number_of_bands    ", & 
            "number_of_spin     ", &
            "number_of_symm_gen ", &
            "number_BZ_kpoints  ", &
            "number_IBZ_kpoints ", &
            "number_plane_waves ", &
            "max_projectors_per_atom", &
            "softgrid_dim1      ", &
            "softgrid_dim2      ", &
            "softgrid_dim3      ", &
            "hardgrid_dim1      ", &
            "hardgrid_dim2      ", &
            "hardgrid_dim3      "/

! -------------------------------------------------------------

status = nf_open(SourceSet, NF_NOWRITE, ncid_in)      

if (status /= nf_noerr) call abort_calc(nconso, &
            "MakeRestrictedSetCopy:  nf_open : error opening SourceSet = " &
             // trim(SourceSet) // "\nnetCDF error message = "// nf_strerror(status) )

status = nf_create(TargetSet, NF_WRITE, ncid_out )             ! TargetSet now in def mode
if (status /= nf_noerr) call abort_calc(nconso, &
           "MakeRestrictedSetCopy:  nf_create : error creating TargetSet = " &
            // trim(TargetSet) // "\nnetCDF error message = " // nf_strerror(status))


! ---- get an overview of SourceSet ----

status = nf_inq(ncid_in, ndims, nvars, ngatts, unlim_dimid)
if (status /= nf_noerr) call abort_calc(nconso, &
           "MakeRestrictedSetCopy:  nf_inq : error performing general inquiry " &
           // "\nnetCDF error message = " // nf_strerror(status))


! -------------------------------------------------------------
! ---------------  copy all global attributes -----------------
! -------------------------------------------------------------

call copy_netcdf_varattributes(ncid_in, ncid_out, nf_global, nf_global)


! -------------------------------------------------------------
! ----  codefine all non restricted dimensions in SourceSet ---
! ----  into TargetSet                                      ---
! -------------------------------------------------------------

if (ndims > 0) allocate ( dim_ID_map (ndims) )
dim_ID_map = -1                                                        ! security initialization
nullify(restricted_dimid)
nullify(variables_to_copy)                                             ! security initialization

do idimid = 1, ndims                                                   ! loop over dimensions in SourceSet

   status = nf_inq_dim(ncid_in, idimid, dimname, dimlen)               ! get (dimname, dimlen)
   if (status /= nf_noerr) call abort_calc(nconso, &
           "MakeRestrictedSetCopy:  nf_inq_dim(idimid ="      // &
            as_string4(idimid) // ")\nnetCDF error message = " // &
            nf_strerror(status))

   if (idimid == unlim_dimid) dimlen = nf_unlimited                    ! signal the UNLIMITED dim

   if ( this_dimension_is_nonrestricted(dimname) ) then
      status = nf_def_dim(ncid_out, dimname, dimlen, idummy)           ! define dim in new set 
      if (status /= nf_noerr) call abort_calc(nconso, &                ! throw away the new dim ID
            "MakeRestrictedSetCopy:  nf_def_dim (dimname=" // &
             trim(dimname) // ")\nnetCDF error message = " // &
             nf_strerror(status))
      dim_ID_map(idimid) = idummy                                      ! restablish ID map: old->new
   else
      call Add_to_set(restricted_dimid, idimid)                        ! remember this dim ID
   endif
  
enddo


! -------------------------------------------------------------
! --  define all variables without non restricted dimensions --
! --  this includes transferring the variable attributes     --
! --  (block all operations requiring define mode)
! -------------------------------------------------------------

if (nvars > 0)  then
  allocate ( var_ID_map (nvars) )
  var_ID_map = -1                                                   ! security initialization
endif   

! --- local allocation/deallocation of pointers ---
! --- dim_IDset_in/dim_IDset_out is this loop   ---

do ivarid = 1, nvars                                                ! check all variables
                                                                    ! see, if it has any dimensions
   status = nf_inq_varndims(ncid_in, ivarid, nvdims)               
   if (status /= nf_noerr) call abort_calc(nconso, &                
         "MakeRestrictedSetCopy: nf_inq_varndims(ivarid=" // & 
          as_string4(ivarid) // ")\nnetCDF error message = " // &
          nf_strerror(status))

   if (nvdims > 0) then
      allocate ( dim_IDset_in(nvdims) )                             ! (nvdims==0) => dim_IDset = []
   else
      allocate ( dim_IDset_in(1) )                                  ! DEC safety device
      dim_IDset_in(1) = -9999                                       ! DEC gives Segmentation fault if not
   endif

   status = nf_inq_var(ncid_in, ivarid, varname, datatype, nvdims, dim_IDset_in, natts) 
   if (status /= nf_noerr) call abort_calc(nconso, &               
           "MakeRestrictedSetCopy:  nf_inq_var (varname=" // &
            trim(varname) // ")\nnetCDF error message = " // &
            nf_strerror(status))


   if ( common_members(dim_IDset_in, restricted_dimid) &            ! common_members = false,
                       .eqv. .false.) then                          ! if dim_IDset_in = []
     if (nvdims > 0) then
        allocate ( dim_IDset_out(nvdims) )
        do n = 1, nvdims
          dim_IDset_out(n) = dim_ID_map( dim_IDset_in(n) )
        enddo
     else
        allocate ( dim_IDset_out(1) )                               ! DEC safety device
        dim_IDset_out(1) = -9999                                    ! DEC gives Segmentation fault if not
     endif
    
     status = nf_def_var(ncid_out, varname, datatype, nvdims, dim_IDset_out, idummy)  ! is in define mode
     if (status /= nf_noerr) call abort_calc(nconso, &                
           "MakeRestrictedSetCopy:  nf_def_var (varname=" // &
            trim(varname) // ")\nnetCDF error message = " // &
            nf_strerror(status)) 

     call copy_netcdf_varattributes(ncid_in, ncid_out, ivarid, idummy)

     var_ID_map(ivarid) = idummy                                    ! build lookup table for def. variables
     call Add_to_set(variables_to_copy, ivarid)                     ! remember to copy this variable

   endif  !   ( ! common_members

   if ( associated(dim_IDset_in)  ) deallocate (dim_IDset_in)       ! deallocate pointers
   if ( associated(dim_IDset_out) ) deallocate (dim_IDset_out)

enddo     !   ivarid = 1, nvars 


! -----------------------------------------------------------
! ----  copy all variables without restricted dimensions ----
! -----------------------------------------------------------

do ivarid = 1, nvars                                         ! check all SourceSet variables
  if ( member_of_set(variables_to_copy, ivarid) ) then
!   call transfer_netcdf_variable(ncid_in, ncid_out, ivarid, var_ID_map(ivarid))
    status = copy_netCDF_variable(ncid_in,ncid_out,variableid=ivarid)  
    if (status /= nf_noerr) call abort_calc(nconso, &
           "MakeRestrictedSetCopy:  copy_netCDF_variable " // & 
           "netCDF error message = "  // &
            nf_strerror(status))          
  endif 
enddo

! --- close sets ---

status = nf_close(ncid_in)
if (status /= nf_noerr) call abort_calc(nconso, &                
           "MakeRestrictedSetCopy:  nf_close ( in set = "   // &
            trim(SourceSet) // ")\nnetCDF error message = " // &
            nf_strerror(status))

status = nf_close(ncid_out)
if (status /= nf_noerr) call abort_calc(nconso, &                
           "MakeRestrictedSetCopy:  nf_close ( out set = "   // &
            trim(TargetSet) // ")\nnetCDF error message = "  // &
            nf_strerror(status))


! --- free local pointers ---

if ( associated (restricted_dimid )   ) deallocate (restricted_dimid) 
if ( associated (variables_to_copy)   ) deallocate (variables_to_copy)
if ( associated (dim_ID_map )         ) deallocate (dim_ID_map )
if ( associated (var_ID_map )         ) deallocate (var_ID_map )


return

! ################################################################
                     contains  
! ################################################################


function this_dimension_is_nonrestricted ( dimname )
!=================================================================
! Return true, if dimname does not appear on RestrictedDimensionsList,
! which is accessible from caller in this private function 
!=================================================================
implicit none

logical                  :: this_dimension_is_nonrestricted
character*(*),intent(in) :: dimname 

integer                  :: i
!------------------------
this_dimension_is_nonrestricted = .true.

do i = 1, NumberOfRestrictedDimensions
  if (index(RestrictedDimensionsList(i), dimname) > 0) &
      this_dimension_is_nonrestricted = .false.
enddo

end function this_dimension_is_nonrestricted

!###################################################################

subroutine Add_to_set(iset,i)
!=============================================
! Add i to the set of integers iset (primitive)
!=============================================
implicit none
integer,intent(in) :: i
integer, pointer   :: iset(:), aux(:)
integer            :: n

if ( .not.associated(iset) ) then      ! make it DEC safe
  allocate ( iset(1) )
  iset(1) = i
  return                               
endif

n = size(iset)
allocate ( aux (n+1) )
aux (n+1) = i
if (n > 0) then                        ! transfer old data
  aux(1:n) = iset(1:n)
  deallocate(iset)
endif
iset => aux

end subroutine Add_to_set

!###################################################################

function member_of_set(iset,i)
!=============================================
! Check whether i is a member of iset  (primitive)
!=============================================
implicit none
logical            :: member_of_set
integer,intent(in) :: i
integer, pointer   :: iset(:)
integer            :: j


member_of_set = .false.

if ( .not.associated(iset) ) return    ! make it DEC safe 

do j=1,size(iset)
  if ( iset(j) == i ) then
     member_of_set = .true.
     return
  endif
enddo

end function member_of_set

!###################################################################

function common_members(iset1, iset2)
!=============================================
! Check whether there is an overlap 
! between iset1, iset2; if so, return .true.
! (primitive)
!=============================================
implicit none
logical            :: common_members
integer, pointer   :: iset1(:), iset2(:)
integer            :: i,j

common_members = .false.

if ( .not.associated(iset1) ) return  
if ( .not.associated(iset2) ) return 

do i=1,size(iset1)
  do j=1,size(iset2)
     if ( iset1(i) == iset2(j) ) then
        common_members = .true.
        return
     endif
  enddo
enddo

end function common_members

!###################################################################

end subroutine MakeRestrictedSetCopy

