#include "definitions.h"
!=======================================================================
      subroutine ionmid(nconso)
      write(nconso,*) '@(#)ionmov.F	1.28 6/10/99'
      return
      end
!=====================================================================
      subroutine ionmov(&
             posion,fortot,rvelo,entot,sizeforce,&
             fallmax,lionmv,&
#            include "ions_args.h"
             idebug,nconso)
!=====================================================================
!     input   posion   : atomic positions 
!             fortot   : force on atoms (eV/A)
!             entot    : total energy 
!             deltim   : timestep in fs
!             iion     : ion dynamics 
!                          1: verlet 
!                          2: qumin 
!                          3: quminsum (preserve symmetry) 
!                          4: bfgs 
!                          5: conjugate gradient
!                         50: external ion motion by script
!             inudged       : 1 (nudged elactic band calculation)
!             icoordsystem  : 0: (xyz) 1: (spherical for two atoms)
!             sphericalcnstr: two atoms for spherical coordinate system
!             rvelo         : velocity on atoms
!             rmass         : mass of atoms (A/fs)
!             rmove         : factor in front of forces 
!                             (from static/dynamic in xprogram)
!             sizeforce     : residual force returned to libtos (parallel to constraints)
!             fallmax       : only move ions, if the (sizeforce > fallmax) for iion = 2,3,4,5
!             lionmv        : tells whether ions were moved or not by ionmov
!             alpha = [ (1.60219d-19 * 1d-30)/(1d-20 * 1.6726d-27) ]
!=======================================================================          
      use basicdata, only : tpi 
      implicit none
     
#     include "ions_decl.h"
      real*8    posion(3,nions,nspec),fortot(3,nions,nspec)
      real*8    entot,rvelo(3,nions),sizeforce,fallmax
      logical*4 lionmv
      integer   idebug,nconso
!
!     locals
!
      integer  natdyn,nat,nsp,ni,m,i,n,na,nq 
      real*8   ekin,posdel
      real*8   cforce(3,nions),cvelo(3,nions),posq(3,nions)
      real*8   dposr(3,nions),dposa(3,nions)
      real*8   forwrk(3,nions) 
      real*8   sizpos
      real*8   alpha 
      parameter(alpha = 0.00957904)    
      logical , save :: init = .true. 

      if (init) then 
!       write header for STEP 
        write(nconso,*) &
         'STEP,FORCE  step size(A)  ResidualForce(eV/A) Energy(eV)'
        init = .false. 
      endif
     
!
!     get constrained forces (cforce), velocities (cvelo) 
!     posq is the transformed coordinates if icoordsystem = 1, 
!     otherwise posq is just a copy of posion. 
!     
      call constrainforce(&
             fortot,cforce,&
             rvelo ,cvelo,&
             posion,posq,&
#            include "ions_args.h"
             nconso,idebug)


      if (idebug.gt.0) then 
         nat = 1
         do nsp=1,nspec
           do ni=1,nionsp(nsp)
             write(nconso,*) 'ionmov cforces ',&
                 nat,(cforce(m,nat),m=1,3),&
                 deltim,rmass(nsp)
             nat=nat+1
           enddo
         enddo
      endif

! ... decide on whether to move ions at all (qumin*/BFGS/conjug-grad), 
! ... based on the magnitude of the constrained force (cforce)

       sizeforce = sqrt(sum(cforce*cforce))
       
       if ((sizeforce < fallmax).and.&
           (iion>=2).and.(iion<=5)) then 
           write(nconso,4605) 0.0d0, sizeforce, entot
           lionmv    = .false.    ! abandon ion motion and return
           return
       else
           lionmv    = .true.     ! continue and move the ions
       endif


!
! Update the velocities in x,y,z coordinates
! using one of four different algorithm
! dposr is returned from all algorithms (real space change in 
! positions, if icoordsystem.eq.1, dposr is in gen. coodinates.i) 
!
        if (iion.eq.1) then 
          call verlet(nconso,alpha,deltim,cvelo,cforce,nions,&
                nspec,nionsp,rmass,rdamp,dposr)
        else if ((iion.eq.2).and.(icoordsystem.eq.0)) then
          call qumin (nconso,alpha,deltim,cvelo,cforce,nions,&
                nspec,nionsp,rmass,dposr)
        else if ((iion.eq.2).and.(icoordsystem.eq.1)) then
          call qumin_gen(cforce,posq,deltim,cvelo,nions,&
                 sphericalcnstr(1),sphericalcnstr(2),dposr)
        else if (iion.eq.3) then
          call quminsum(nconso,alpha,deltim,cvelo,cforce,nions,&
                nspec,nionsp,rmass,dposr)

        else if (iion.eq.4) then 
          n = 3*nions
!         transform posq to real space
          if (icoordsystem.eq.0) then 
            call posa_to_r(posq,dirc,nions)
          endif
          call Broyfletgoldshan(nconso,n,posq,entot,&
                cforce,deltim,alpha,dposr)   

        else if (iion.eq.5) then 
!         transform posq to real space
          if (icoordsystem.eq.0) then 
            call posa_to_r(posq,dirc,nions)
          endif
          call ion_conjgrad (nconso,3*nions,posq,entot,cforce, &
                             alpha*deltim**2,alpha*deltim**2,dposr) 
          
        else if (iion.eq.50) then 
          if (icoordsystem .ne. 0) &
             call abort_calc(nconso,"ionmov: generalized coordinates"//&
                           " not allowed for ExternalIonMotion_script")
          call externalmotion_script(nions,posq,dirc,dposr,cvelo) 
        else
           write(nconso,*) 'There is no such algorithm: iion=',iion
           call clexit(nconso)
        endif

!     get positions in real space back
      if (icoordsystem.eq.1) then 
        do nat = 1,nions
          do m = 1,3 
            dposr(m,nat) = posq(m,nat) + dposr(m,nat)   
          enddo 
        enddo 
        call posq_to_r(dposr,dposr,sphericalcnstr(1),&
                       sphericalcnstr(2),nions)
        call posq_to_r(posq,posq,sphericalcnstr(1),&
                       sphericalcnstr(2),nions)
        do nat = 1,nions
            do m = 1,3
               dposr(m,nat) = dposr(m,nat) - posq(m,nat) 
            enddo 
        enddo
      endif
      do nat = 1,nions
        do m = 1,3 
            rvelo(m,nat) = cvelo(m,nat)
        enddo 
      enddo 

!     symmetrizise the velocities and the changes in the positions,
!     so that no problems will occur if any operations in the above
!     routines has spoiled the symmetries already present in the forces
      call symvel(nconso,nmsfor,dirnms,rvelo,forwrk,nions,&
                  numsym,nmstyp)
      call symvel(nconso,nmsfor,dirnms,dposr,forwrk,nions,&
                  numsym,nmstyp)                        

!     get change in a1,a2,a3 coordinates, dposa
      call posr_to_a(dposr,dposa,recc,tpi,nions)

!     update the positions
      nat = 1
      do  nsp=1,nspec
        do  ni=1,nionsp(nsp)
          do  m=1,3
             posion(m,ni,nsp) = posion(m,ni,nsp) + dposa(m,nat)
             if (idebug.gt.0) then 
               write(nconso,*) 'ionmov: update ',posion(m,ni,nsp),&
                                dposa(m,nat) 
             endif
          enddo
          nat = nat + 1
        enddo
      enddo              

!
!     find size of the changes
!
      sizpos   =0.0d0
      do nat=1,nions
         do m=1,3
           sizpos=sizpos+dposr(m,nat)**2
         enddo
      enddo
      sizpos=sqrt(sizpos)

      write(nconso,4600) sizpos,sizeforce,entot
 4600 format(1x,'STEP,FORCE ',f12.6,6x,f12.6,1x,f14.6)
 4605 format(1x,'STEP,FORCE (ionmov: no step)',f12.6,6x,f12.6,1x,f14.6)

!     verlet
      if (iion.eq.1) then 
!         calculate kinetic energy
          call kinetic_energy(nspec,nions,nionsp,rvelo,&
                               rmass,alpha,ekin)
         write(nconso,4500) 'MDYN: ',entot,ekin,entot+ekin
      endif
 4500 format(1x,a5,1x,3(f12.6,1x))

!
! Write out the new positions
!
      nat=1
      do 4800 nsp=1,nspec
         do 4700 ni=1,nionsp(nsp)
            write(nconso,4900) 'Ion123',nat,ni,nsp,&
                 (posion(i,ni,nsp),i=1,3),(rmove(i,nat),i=1,3)
            nat=nat+1
 4700    continue
 4800 continue
 4900 format(1x,a6,2i4,i2,1x,3f12.8,4x,3f4.1)

      return
      end


! ========================================================================
      subroutine constrainforce(&
             force,cforce,&
             rvelo ,cvelo,&
             posion,posq,           &
#            include "ions_args.h"
             nconso,idebug)                 
!=========================================================================
!     return force  
!
      implicit none
#     include "ions_decl.h" 
      real*8   force(3,nions,nspec) ,cforce(3,nions)
      real*8   rvelo(3,nions) ,cvelo(3,nions)
      real*8   posion(3,nions,nspec),posq(3,nions)
      integer  nconso,idebug                           
!
! locals
      integer nsp,ni,m,i,na
      real*8 posic(3,nions),forwrk(3,nions)
!
!     copy force -> cforce, rvelo -> cvelo, and posion -> posq
      na = 1
      do nsp = 1 , nspec
        do ni = 1 , nionsp(nsp)
          do m = 1 , 3
            posq  (m,na) = posion(m,ni,nsp)
            cforce(m,na) = force(m,ni,nsp)
            cvelo (m,na) = rvelo(m,na)
          enddo
          na = na + 1
        enddo
      enddo

!
!     a possible shift in coordinate system
      if (icoordsystem.eq.1) then
!       transform the velocities and position in spherical coordinates
        call gencoord(&
           posion,posq,&
           force,cforce,&
           rvelo ,cvelo,            &
#          include "ions_args.h"
           nconso)

      endif

!     Constrain the ionic motion
      na = 1
      do nsp = 1 , nspec
        do ni = 1 , nionsp(nsp)
          do m = 1 , 3
            posic(m,na)=posion(m,ni,nsp)
            cforce(m,na) = rmove(m,na)*cforce(m,na)
          enddo
          na = na + 1
        enddo
      enddo
!
      if (inudged.eq.1) then 
!        nudged elactic band method 
         call bndcnst(cforce,nions,posic,dirc,nconso,forwrk)
      endif

      return
      end


! =======================================================================

      subroutine verlet(nconso,alpha,deltim,rvelo,force,nions,nspec,&
           nionsp,rmass,rdamp,dpos)
!
! if not_first_time 
!   p_n = p_n + 1/2 F_n h
! endif
! p_n = (1-lambda) p_n         [Formally not a part of the Verlet alg.]
! p_n+1 = p_n + 1/2 F_n h
!
      implicit  none  
      integer   nconso
      integer   nions,nspec
      real*8    force(3,nions*nspec)
      integer   nionsp(nspec)
      real*8    rvelo(3,nions)
      real*8    rmass(nspec)
      real*8    rdamp(nspec)
      real*8    deltim,alpha
      real*8    dpos(3,nions)

      integer   nat,nsp,ni,m,i
      logical*4 linit
      data      linit/.false./
      save      linit
      real*8    eps,tpi 
      parameter(eps = 1.0d-6,tpi=2.0d0*3.1415926535)
!
      nat=1
      do 2300 nsp=1,nspec
         do 2200 ni=1,nionsp(nsp)
            do 2100 m=1,3
               if (linit) then
                  rvelo(m,nat)=rvelo(m,nat)+&
                       alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
                  rvelo(m,nat)=(1.0d0-rdamp(nsp))*rvelo(m,nat)
               endif
               write(nconso,100) nat,(rvelo(i,nat),i=1,3)

               rvelo(m,nat)=rvelo(m,nat)+&
                      alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
 2100       continue
            nat=nat+1
 2200    continue
 2300 continue

!     find the change in the atomic positions
      nat = 1
      do  nsp=1,nspec
        do  ni=1,nionsp(nsp)
          do  m=1,3
             dpos(m,nat) = rvelo(m,nat)/tpi*deltim
          enddo
          nat = nat + 1
        enddo 
      enddo
      
 100  format(1x,'VEL:',i3,3(1x,f8.5)) 
      linit=.true.
      return 
      end

! =======================================================================
      subroutine qumin(nconso,alpha,deltim,rvelo,force,nions,nspec,&
           nionsp,rmass,dpos)
!
! if not_first_time 
!   p_n = p_n + 1/2 F_n h
! endif
! if <F_n|p_n> > 0
!     p_n = F_n <F_n|p_n>/|F_n|^2  ; i.e. p_n || F_n
! else
!     p_n = 0
! endif
! p_n+1 = p_n + 1/2 F_n h
!
      implicit  none
      integer   nconso
      integer   nspec,nions
      real*8    force(3,nions)
      integer   nionsp(nspec)
      real*8    rvelo(3,nions)
      real*8    rmass(nspec)
      real*8    deltim,alpha
      real*8    dpos(3,nions)
!
      integer   nat,nsp,ni,m
      logical*4 linit
      data      linit/.false./
      save      linit
      real*8    eps,tpi 
      parameter(eps = 1.0d-6,tpi=2.0d0*3.1415926535)
      real*8    force_size,dot_product
!
! first update the velocities according to the Verlet algorithm
! p_n = p_n + 1/2 F_n h
!
      if (linit) then
         nat=1
         do nsp=1,nspec
            do ni=1,nionsp(nsp)
               do m=1,3
                 rvelo(m,nat)=rvelo(m,nat)+&
                    alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
               enddo
               nat=nat+1
            enddo
         enddo
      endif

      call write_vel(nconso,nions,rvelo)

!
! now, keep only the part of the velocities that is parallel to the force
! if <F_n|p_n> > 0
!     p_n = F_n <F_n|p_n>/|F_n|^2  ; i.e. p_n || F_n
! else
!     p_n = 0
! endif
!
      do nat=1,nions
         force_size=force(1,nat)**2+force(2,nat)**2&
              +force(3,nat)**2
         if (force_size.gt.eps) then
            dot_product=force(1,nat)*rvelo(1,nat)&
                 +force(2,nat)*rvelo(2,nat)&
                 +force(3,nat)*rvelo(3,nat)
            if (dot_product.gt.eps) then
               do m=1,3
                  rvelo(m,nat)=force(m,nat)*dot_product/force_size
               enddo
            else
               do m=1,3
                  rvelo(m,nat)=0.0d0
               enddo
            endif
         endif
      enddo

!
! finally update the velocities according to the Verlet algorithm
! p_n+1 = p_n + 1/2 F_n h
!
      nat=1
      do nsp=1,nspec
         do ni=1,nionsp(nsp)
            do m=1,3
               rvelo(m,nat)=rvelo(m,nat)+&
                      alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
            enddo
            nat=nat+1
         enddo
      enddo

!     find the change in the atomic positions
      nat = 1
      do  nsp=1,nspec
        do  ni=1,nionsp(nsp)
          do  m=1,3
             dpos(m,nat) = rvelo(m,nat)*deltim
          enddo
          nat = nat + 1
        enddo
      enddo                 

      linit=.true.
      return 
      end

! ======================================================================
      subroutine quminsum(nconso,alpha,deltim,rvelo,force,nions,nspec,&
           nionsp,rmass,dpos)
!
! if not_first_time 
!   p_n = p_n + 1/2 F_n h
! endif
! if Sum_ions <F_n|p_n> > 0
!   p_n = F_n (Sum_n <F_n|p_n>) /|F_n|^2  ; i.e. p_n || F_n
! else
!   p_n = 0
! endif
! p_n+1 = p_n + 1/2 F_n h
!
      implicit  none
      integer   nspec,nions,nconso
      real*8    force(3,nions)
      integer   nionsp(nspec)
      real*8    rvelo(3,nions)
      real*8    rmass(nspec)
      real*8    deltim,alpha
      real*8    dpos(3,nions)
!
      integer   nat,nsp,ni,m
      logical*4 linit
      data      linit/.false./
      save      linit
      real*8    eps,tpi 
      parameter(eps = 1.0d-6,tpi=2.0d0*3.1415926535)
      real*8    force_size,dot_product
      real*8    sum
!
! first update the velocities according to the Verlet algorithm
! p_n = p_n + 1/2 F_n h
!
      if (linit) then
         nat=1
         do nsp=1,nspec
            do ni=1,nionsp(nsp)
               do m=1,3
                  rvelo(m,nat)=rvelo(m,nat)+&
                       alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
               enddo
               nat=nat+1
            enddo
         enddo
      endif

      call write_vel(nconso,nions,rvelo)

!
! now, keep only the part of the velocities that is parallel to the force
!
      sum=0.0d0
      force_size=0.0d0
      do nat=1,nions
         force_size=force_size+force(1,nat)**2&
              +force(2,nat)**2&
              +force(3,nat)**2
         if (force_size.gt.eps) then
            sum=sum+force(1,nat)*rvelo(1,nat)&
                 +force(2,nat)*rvelo(2,nat)&
                 +force(3,nat)*rvelo(3,nat)
         endif
      enddo

!
! zero all velocities if the sum of the dot products is smaller than zero
! if Sum_ions <F_n|p_n> > 0
!   p_n = F_n (Sum_ions <F_n|p_n>) /|F_n|^2  ; i.e. p_n || F_n
! else
!   p_n = 0
! endif
!
      if (sum.ge.eps) then
         do nat=1,nions
            do m=1,3
               rvelo(m,nat)=force(m,nat)*sum/force_size
            enddo
         enddo
      else
         do nat=1,nions
            do m=1,3
               rvelo(m,nat)=0.0d0
            enddo
         enddo
      endif

!
! finally update the velocities according to the Verlet algorithm
! p_n+1 = p_n + 1/2 F_n h
!
      nat=1
      do nsp=1,nspec
         do ni=1,nionsp(nsp)
            do m=1,3
               rvelo(m,nat)=rvelo(m,nat)+&
                      alpha*force(m,nat)/rmass(nsp)*deltim*0.5d0
            enddo
            nat=nat+1
         enddo
      enddo

!     find the change in the atomic positions
      nat = 1
      do  nsp=1,nspec
        do  ni=1,nionsp(nsp)
          do  m=1,3
             dpos(m,nat) = rvelo(m,nat)/tpi*deltim
          enddo
          nat = nat + 1
        enddo
      enddo                

      linit=.true.
      return 
      end


!=======================================================================
      subroutine bndcnst(force,nions,posic,dirc,nconso,forwrk)
!
! Hyperplane adaptive constraint method (HAC) 
! (=Nudged Elastic Band Method with force constant = 0)
!
!   substract the component of the force parallel to the constraint
!   direction from the total force.  
!   The resulting force is retuned in force. 
!
! 
      implicit none
!
      integer nions
      real*8 force(3,nions)
      real*8 posic(3,nions),dirc(3,3)
      integer nconso
      real*8 forwrk(3,nions)
! locals
      real*8 dirnorm
      integer ni,ia,ia1
      real*8 a1,a2,a3,xyz,dotpos
      real*8 grnorm
      real*8 resnorm
      logical*4 linit
      data linit /.true./
      save linit

!
! open the file #22 and read the constraint direction
! and convert it to the cartesian coordinates.
!
      dirnorm=0.0d0
      
      open(22,FILE='constraint',STATUS='OLD')
      rewind 22
      
      do ni=1,nions
         read(22,*)a1,a2,a3
         do ia=1,3
            forwrk(ia,ni)=dirc(1,ia)*a1&
                 +dirc(2,ia)*a2&
                 +dirc(3,ia)*a3
            dirnorm=dirnorm+forwrk(ia,ni)**2
         enddo
         write(nconso,31) ni,forwrk(1,ni),forwrk(2,ni),forwrk(3,ni)
 31      format('CNST:',i4,3f16.8)
      enddo

      close(unit=22)

!
! normalize the constraint direction
!
      dirnorm=1.d0/dsqrt(dirnorm)
      do ni=1,nions
         do ia=1,3
            forwrk(ia,ni)=forwrk(ia,ni)*dirnorm
         enddo
      enddo
!
! calculate the dot product between the constraint direction and 
! the force (dirnorm) and calculate the dot-product between the 
! current real space positions and the constrain direction
! (should be constant) 
! 
!
      dirnorm=0.d0
      grnorm=0.0d0
      dotpos =0.d0
      do ni=1,nions
         do ia=1,3
            dirnorm=dirnorm+forwrk(ia,ni)*force(ia,ni)
            grnorm=grnorm+force(ia,ni)*force(ia,ni)
            xyz = 0.0d0
            do ia1 = 1,3
              xyz = xyz+ dirc(ia1,ia)*posic(ia1,ni)
            enddo
            dotpos=dotpos+forwrk(ia,ni)*xyz
         enddo
      enddo
!
! substract the component of the force parallel to the constraint
! direction from the total force.
!
      resnorm=0.0d0
      do ni=1,nions
         do ia=1,3
            force(ia,ni)=force(ia,ni)-dirnorm*forwrk(ia,ni)
            resnorm=resnorm+force(ia,ni)*force(ia,ni)
         enddo
      enddo
      if (linit) then
         write(nconso,*) &
      'HAC       |F|        |F_path|  |F_constrained|    POS*CDIR'
         linit=.false.
      endif
      write(nconso,4600)sqrt(grnorm),dirnorm,sqrt(resnorm),dotpos
 4600 format(1x,'HAC:',3f12.6,1x,f15.8)

      return 
      end


!=========================================================================
      subroutine read_fast_dof(nconso,max_fast_dof,num_fast_dof,&
                               ifast_dof,fast_dof_amp)

!     initialize the array of fast degrees of freedom, ifast_dof
!     ifast_dof point to two ions, the internal vibration of
!     which will be damped by a factor 10.

!=========================================================================

      implicit none
      integer nconso,max_fast_dof
      integer num_fast_dof,ifast_dof(2,max_fast_dof)
      real*8  fast_dof_amp(max_fast_dof)

      integer i

      num_fast_dof = 0

      read(15,*,err=200,end=200) num_fast_dof
      do i=1,num_fast_dof
         read(15,*) ifast_dof(1,i),ifast_dof(2,i),&
                    fast_dof_amp(i)
      enddo
!     write to out file
!     FDOF: <ndof> <ion1> <ion2> <amplitude> 

      do i=1,num_fast_dof
       write(nconso,100)  i,ifast_dof(1,i),ifast_dof(2,i),&
                          fast_dof_amp(i)
100    format('FDOF: ',i1,1x,i2,1x,i2,f8.4) 
      enddo

200   continue

      return
      end

!=======================================================================
      subroutine write_vel(nconso, nions, rvelo )
! 
!     write velocities to out-file
      integer nions,nconso
      real*8  rvelo(3,nions)

      do nion = 1, nions
         write(nconso,100) nion,(rvelo(i,nion),i=1,3) 
      enddo
100   format(1x,'VEL:',i3,3(1x,f8.5)) 

      return 
      end

!=========================================================================
      subroutine write_realpos(nconso,&
          nspec, nions, nionsp, ntype, posion, dirc, rmove, rvelo )
!
!     write real space positions
!
!=========================================================================

      implicit none

      integer     nconso, nspec,nions,nionsp(nspec),ntype(nspec)
      real*8      posion(3,nions,nspec)
      real*8      dirc(3,3)
      real*8      rvelo(3,nions),rmove(3,nions*nspec)

      integer     nsp,ni,ng,i

      ng = 1
      do nsp = 1,nspec
        do ni = 1,nionsp(nsp) 

         write(nconso,100) 'Ionxyz',ng,ni,nsp,&
         dirc(1,1)*posion(1,ni,nsp)+&
         dirc(2,1)*posion(2,ni,nsp)+dirc(3,1)*posion(3,ni,nsp),&
         dirc(1,2)*posion(1,ni,nsp)+&
         dirc(2,2)*posion(2,ni,nsp)+dirc(3,2)*posion(3,ni,nsp),&
         dirc(1,3)*posion(1,ni,nsp)+&
         dirc(2,3)*posion(2,ni,nsp)+dirc(3,3)*posion(3,ni,nsp),&
         (rmove(i,ng),i=1,3),ntype(nsp),ni,nsp,&
         (rvelo(i,ng),i=1,3)
         ng=ng+1
        enddo
      enddo
100   format(1x,a6,2i4,i2,1x,3f12.6,4x,3f4.1,3i3,3f7.4)

      return 
      end


!=========================================================================
      subroutine kinetic_energy(nspec,nions,nionsp,&
                                rvelo,rmass,alpha,ekin)

!     Calculate kinetic energy for check of energy conservation in 
!     molecular dynamic simulation
!
!     Mass : 1.6726 10^-27 Kg
!     V    : A/fs
!=========================================================================
      implicit  none 
      integer   nspec,nions,nionsp(nspec)
      real*8    rvelo(3,nions)
      real*8    rmass(nspec)
      real*8    alpha,ekin

!     locals
      integer   nsp,ng,ni,i

      ng   = 1
      ekin = 0.0d0
      do nsp = 1,nspec 
        do ni = 1,nionsp(nsp)
          do i = 1,3
            ekin = ekin + 0.5d0*rmass(nsp)*(rvelo(i,ng)**2)
          enddo
          ng = ng + 1
        enddo
      enddo
      ekin = ekin/alpha

      return 
      end
!=========================================================================
      subroutine externalmotion_script(nions,posq,dirc,dposr,velocit) 
!=========================================================================
!     posq:    current positions in scaled coordinates               (input)
!     dirc:    current unit cell                                     (input)
!     dposr:   change in ionic coordinates, deduced by running       (output)
!              the external ion motion script                          
!              dposr is returned in cartesian coordinates, using dirc
!     velocit: cartesian atom velocities, corresponding to posq      (output)
!              are read from file and passed on. No checks is performed
!              on whether the read time slot corresponds to NF_FILL
!     imported symbols from run_context: ExternalIonMotion_script
!                                        netCDF_output_filename
!-------------------------------------------------------------------------
!     externalmotion_script reads the last entry in DynamicAtomPositions
!     in file: netCDF_output_filename. Parallel mode: only master
!     executes this subroutine. Broadcast for result (dposr) to slaves
!     is outside this subroutine
!     Primitive consistency checks: do not accept changes in the unit cell  
!=========================================================================
      use run_context
      use netcdfinterface
      implicit none

      integer, intent(in)  ::  nions
      real*8, intent(in)   ::  dirc(3,3), posq(3,nions)        
      real*8, intent(out)  ::  dposr(3,nions) 
      real*8, intent(out)  ::  velocit(3,nions) 

! --- locals ---

      integer  :: last_step, ncid, status, i
      real*8   :: position_buffer(3,nions)
      real*8   :: dirc_buffer(3,3), rlatc

         call system_opsys_interface(trim(ExternalIonMotion_script))

         status =  nf_open(netCDF_output_filename, NF_NOWRITE, ncid)  
         if (status /= nf_noerr) call abort_calc(nconso,  &
              "externalmotion_script -> nf_open : "//&
              "error opening nc-outfile")

!.....Get the unlimited dimension in the SouceSet

         status = nfgetglobaldim(ncid,'number_ionic_steps', last_step) 
         if (status /= nfif_OK) call abort_calc(nconso, &
                "externalmotion_script -> nfgetglobaldim : error "// &
                "reading unlimited dimension in SourceSet")
         if (last_step < 1) call abort_calc(nconso, &
                "externalmotion_script -> last_step < 1")

!.....Retrieve new positions (in scaled coordinates)
!     as well as cartesian velocities

         status = nfget(ncid,"DynamicAtomPositions", position_buffer, &
                  startnf=(/1,1,last_step/),countnf=(/3,nions,1/))
         if (status /= nfif_OK) call abort_calc(nconso, &
               "externalmotion_script -> error reading new positions")
         if (any(position_buffer == nf_fill_double))&
                call abort_calc(nconso, "externalmotion_script: "//&
                                        "new positions void")

         status = nfget(ncid,"DynamicAtomVelocities", velocit, &
                  startnf=(/1,1,last_step/),countnf=(/3,nions,1/))
         if (status /= nfif_OK) call abort_calc(nconso, &
               "externalmotion_script -> error reading new velocities")
       

!.....reorder positions/velocity arrays corresponding to internal atom order

      do i = 1, 3
        call Reorder_atomvector(nconso, position_buffer(i,:), &
                               "netCDF_to_internal")  
        call Reorder_atomvector(nconso, velocit(i,:),   &
                               "netCDF_to_internal")          
      enddo

!.....Primitive consistency checks:
!              1)  do not accept changes in the unit cell    


         status = nfget(ncid, "UnitCell_ScaleFactor", rlatc)     
         if (status /= nfif_OK ) then
            rlatc = 1.0d0       ! set defaults
         else
            if (rlatc == nf_fill_double) rlatc = 1.0 ! set default
         endif


         status = nfget(ncid,"UnitCell", dirc_buffer, &
                  startnf=(/1,1,last_step/),countnf=(/3,3,1/))
         if (status == nfif_OK) then
            if (.not.any(dirc_buffer == nf_fill_double)) then
               dirc_buffer = transpose( rlatc * dirc_buffer )
               if (any(abs(dirc_buffer-dirc) > 1.0d-6)) then

                  call abort_calc(nconso, "externalmotion_script: "//&
                        "external unit cell motion not supported yet")
               endif
            endif
         endif


         status = nf_close(ncid)
         if (status /= nf_noerr) call abort_calc(nconso, &
                "externalmotion_script -> nf_close : "//&
                "error closing nc-outfile")

         dposr = position_buffer - posq
         call posa_to_r(dposr,dirc,nions) 


      end subroutine externalmotion_script













