#include "definitions.h"
! This module generates the Brillouin Zone k-points. 
! The k-points is defined by     
!    - the variable BZ_kpoints in the netCDF file 
!    - or given by a stardard set of k-points given by the 
!      variable KpointSetup:gridtype (default=1).



module generate_kpoints
implicit none

! entry  :  generate_bz_kpoints
! syntax :  subroutine generate_bz_kpoints(nkpnun,kpoints,nconso)

contains

!--------------------------------------------------------------
subroutine generate_bz_kpoints(nkpnum,kpoints,nkpmem,idebug,lmastr) 
use netcdfinterface
use run_context
#ifdef PARAL
use par_functions_module
#endif PARAL
implicit none

! nkpnum  (out)  :  number of kpoints returned
! kpoints (out)  :  all nkpnum k-points in the 1st Brillouin zone
! nkpmem  (out)  :  The number of k-points in memory
! idebug (in)    :  debugging level

! If the dimension number_BZ_kpoints is present in the netCDF file 
! the k-points are read from the BZKpoints variable. 
! If number_BZ_kpoints is not present the variable KpointSetup 
! is used to either generate a 
!    Monkhorst-Pack grid   KpointSetup:gridtype='MonkhorstPack'
!                          KpointSetup(1:3) gives the 3D grid. 
!    Chadi-Cohen 1dim k-point sets 
!                          KpointSetup:gridtype='Chadi_Cohen'
! If neither the MonkhorstPack or the Chadi_Cohen and the 
! number_BZ_kpointa is not set, the Gamma-point is returned and 
! nkpnum=1.

integer, intent(out) ::  nkpnum,nkpmem
integer, intent(in)  ::  idebug
real*8,  pointer     ::  kpoints(:,:)
logical*4,intent(in) ::  lmastr
! locals

integer :: nkp,status,ncid,nOK
character(*), parameter :: MonkhorstPack = 'MonkhorstPack'
character(*), parameter :: ChadiCohen    = 'ChadiCohen'
character(100)          :: gridtype
#include "ms.h"

if (lmastr) then 

! open netCDF file 
status = nf_open(netCDF_input_filename,NF_NOWRITE, ncid )
if (status /= nf_noerr) stop "nf_open: error in generate_bz"

! read in any k-points provided by the user
status = nfgetglobaldim(ncid,'number_BZ_kpoints',nkpnum) 

if (status == nf_noerr) then 
   write(nconso,*) 'KPT: ',nkpnum,' BZ kpoints specified in file'
   call uflush(nconso)
   call read_kpoints(ncid,nkpnum,kpoints,nconso) 
else

! check if a Monkhorst-Pack grid or Chadi-Cohen is specified
   gridtype = ""  ! clear buffer 
   status = nfget(ncid,'KpointSetup%gridtype',gridtype) 

   if (status==nf_noerr) then 

      write(nconso,*) 'KPT: KpointSetup % gridtype = ', trim(gridtype)

      if (gridtype==MonkhorstPack) then 
        call Monkhorst_Pack_kpoints(ncid,nkpnum,kpoints,nconso)
      elseif (gridtype==ChadiCohen) then
        call Chadi_Cohen_kpoints(ncid,nkpnum,kpoints,nconso) 
      else 
        write(nconso,*) 'KPT: Could not recognize KpointSetup%gridtype:',gridtype
!       set default (Gamma-point) 
        call set_default_kpoints(nkpnum,kpoints,nconso)
      endif
   else
      call set_default_kpoints(nkpnum,kpoints,nconso)
   endif 
endif
    

! write out k-points
if (idebug > 0) then
  do  nkp=1,nkpnum 
     write(nconso,1100) kpoints(1,nkp),kpoints(2,nkp),kpoints(3,nkp)
1100 format(1x,'KPT: BZ_coordinate: ',3f12.8)
  enddo
endif

! get the number of k-points in memory 
! either from BZKpoints%HowManyInMemory or from KpointSetup%HowManyInMemory 
status = nfget(ncid,'BZKpoints%HowManyKpointsInMemory',nkpmem) 
if (status==nf_noerr) then
  if (nkpmem<=0) then 
     write(nconso,*) 'KPT: nkpmen <= 0 : nkpmem = 1 used' 
     nkpmem = 1
  endif
else 
  nkpmem = 1
endif

status = nfget(ncid,'KpointSetup%HowManyKpointsInMemory',nkpmem)
if (status==nf_noerr) then
  if (nkpmem<=0) then
     write(nconso,*) 'KPT: nkpmen <= 0 : nkpmem = 1 used'
     nkpmem = 1
  endif
else
  nkpmem = 1
endif                                                                 

status = nf_close(ncid)

endif ! lmastr

#ifdef PARAL
! send nkpnum,kpoints,nkpmem to slaves 
call mspack_integer_scalar (nconso, ANY, MSG_SETUP,&
        INTEGER4, nkpnum,  1, nOK)
if (.not.lmastr) then 
  allocate(kpoints(3,nkpnum))
endif
call mspack_double_array (nconso,ANY,MSG_SETUP,&
        REAL8, kpoints,3*nkpnum, nOK)
call mspack_integer_scalar (nconso, ANY, MSG_SETUP,&
        INTEGER4, nkpmem,  1, nOK)
#endif PARAL

end subroutine generate_bz_kpoints


!-------------------------------------------------------------

subroutine read_kpoints(ncid,nkpnum,kpoints,nconso) 
use netcdfinterface
implicit none

! read k-points from the netCDF variable 
! BZKpoints(number_BZ_kpoints,3)  

! ncid(in)      : id for netCDF file
! nkpnum(in)    : (=number_BZ_kpoints) 
! kpoints(out)  : kpoints read from the BZKpoints variable
! nconso(in)    : text file

integer, intent(in) :: ncid,nkpnum
real*8 , pointer      :: kpoints(:,:) 
integer, intent(in) :: nconso 

! locals
integer :: status


! check nkpnum
if (nkpnum.le.0) then
  write(nconso,*) 'KPT: number of k-points specified should be > 0'
  call clexit(nconso)
endif
if (nkpnum.gt.100000) then
     write(nconso,*) 'KPT: Number of k-points is too big'
     call clexit(nconso)
endif

! allocate room for k-points
allocate( kpoints(3,nkpnum))

! read kpoints
status = nfget(ncid,'BZKpoints',kpoints)
if (status /= nf_noerr) then 
   write(nconso,*) 'KPT: error reading netCDF variable BZKpoints'
   call clexit(nconso)
endif

end subroutine  read_kpoints


!--------------------------------------------------------------
subroutine set_default_kpoints(nkpnum,kpoints,nconso) 
implicit none

! set default k-point set, now defined as the Gamma point

! nkpnum  : number of k-points returned
! kpoints : kpoints
! nconso  : text output

integer, intent(out) :: nkpnum 
real*8 , pointer :: kpoints(:,:) 
integer, intent(in)  :: nconso

write(nconso,*) 'KPT: Using default (Gamma-point) k-points'
nkpnum = 1
! allocate room for k-points
allocate(kpoints(3,nkpnum))
kpoints(1,1) = 0.0d0
kpoints(2,1) = 0.0d0
kpoints(3,1) = 0.0d0

end subroutine set_default_kpoints


!--------------------------------------------------------------
subroutine Monkhorst_Pack_kpoints(ncid,nkpnum,kpoints,nconso)
use netcdfinterface
implicit none

! Define Monkhorst-Pack kpoint set, 
! defined by the KpointSetup variable  

! ncid   (in)   : id for the netCDF file 
! nkpnum (out)  : The number of k-points generated 
! kpoints (out) : k-points returned
! nconso(in)    : text file

integer, intent(in)  :: ncid
integer, intent(out) :: nkpnum
real*8 , pointer :: kpoints(:,:)
integer, intent(in)  :: nconso



! locals 
integer  :: nkp,mp_grid(3),i1,i2,i3,status,i

status = nfget(ncid,'KpointSetup',mp_grid)
if (status /= nf_noerr) then 
!    at this point this should be defined'
     stop 'nfget error reading KpointSetup'
endif

nkpnum = mp_grid(1)*mp_grid(2)*mp_grid(3)
if ((mp_grid(1)<1).or.(mp_grid(2)<1).or.(mp_grid(3)<1)) then 
   write(nconso,*) 'KPT: Warning zero or negative dimension for MonkhortsPack grid'
   write(nconso,*) 'KPT: using 1x1x1'
   mp_grid(1)=1 ; mp_grid(2)=1 ; mp_grid(3)=1
   nkpnum = 1
endif
write(nconso,*) 'KPT: Using MonkhortsPack grid (',(mp_grid(i),i=1,3),')' 

! allocate room for k-points
allocate(kpoints(3,nkpnum))

nkp = 1
do i1=1,mp_grid(1)
  do i2=1,mp_grid(2)
    do i3=1,mp_grid(3)

       kpoints(1,nkp) = dble(1+2*(i1-1)-mp_grid(1))/(2*mp_grid(1))
       kpoints(2,nkp) = dble(1+2*(i2-1)-mp_grid(2))/(2*mp_grid(2))
       kpoints(3,nkp) = dble(1+2*(i3-1)-mp_grid(3))/(2*mp_grid(3))
       nkp = nkp + 1

    enddo 
  enddo 
enddo

end subroutine Monkhorst_Pack_kpoints


!--------------------------------------------------------------

subroutine Chadi_Cohen_kpoints(ncid,nkpnum,kpoints,nconso) 
use netcdfinterface
implicit none

! Define Chadi-Cohen 1-dimensional k-points special k-points  
! sets

! ncid(in)     : id for netCDF file
! nkpnum (out) : number of k-points returned
! kpoints(out) : kpoints  returned

! The type is read from the netCDF variable KpointSetup(1)
! type  : one of 
!   1     cha_coh_1x1_6             6 k-points 1x1  
!   2     cha_coh_s3xs3_18          18-kpoints sqrt(3)*sqrt(3)
!   3     cha_coh_1x1_18            18-kpoints  1x1
!   4     cha_coh_s3xs3_54          54-kpoints sqrt(3)*sqrt(3)
!   5     cha_coh_1x1_54            54-kpoints  1x1
!   6     cha_coh_1x1_162           162-kpoints 1x1
!   7     cha_coh_2x3_12            12-kpoints  2x3
!   8     cha_coh_3xs3_162          162-kpoints  3xsqrt 3


integer , intent(in)  ::   ncid
integer , intent(out) ::   nkpnum
real*8  , pointer :: kpoints(:,:) 
integer , intent(in)  :: nconso

! define constants for k-point sets

! 6 k-points 1x1
integer, parameter :: cha_coh_1x1_6_dim = 6
character(*), parameter :: cha_coh_1x1_6_txt = '6 k-points 1x1'
real*8, parameter ::  cha_coh_1x1_6(3*6) = (/ &
      .33333333d0,  .33333333d0,  .0d0, &
      .33333333d0,  .00000000d0,  .0d0, &
      .00000000d0, -.33333333d0,  .0d0, &
     -.33333333d0, -.33333333d0,  .0d0, &
     -.33333333d0,  .00000000d0,  .0d0, &
      .00000000d0,  .33333333d0,  .0d0 /)

! 18 k-points sqrt(3) x sqrt(3) 
integer, parameter :: cha_coh_s3xs3_18_dim = 18 
character(*), parameter :: cha_coh_s3xs3_18_txt = '18 k-points sqrt(3) x sqrt(3)'
real*8 :: cha_coh_s3xs3_18(3*18) = (/ & 
      .11111111d0,  .11111111d0,  .0d0, &
      .22222222d0,  .22222222d0,  .0d0, &
      .44444444d0,  .11111111d0,  .0d0, &
      .22222222d0, -.11111111d0,  .0d0, &
      .44444444d0, -.22222222d0,  .0d0, &
      .55555555d0, -.11111111d0,  .0d0, &
      .55555555d0, -.44444444d0,  .0d0, &
      .44444444d0, -.55555555d0,  .0d0, &
      .11111111d0, -.55555555d0,  .0d0, &
      .22222222d0, -.44444444d0,  .0d0, &
     -.11111111d0, -.44444444d0,  .0d0, &
      .11111111d0, -.22222222d0,  .0d0, &
     -.22222222d0, -.22222222d0,  .0d0, &
     -.11111111d0, -.11111111d0,  .0d0, &
     -.22222222d0,  .11111111d0,  .0d0, &
     -.11111111d0,  .22222222d0,  .0d0, &
     -.44444444d0,  .22222222d0,  .0d0, &
     -.22222222d0,  .44444444d0,  .0d0 /)

! 18 k-points 1x1 
integer, parameter :: cha_coh_1x1_18_dim = 18 
character(*), parameter :: cha_coh_1x1_18_txt = '18 k-points 1x1' 
real*8, parameter :: cha_coh_1x1_18(3*18) = (/ &  
      .11111111d0,  .22222222d0,  .000d0, &
      .11111111d0,  .55555555d0,  .000d0, &
      .22222222d0,  .44444444d0,  .000d0, &
      .44444444d0,  .22222222d0,  .000d0, &
      .44444444d0,  .55555555d0,  .000d0, &
      .55555555d0,  .44444444d0,  .000d0, &
      .11111111d0, -.11111111d0,  .000d0, &
      .22222222d0, -.22222222d0,  .000d0, &
      .22222222d0,  .11111111d0,  .000d0, &
     -.11111111d0,  .44444444d0,  .000d0, &
     -.11111111d0,  .11111111d0,  .000d0, &
     -.11111111d0, -.22222222d0,  .000d0, &
     -.22222222d0,  .22222222d0,  .000d0, &
     -.22222222d0, -.11111111d0,  .000d0, &
     -.22222222d0, -.44444444d0,  .000d0, &
     -.44444444d0,  .11111111d0,  .000d0, &
     -.44444444d0, -.22222222d0,  .000d0, &
     -.55555555d0, -.11111111d0,  .000d0 /)

! 54 k-points sqrt(3) x sqrt(3) 
integer, parameter :: cha_coh_s3xs3_54_dim = 54 
character(*), parameter :: cha_coh_s3xs3_54_txt = '54 k-points sqrt(3) x sqrt(3)' 
real*8, parameter ::  cha_coh_s3xs3_54(3*54) = (/ &
      .22222222d0, -.55555556d0,  .00000000d0, &
      .33333333d0, -.55555556d0,  .00000000d0, &
      .00000000d0, -.44444444d0,  .00000000d0, &
      .11111111d0, -.44444444d0,  .00000000d0, &
      .33333333d0, -.44444444d0,  .00000000d0, &
      .44444444d0, -.44444444d0,  .00000000d0, &
     -.22222222d0, -.33333333d0,  .00000000d0, &
     -.11111111d0, -.33333333d0,  .00000000d0, &
      .11111111d0, -.33333333d0,  .00000000d0, &
      .22222222d0, -.33333333d0,  .00000000d0, &
      .44444444d0, -.33333333d0,  .00000000d0, &
      .55555556d0, -.33333333d0,  .00000000d0, &
     -.33333333d0, -.22222222d0,  .00000000d0, &
     -.11111111d0, -.22222222d0,  .00000000d0, &
      .00000000d0, -.22222222d0,  .00000000d0, &
      .22222222d0, -.22222222d0,  .00000000d0, &
      .33333333d0, -.22222222d0,  .00000000d0, &
      .55555556d0, -.22222222d0,  .00000000d0, &
     -.33333333d0, -.11111111d0,  .00000000d0, &
     -.22222222d0, -.11111111d0,  .00000000d0, &
      .00000000d0, -.11111111d0,  .00000000d0, &
      .11111111d0, -.11111111d0,  .00000000d0, &
      .33333333d0, -.11111111d0,  .00000000d0, &
      .44444444d0, -.11111111d0,  .00000000d0, &
     -.44444444d0,  .00000000d0,  .00000000d0, &
     -.22222222d0,  .00000000d0,  .00000000d0, &
     -.11111111d0,  .00000000d0,  .00000000d0, &
      .11111111d0,  .00000000d0,  .00000000d0, &
      .22222222d0,  .00000000d0,  .00000000d0, &
      .44444444d0,  .00000000d0,  .00000000d0, &
     -.44444444d0,  .11111111d0,  .00000000d0, &
     -.33333333d0,  .11111111d0,  .00000000d0, &
     -.11111111d0,  .11111111d0,  .00000000d0, &
      .00000000d0,  .11111111d0,  .00000000d0, &
      .22222222d0,  .11111111d0,  .00000000d0, &
      .33333333d0,  .11111111d0,  .00000000d0, &
     -.55555556d0,  .22222222d0,  .00000000d0, &
     -.33333333d0,  .22222222d0,  .00000000d0, &
     -.22222222d0,  .22222222d0,  .00000000d0, &
      .00000000d0,  .22222222d0,  .00000000d0, &
      .11111111d0,  .22222222d0,  .00000000d0, &
      .33333333d0,  .22222222d0,  .00000000d0, &
     -.55555556d0,  .33333333d0,  .00000000d0, &
     -.44444444d0,  .33333333d0,  .00000000d0, &
     -.22222222d0,  .33333333d0,  .00000000d0, &
     -.11111111d0,  .33333333d0,  .00000000d0, &
      .11111111d0,  .33333333d0,  .00000000d0, &
      .22222222d0,  .33333333d0,  .00000000d0, &
     -.44444444d0,  .44444444d0,  .00000000d0, &
     -.33333333d0,  .44444444d0,  .00000000d0, &
     -.11111111d0,  .44444444d0,  .00000000d0, &
      .00000000d0,  .44444444d0,  .00000000d0, &
     -.33333333d0,  .55555556d0,  .00000000d0, &
     -.22222222d0,  .55555556d0,  .00000000d0 /)

! 1x1 54 k-points 
integer, parameter :: cha_coh_1x1_54_dim = 54
character(*), parameter :: cha_coh_1x1_54_txt = '1x1 54 k-points'
real*8, parameter ::  cha_coh_1x1_54(3*54)  = (/ &
      .11111111d0,  .11111111d0,  .000d0, &
      .22222222d0,  .22222222d0,  .000d0, &
      .44444444d0,  .44444444d0,  .000d0, &
      .33333333d0,  .44444444d0,  .000d0, &
      .22222222d0,  .33333333d0,  .000d0, &
      .33333333d0,  .55555555d0,  .000d0, &
      .22222222d0,  .55555555d0,  .000d0, &
      .11111111d0,  .33333333d0,  .000d0, &
      .11111111d0,  .44444444d0,  .000d0, &
      .00000000d0,  .11111111d0,  .000d0, &
      .00000000d0,  .22222222d0,  .000d0, &
      .00000000d0,  .44444444d0,  .000d0, &
     -.11111111d0,  .33333333d0,  .000d0, &
     -.11111111d0,  .22222222d0,  .000d0, &
     -.22222222d0,  .33333333d0,  .000d0, &
     -.33333333d0,  .22222222d0,  .000d0, &
     -.22222222d0,  .11111111d0,  .000d0, &
     -.33333333d0,  .11111111d0,  .000d0, &
     -.11111111d0, -.00000000d0,  .000d0, &
     -.22222222d0, -.00000000d0,  .000d0, &
     -.44444444d0, -.00000000d0,  .000d0, &
     -.44444444d0, -.11111111d0,  .000d0, &
     -.33333333d0, -.11111111d0,  .000d0, &
     -.55555555d0, -.22222222d0,  .000d0, &
     -.55555555d0, -.33333333d0,  .000d0, &
     -.33333333d0, -.22222222d0,  .000d0, &
     -.44444444d0, -.33333333d0,  .000d0, &
     -.11111111d0, -.11111111d0,  .000d0, &
     -.22222222d0, -.22222222d0,  .000d0, &
     -.44444444d0, -.44444444d0,  .000d0, &
      .22222222d0, -.11111111d0,  .000d0, &
      .33333333d0, -.11111111d0,  .000d0, &
      .33333333d0, -.22222222d0,  .000d0, &
      .11111111d0,  .00000000d0,  .000d0, &
      .22222222d0,  .00000000d0,  .000d0, &
      .33333333d0,  .11111111d0,  .000d0, &
      .33333333d0,  .22222222d0,  .000d0, &
      .44444444d0,  .33333333d0,  .000d0, &
      .44444444d0,  .00000000d0,  .000d0, &
      .44444444d0,  .11111111d0,  .000d0, &
      .55555555d0,  .22222222d0,  .000d0, &
      .55555555d0,  .33333333d0,  .000d0, &
      .11111111d0, -.22222222d0,  .000d0, &
      .11111111d0, -.33333333d0,  .000d0, &
      .22222222d0, -.33333333d0,  .000d0, &
     -.00000000d0, -.11111111d0,  .000d0, &
     -.00000000d0, -.22222222d0,  .000d0, &
     -.11111111d0, -.33333333d0,  .000d0, &
     -.22222222d0, -.33333333d0,  .000d0, &
     -.33333333d0, -.44444444d0,  .000d0, &
     -.00000000d0, -.44444444d0,  .000d0, &
     -.11111111d0, -.44444444d0,  .000d0, &
     -.22222222d0, -.55555555d0,  .000d0, &
     -.33333333d0, -.55555555d0,  .000d0  /)

! 1x1 162 points 
integer, parameter :: cha_coh_1x1_162_dim = 162
character(*), parameter :: cha_coh_1x1_162_txt = '1x1 162 points'
real*8                  ::  cha_coh_1x1_162(3*162) 
data  cha_coh_1x1_162(1:156)  / &
     -.2962962963d0, -.5925925926d0,  .0000000000d0, &
     -.3703703704d0, -.5185185185d0,  .0000000000d0, &
     -.2592592593d0, -.5185185185d0,  .0000000000d0, &
     -.1481481481d0, -.5185185185d0,  .0000000000d0, &
     -.4074074074d0, -.4814814815d0,  .0000000000d0, &
     -.2962962963d0, -.4814814815d0,  .0000000000d0, &
     -.1851851852d0, -.4814814815d0,  .0000000000d0, &
     -.0740740741d0, -.4814814815d0,  .0000000000d0, &
     -.4814814815d0, -.4074074074d0,  .0000000000d0, &
     -.3703703704d0, -.4074074074d0,  .0000000000d0, &
     -.2592592593d0, -.4074074074d0,  .0000000000d0, &
     -.1481481481d0, -.4074074074d0,  .0000000000d0, &
     -.0370370370d0, -.4074074074d0,  .0000000000d0, &
      .0740740741d0, -.4074074074d0,  .0000000000d0, &
     -.5185185185d0, -.3703703704d0,  .0000000000d0, &
     -.4074074074d0, -.3703703704d0,  .0000000000d0, &
     -.2962962963d0, -.3703703704d0,  .0000000000d0, &
     -.1851851852d0, -.3703703704d0,  .0000000000d0, &
     -.0740740741d0, -.3703703704d0,  .0000000000d0, &
      .0370370370d0, -.3703703704d0,  .0000000000d0, &
      .1481481481d0, -.3703703704d0,  .0000000000d0, &
     -.5925925926d0, -.2962962963d0,  .0000000000d0, &
     -.4814814815d0, -.2962962963d0,  .0000000000d0, &
     -.3703703704d0, -.2962962963d0,  .0000000000d0, &
     -.2592592593d0, -.2962962963d0,  .0000000000d0, &
     -.1481481481d0, -.2962962963d0,  .0000000000d0, &
     -.0370370370d0, -.2962962963d0,  .0000000000d0, &
      .0740740741d0, -.2962962963d0,  .0000000000d0, &
      .1851851852d0, -.2962962963d0,  .0000000000d0, &
      .2962962963d0, -.2962962963d0,  .0000000000d0, &
     -.5185185185d0, -.2592592593d0,  .0000000000d0, &
     -.4074074074d0, -.2592592593d0,  .0000000000d0, &
     -.2962962963d0, -.2592592593d0,  .0000000000d0, &
     -.1851851852d0, -.2592592593d0,  .0000000000d0, &
     -.0740740741d0, -.2592592593d0,  .0000000000d0, &
      .0370370370d0, -.2592592593d0,  .0000000000d0, &
      .1481481481d0, -.2592592593d0,  .0000000000d0, &
      .2592592593d0, -.2592592593d0,  .0000000000d0, &
      .3703703704d0, -.2592592593d0,  .0000000000d0, &
     -.4814814815d0, -.1851851852d0,  .0000000000d0, &
     -.3703703704d0, -.1851851852d0,  .0000000000d0, &
     -.2592592593d0, -.1851851852d0,  .0000000000d0, &
     -.1481481481d0, -.1851851852d0,  .0000000000d0, &
     -.0370370370d0, -.1851851852d0,  .0000000000d0, &
      .0740740741d0, -.1851851852d0,  .0000000000d0, &
      .1851851852d0, -.1851851852d0,  .0000000000d0, &
      .2962962963d0, -.1851851852d0,  .0000000000d0, &
      .4074074074d0, -.1851851852d0,  .0000000000d0, &
     -.5185185185d0, -.1481481481d0,  .0000000000d0, &
     -.4074074074d0, -.1481481481d0,  .0000000000d0, &
     -.2962962963d0, -.1481481481d0,  .0000000000d0, &
     -.1851851852d0, -.1481481481d0,  .0000000000d0  /

data  cha_coh_1x1_162(157:312)  / &
     -.0740740741d0, -.1481481481d0,  .0000000000d0, &
      .0370370370d0, -.1481481481d0,  .0000000000d0, &
      .1481481481d0, -.1481481481d0,  .0000000000d0, &
      .2592592593d0, -.1481481481d0,  .0000000000d0, &
      .3703703704d0, -.1481481481d0,  .0000000000d0, &
     -.4814814815d0, -.0740740741d0,  .0000000000d0, &
     -.3703703704d0, -.0740740741d0,  .0000000000d0, &
     -.2592592593d0, -.0740740741d0,  .0000000000d0, &
     -.1481481481d0, -.0740740741d0,  .0000000000d0, &
     -.0370370370d0, -.0740740741d0,  .0000000000d0, &
      .0740740741d0, -.0740740741d0,  .0000000000d0, &
      .1851851852d0, -.0740740741d0,  .0000000000d0, &
      .2962962963d0, -.0740740741d0,  .0000000000d0, &
      .4074074074d0, -.0740740741d0,  .0000000000d0, &
     -.4074074074d0, -.0370370370d0,  .0000000000d0, &
     -.2962962963d0, -.0370370370d0,  .0000000000d0, &
     -.1851851852d0, -.0370370370d0,  .0000000000d0, &
     -.0740740741d0, -.0370370370d0,  .0000000000d0, &
      .0370370370d0, -.0370370370d0,  .0000000000d0, &
      .1481481481d0, -.0370370370d0,  .0000000000d0, &
      .2592592593d0, -.0370370370d0,  .0000000000d0, &
      .3703703704d0, -.0370370370d0,  .0000000000d0, &
      .4814814815d0, -.0370370370d0,  .0000000000d0, &
     -.3703703704d0,  .0370370370d0,  .0000000000d0, &
     -.2592592593d0,  .0370370370d0,  .0000000000d0, &
     -.1481481481d0,  .0370370370d0,  .0000000000d0, &
     -.0370370370d0,  .0370370370d0,  .0000000000d0, &
      .0740740741d0,  .0370370370d0,  .0000000000d0, &
      .1851851852d0,  .0370370370d0,  .0000000000d0, &
      .2962962963d0,  .0370370370d0,  .0000000000d0, &
      .4074074074d0,  .0370370370d0,  .0000000000d0, &
      .5185185185d0,  .0370370370d0,  .0000000000d0, &
     -.4074074074d0,  .0740740741d0,  .0000000000d0, &
     -.2962962963d0,  .0740740741d0,  .0000000000d0, &
     -.1851851852d0,  .0740740741d0,  .0000000000d0, &
     -.0740740741d0,  .0740740741d0,  .0000000000d0, &
      .0370370370d0,  .0740740741d0,  .0000000000d0, &
      .1481481481d0,  .0740740741d0,  .0000000000d0, &
      .2592592593d0,  .0740740741d0,  .0000000000d0, &
      .3703703704d0,  .0740740741d0,  .0000000000d0, &
      .4814814815d0,  .0740740741d0,  .0000000000d0, &
     -.3703703704d0,  .1481481481d0,  .0000000000d0, &
     -.2592592593d0,  .1481481481d0,  .0000000000d0, &
     -.1481481481d0,  .1481481481d0,  .0000000000d0, &
     -.0370370370d0,  .1481481481d0,  .0000000000d0, &
      .0740740741d0,  .1481481481d0,  .0000000000d0, &
      .1851851852d0,  .1481481481d0,  .0000000000d0, &
      .2962962963d0,  .1481481481d0,  .0000000000d0, &
      .4074074074d0,  .1481481481d0,  .0000000000d0, &
      .5185185185d0,  .1481481481d0,  .0000000000d0, &
     -.2962962963d0,  .1851851852d0,  .0000000000d0, &
     -.1851851852d0,  .1851851852d0,  .0000000000d0  /

data  cha_coh_1x1_162(313:468)  / &
     -.0740740741d0,  .1851851852d0,  .0000000000d0, &
      .0370370370d0,  .1851851852d0,  .0000000000d0, &
      .1481481481d0,  .1851851852d0,  .0000000000d0, &
      .2592592593d0,  .1851851852d0,  .0000000000d0, &
      .3703703704d0,  .1851851852d0,  .0000000000d0, &
      .4814814815d0,  .1851851852d0,  .0000000000d0, &
      .5925925926d0,  .1851851852d0,  .0000000000d0, &
     -.2592592593d0,  .2592592593d0,  .0000000000d0, &
     -.1481481481d0,  .2592592593d0,  .0000000000d0, &
     -.0370370370d0,  .2592592593d0,  .0000000000d0, &
      .0740740741d0,  .2592592593d0,  .0000000000d0, &
      .1851851852d0,  .2592592593d0,  .0000000000d0, &
      .2962962963d0,  .2592592593d0,  .0000000000d0, &
      .4074074074d0,  .2592592593d0,  .0000000000d0, &
      .5185185185d0,  .2592592593d0,  .0000000000d0, &
      .6296296296d0,  .2592592593d0,  .0000000000d0, &
     -.2962962963d0,  .2962962963d0,  .0000000000d0, &
     -.1851851852d0,  .2962962963d0,  .0000000000d0, &
     -.0740740741d0,  .2962962963d0,  .0000000000d0, &
      .0370370370d0,  .2962962963d0,  .0000000000d0, &
      .1481481481d0,  .2962962963d0,  .0000000000d0, &
      .2592592593d0,  .2962962963d0,  .0000000000d0, &
      .3703703704d0,  .2962962963d0,  .0000000000d0, &
      .4814814815d0,  .2962962963d0,  .0000000000d0, &
      .5925925926d0,  .2962962963d0,  .0000000000d0, &
     -.2592592593d0,  .3703703704d0,  .0000000000d0, &
     -.1481481481d0,  .3703703704d0,  .0000000000d0, &
     -.0370370370d0,  .3703703704d0,  .0000000000d0, &
      .0740740741d0,  .3703703704d0,  .0000000000d0, &
      .1851851852d0,  .3703703704d0,  .0000000000d0, &
      .2962962963d0,  .3703703704d0,  .0000000000d0, &
      .4074074074d0,  .3703703704d0,  .0000000000d0, &
      .5185185185d0,  .3703703704d0,  .0000000000d0, &
      .6296296296d0,  .3703703704d0,  .0000000000d0, &
     -.1851851852d0,  .4074074074d0,  .0000000000d0, &
     -.0740740741d0,  .4074074074d0,  .0000000000d0, &
      .0370370370d0,  .4074074074d0,  .0000000000d0, &
      .1481481481d0,  .4074074074d0,  .0000000000d0, &
      .2592592593d0,  .4074074074d0,  .0000000000d0, &
      .3703703704d0,  .4074074074d0,  .0000000000d0, &
      .4814814815d0,  .4074074074d0,  .0000000000d0, &
      .5925925926d0,  .4074074074d0,  .0000000000d0, &
     -.0370370370d0,  .4814814815d0,  .0000000000d0, &
      .0740740741d0,  .4814814815d0,  .0000000000d0, &
      .1851851852d0,  .4814814815d0,  .0000000000d0, &
      .2962962963d0,  .4814814815d0,  .0000000000d0, &
      .4074074074d0,  .4814814815d0,  .0000000000d0, &
      .5185185185d0,  .4814814815d0,  .0000000000d0, &
      .0370370370d0,  .5185185185d0,  .0000000000d0, &
      .1481481481d0,  .5185185185d0,  .0000000000d0, &
      .2592592593d0,  .5185185185d0,  .0000000000d0, &
      .3703703704d0,  .5185185185d0,  .0000000000d0 /

data  cha_coh_1x1_162(469:486)  / &
      .4814814815d0,  .5185185185d0,  .0000000000d0, &
      .1851851852d0,  .5925925926d0,  .0000000000d0, &
      .2962962963d0,  .5925925926d0,  .0000000000d0, &
      .4074074074d0,  .5925925926d0,  .0000000000d0, &
      .2592592593d0,  .6296296296d0,  .0000000000d0, &
      .3703703704d0,  .6296296296d0,  .0000000000d0 /

! 12 k-points 2x3 
integer, parameter :: cha_coh_2x3_12_dim = 12
character(*), parameter :: cha_coh_2x3_12_txt = '12 k-points 2x3'
real*8, parameter ::  cha_coh_2x3_12(3*12)  = (/ &
      .1666666667d0,  .2222222222d0,  .0000000000d0, &
      .1666666667d0,  .5555555555d0,  .0000000000d0, &
      .3333333333d0,  .4444444444d0,  .0000000000d0, &
      .1666666667d0, -.1111111111d0,  .0000000000d0, &
      .3333333333d0, -.2222222222d0,  .0000000000d0, &
      .3333333333d0,  .1111111111d0,  .0000000000d0, &
     -.1666666667d0,  .4444444444d0,  .0000000000d0, &
     -.1666666667d0,  .1111111111d0,  .0000000000d0, &
     -.1666666667d0, -.2222222222d0,  .0000000000d0, &
     -.3333333333d0,  .2222222222d0,  .0000000000d0, &
     -.3333333333d0, -.1111111111d0,  .0000000000d0, &
     -.3333333333d0, -.4444444444d0,  .0000000000d0 /)

! 162 sqrt 3 x sqrt 3     
integer, parameter :: cha_coh_s3xs3_162_dim = 162
character(*), parameter :: cha_coh_s3xs3_162_txt = '162 sqrt 3 x sqrt 3'
real*8                  ::  cha_coh_s3xs3_162(3*162) 
data  cha_coh_s3xs3_162(1:156) / &
     -0.2962962963d0,  0.5925925926d0,  0.00000000d0, &
     -0.3703703704d0,  0.5185185185d0,  0.00000000d0, &
     -0.2592592593d0,  0.5185185185d0,  0.00000000d0, &
     -0.1481481481d0,  0.5185185185d0,  0.00000000d0, &
     -0.4074074074d0,  0.4814814815d0,  0.00000000d0, &
     -0.2962962963d0,  0.4814814815d0,  0.00000000d0, &
     -0.1851851852d0,  0.4814814815d0,  0.00000000d0, &
     -0.0740740741d0,  0.4814814815d0,  0.00000000d0, &
     -0.4814814815d0,  0.4074074074d0,  0.00000000d0, &
     -0.3703703704d0,  0.4074074074d0,  0.00000000d0, &
     -0.2592592593d0,  0.4074074074d0,  0.00000000d0, &
     -0.1481481481d0,  0.4074074074d0,  0.00000000d0, &
     -0.0370370370d0,  0.4074074074d0,  0.00000000d0, &
      0.0740740741d0,  0.4074074074d0,  0.00000000d0, &
     -0.5185185185d0,  0.3703703704d0,  0.00000000d0, &
     -0.4074074074d0,  0.3703703704d0,  0.00000000d0, &
     -0.2962962963d0,  0.3703703704d0,  0.00000000d0, &
     -0.1851851852d0,  0.3703703704d0,  0.00000000d0, &
     -0.0740740741d0,  0.3703703704d0,  0.00000000d0, &
      0.0370370370d0,  0.3703703704d0,  0.00000000d0, &
      0.1481481481d0,  0.3703703704d0,  0.00000000d0, &
     -0.5925925926d0,  0.2962962963d0,  0.00000000d0, &
     -0.4814814815d0,  0.2962962963d0,  0.00000000d0, &
     -0.3703703704d0,  0.2962962963d0,  0.00000000d0, &
     -0.2592592593d0,  0.2962962963d0,  0.00000000d0, &
     -0.1481481481d0,  0.2962962963d0,  0.00000000d0, &
     -0.0370370370d0,  0.2962962963d0,  0.00000000d0, &
      0.0740740741d0,  0.2962962963d0,  0.00000000d0, &
      0.1851851852d0,  0.2962962963d0,  0.00000000d0, &
      0.2962962963d0,  0.2962962963d0,  0.00000000d0, &
     -0.5185185185d0,  0.2592592593d0,  0.00000000d0, &
     -0.4074074074d0,  0.2592592593d0,  0.00000000d0, &
     -0.2962962963d0,  0.2592592593d0,  0.00000000d0, &
     -0.1851851852d0,  0.2592592593d0,  0.00000000d0, &
     -0.0740740741d0,  0.2592592593d0,  0.00000000d0, &
      0.0370370370d0,  0.2592592593d0,  0.00000000d0, &
      0.1481481481d0,  0.2592592593d0,  0.00000000d0, &
      0.2592592593d0,  0.2592592593d0,  0.00000000d0, &
      0.3703703704d0,  0.2592592593d0,  0.00000000d0, &
     -0.4814814815d0,  0.1851851852d0,  0.00000000d0, &
     -0.3703703704d0,  0.1851851852d0,  0.00000000d0, &
     -0.2592592593d0,  0.1851851852d0,  0.00000000d0, &
     -0.1481481481d0,  0.1851851852d0,  0.00000000d0, &
     -0.0370370370d0,  0.1851851852d0,  0.00000000d0, &
      0.0740740741d0,  0.1851851852d0,  0.00000000d0, &
      0.1851851852d0,  0.1851851852d0,  0.00000000d0, &
      0.2962962963d0,  0.1851851852d0,  0.00000000d0, &
      0.4074074074d0,  0.1851851852d0,  0.00000000d0, &
     -0.5185185185d0,  0.1481481481d0,  0.00000000d0, &
     -0.4074074074d0,  0.1481481481d0,  0.00000000d0, &
     -0.2962962963d0,  0.1481481481d0,  0.00000000d0, &
     -0.1851851852d0,  0.1481481481d0,  0.00000000d0  /

data  cha_coh_s3xs3_162(157:312) / &
     -0.0740740741d0,  0.1481481481d0,  0.00000000d0, &
      0.0370370370d0,  0.1481481481d0,  0.00000000d0, &
      0.1481481481d0,  0.1481481481d0,  0.00000000d0, &
      0.2592592593d0,  0.1481481481d0,  0.00000000d0, &
      0.3703703704d0,  0.1481481481d0,  0.00000000d0, &
     -0.4814814815d0,  0.0740740741d0,  0.00000000d0, &
     -0.3703703704d0,  0.0740740741d0,  0.00000000d0, &
     -0.2592592593d0,  0.0740740741d0,  0.00000000d0, &
     -0.1481481481d0,  0.0740740741d0,  0.00000000d0, &
     -0.0370370370d0,  0.0740740741d0,  0.00000000d0, &
      0.0740740741d0,  0.0740740741d0,  0.00000000d0, &
      0.1851851852d0,  0.0740740741d0,  0.00000000d0, &
      0.2962962963d0,  0.0740740741d0,  0.00000000d0, &
      0.4074074074d0,  0.0740740741d0,  0.00000000d0, &
     -0.4074074074d0,  0.0370370370d0,  0.00000000d0, &
     -0.2962962963d0,  0.0370370370d0,  0.00000000d0, &
     -0.1851851852d0,  0.0370370370d0,  0.00000000d0, &
     -0.0740740741d0,  0.0370370370d0,  0.00000000d0, &
      0.0370370370d0,  0.0370370370d0,  0.00000000d0, &
      0.1481481481d0,  0.0370370370d0,  0.00000000d0, &
      0.2592592593d0,  0.0370370370d0,  0.00000000d0, &
      0.3703703704d0,  0.0370370370d0,  0.00000000d0, &
      0.4814814815d0,  0.0370370370d0,  0.00000000d0, &
     -0.3703703704d0, -0.0370370370d0,  0.00000000d0, &
     -0.2592592593d0, -0.0370370370d0,  0.00000000d0, &
     -0.1481481481d0, -0.0370370370d0,  0.00000000d0, &
     -0.0370370370d0, -0.0370370370d0,  0.00000000d0, &
      0.0740740741d0, -0.0370370370d0,  0.00000000d0, &
      0.1851851852d0, -0.0370370370d0,  0.00000000d0, &
      0.2962962963d0, -0.0370370370d0,  0.00000000d0, &
      0.4074074074d0, -0.0370370370d0,  0.00000000d0, &
      0.5185185185d0, -0.0370370370d0,  0.00000000d0, &
     -0.4074074074d0, -0.0740740741d0,  0.00000000d0, &
     -0.2962962963d0, -0.0740740741d0,  0.00000000d0, &
     -0.1851851852d0, -0.0740740741d0,  0.00000000d0, &
     -0.0740740741d0, -0.0740740741d0,  0.00000000d0, &
      0.0370370370d0, -0.0740740741d0,  0.00000000d0, &
      0.1481481481d0, -0.0740740741d0,  0.00000000d0, &
      0.2592592593d0, -0.0740740741d0,  0.00000000d0, &
      0.3703703704d0, -0.0740740741d0,  0.00000000d0, &
      0.4814814815d0, -0.0740740741d0,  0.00000000d0, &
     -0.3703703704d0, -0.1481481481d0,  0.00000000d0, &
     -0.2592592593d0, -0.1481481481d0,  0.00000000d0, &
     -0.1481481481d0, -0.1481481481d0,  0.00000000d0, &
     -0.0370370370d0, -0.1481481481d0,  0.00000000d0, &
      0.0740740741d0, -0.1481481481d0,  0.00000000d0, &
      0.1851851852d0, -0.1481481481d0,  0.00000000d0, &
      0.2962962963d0, -0.1481481481d0,  0.00000000d0, &
      0.4074074074d0, -0.1481481481d0,  0.00000000d0, &
      0.5185185185d0, -0.1481481481d0,  0.00000000d0, &
     -0.2962962963d0, -0.1851851852d0,  0.00000000d0, &
     -0.1851851852d0, -0.1851851852d0,  0.00000000d0  /


data  cha_coh_s3xs3_162(313:468) / &
     -0.0740740741d0, -0.1851851852d0,  0.00000000d0, &
      0.0370370370d0, -0.1851851852d0,  0.00000000d0, &
      0.1481481481d0, -0.1851851852d0,  0.00000000d0, &
      0.2592592593d0, -0.1851851852d0,  0.00000000d0, &
      0.3703703704d0, -0.1851851852d0,  0.00000000d0, &
      0.4814814815d0, -0.1851851852d0,  0.00000000d0, &
      0.5925925926d0, -0.1851851852d0,  0.00000000d0, &
     -0.2592592593d0, -0.2592592593d0,  0.00000000d0, &
     -0.1481481481d0, -0.2592592593d0,  0.00000000d0, &
     -0.0370370370d0, -0.2592592593d0,  0.00000000d0, &
      0.0740740741d0, -0.2592592593d0,  0.00000000d0, &
      0.1851851852d0, -0.2592592593d0,  0.00000000d0, &
      0.2962962963d0, -0.2592592593d0,  0.00000000d0, &
      0.4074074074d0, -0.2592592593d0,  0.00000000d0, &
      0.5185185185d0, -0.2592592593d0,  0.00000000d0, &
      0.6296296296d0, -0.2592592593d0,  0.00000000d0, &
     -0.2962962963d0, -0.2962962963d0,  0.00000000d0, &
     -0.1851851852d0, -0.2962962963d0,  0.00000000d0, &
     -0.0740740741d0, -0.2962962963d0,  0.00000000d0, &
      0.0370370370d0, -0.2962962963d0,  0.00000000d0, &
      0.1481481481d0, -0.2962962963d0,  0.00000000d0, &
      0.2592592593d0, -0.2962962963d0,  0.00000000d0, &
      0.3703703704d0, -0.2962962963d0,  0.00000000d0, &
      0.4814814815d0, -0.2962962963d0,  0.00000000d0, &
      0.5925925926d0, -0.2962962963d0,  0.00000000d0, &
     -0.2592592593d0, -0.3703703704d0,  0.00000000d0, &
     -0.1481481481d0, -0.3703703704d0,  0.00000000d0, &
     -0.0370370370d0, -0.3703703704d0,  0.00000000d0, &
      0.0740740741d0, -0.3703703704d0,  0.00000000d0, &
      0.1851851852d0, -0.3703703704d0,  0.00000000d0, &
      0.2962962963d0, -0.3703703704d0,  0.00000000d0, &
      0.4074074074d0, -0.3703703704d0,  0.00000000d0, &
      0.5185185185d0, -0.3703703704d0,  0.00000000d0, &
      0.6296296296d0, -0.3703703704d0,  0.00000000d0, &
     -0.1851851852d0, -0.4074074074d0,  0.00000000d0, &
     -0.0740740741d0, -0.4074074074d0,  0.00000000d0, &
      0.0370370370d0, -0.4074074074d0,  0.00000000d0, &
      0.1481481481d0, -0.4074074074d0,  0.00000000d0, &
      0.2592592593d0, -0.4074074074d0,  0.00000000d0, &
      0.3703703704d0, -0.4074074074d0,  0.00000000d0, &
      0.4814814815d0, -0.4074074074d0,  0.00000000d0, &
      0.5925925926d0, -0.4074074074d0,  0.00000000d0, &
     -0.0370370370d0, -0.4814814815d0,  0.00000000d0, &
      0.0740740741d0, -0.4814814815d0,  0.00000000d0, &
      0.1851851852d0, -0.4814814815d0,  0.00000000d0, &
      0.2962962963d0, -0.4814814815d0,  0.00000000d0, &
      0.4074074074d0, -0.4814814815d0,  0.00000000d0, &
      0.5185185185d0, -0.4814814815d0,  0.00000000d0, &
      0.0370370370d0, -0.5185185185d0,  0.00000000d0, &
      0.1481481481d0, -0.5185185185d0,  0.00000000d0, &
      0.2592592593d0, -0.5185185185d0,  0.00000000d0, &
      0.3703703704d0, -0.5185185185d0,  0.00000000d0  /


data  cha_coh_s3xs3_162(469:486) / &
      0.4814814815d0, -0.5185185185d0,  0.00000000d0, &
      0.1851851852d0, -0.5925925926d0,  0.00000000d0, &
      0.2962962963d0, -0.5925925926d0,  0.00000000d0, &
      0.4074074074d0, -0.5925925926d0,  0.00000000d0, &
      0.2592592593d0, -0.6296296296d0,  0.00000000d0, &
      0.3703703704d0, -0.6296296296d0,  0.00000000d0 /

! locals 
character(50) :: txt
integer       :: ChadiCohenType,type(3),status


! get the type from KpointSetup(1)
status = nfget(ncid,'KpointSetup',type)
if (status /= nf_noerr) then
!    at this point this should be defined'
     stop 'nfget error reading ChadiCohen type from KpointSetup'
endif

ChadiCohenType = type(1)

select case (ChadiCohenType) 

   case(1)
     nkpnum = cha_coh_1x1_6_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_1x1_6,shape=(/3,nkpnum/)) 
     txt = cha_coh_1x1_6_txt
   case(2)
     nkpnum = cha_coh_s3xs3_18_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_s3xs3_18,shape=(/3,nkpnum/)) 
     txt = cha_coh_s3xs3_18_txt
   case(3)
     nkpnum =cha_coh_1x1_18_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_1x1_18 ,shape=(/3,nkpnum/)) 
     txt = cha_coh_1x1_18_txt
   case(4)
     nkpnum = cha_coh_s3xs3_54_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_s3xs3_54 ,shape=(/3,nkpnum/)) 
     txt = cha_coh_s3xs3_54_txt
   case(5)
     nkpnum = cha_coh_1x1_54_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_1x1_54 ,shape=(/3,nkpnum/)) 
     txt = cha_coh_1x1_54_txt
   case(6)
     nkpnum = cha_coh_1x1_162_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_1x1_162 ,shape=(/3,nkpnum/)) 
     txt = cha_coh_1x1_162_txt
   case(7)
     nkpnum = cha_coh_2x3_12_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_2x3_12 ,shape=(/3,nkpnum/)) 
     txt = cha_coh_2x3_12_txt
   case(8)
 !    nkpnum =cha_coh_s3xs3_162_dim ;  allocate(kpoints(3,nkpnum))
 !    kpoints = reshape(source= cha_coh_s3xs3_162 ,shape=(/3,nkpnum/)) 
 !    txt = cha_coh_s3xs3_162_txt
     write(nconso,*) 'KPT: Warning ChadiCohen type 8 not implemented'
     call clexit(nconso)
   case default
     write(nconso,*) 'KPT: Warning ChadiCohen type (KpointSetup(1)) should be in range 1..8'
     write(nconso,*) 'KPT: Using type 1' 
     nkpnum = cha_coh_1x1_6_dim ;  allocate(kpoints(3,nkpnum))
     kpoints = reshape(source=cha_coh_1x1_6,shape=(/3,nkpnum/))
     txt = cha_coh_1x1_6_txt

end select

write(nconso,*) 'KPT: Using Chadi-Cohen special kpoint set : ',trim(txt)


end subroutine Chadi_Cohen_kpoints

end module generate_kpoints
