     SUBROUTINE ion_conjgrad (nconso, ndim, x_external, etot, force, &
                              initial_step, safe_step, dx_external)
!=======================================================================
!    Dacapo print label: ICGRAD
!
!     Conjugate gradient algorithm with internal predictor/corrector line search !
!     This version features cubic fit along searchdirection   !
!     stepsize is proportional to gradient and step; dynamic step adjustment based 
!     on statistical assumption for optimal step sizes
!     Autodetect number of active_DOF (|force(i)| > 0) - explicit reset conjugation
!     when conjug_counter >  active_DOFs
!     - 20Sep2000: suppressed maxtry dimension, to simplify things for now
!                  reject too large predictor steps (-> eps_reject_trial)
!-----------------------------------------------------------------------
! major internal variables :
!     x          : last accepted configuration point
!     frc        : force at x (-grad f(x))
!     f          : f(x)  (energy) 
!     df         : -grad f(x) . sdir * step)   -grad f(x) = frc
!     ------------
!     step       : step along search dir (adjusted) 
!                  "delta_x ~ force * deltim"
!                  initially:  step == initial_step; 
!                  problems:   step -> safe_step
!     sdir       : present  search direction (starting from x) - always downhill
!                  line minimization along (sdir*force < 0) this direction 
!     sdir_prev  : previous search direction
!     hdir       : +/- sdir      : internal to CG directional generator
!                                  reversion of sdir may confuse the CG directional generator
!                                  therefore keep separate tracks
!                                   hdir -> +/- sdir
!                                   sdir -> sdir_prev
!                                   hdir -> hdir_prev
!     hdir_prev  : +/- sdir_prev
!     smove      : fraction of step, x is moved in this iteration
!                  delta_x = sdir * step * smove
!     ------------     
!     x_try      : trial vector, following x
!     frc_try    : force at x_try (-grad f(x_try))
!     f_try      : f(x_try)
!     df_try     : -grad f(x_try) . sdir * step
!--------------------------------------------------------------
!  Meaning of icase/exit: see where ICGRAD header is printed below
!==============================================================!
      implicit none

! input/output variables :
      
      integer, intent(in)   :: nconso            ! print unit
      integer, intent(in)   :: ndim              ! dimension of phase space
      real*8, intent(in)    :: x_external(ndim)  ! config, corresponding to etot, force
      real*8, intent(in)    :: etot              ! energy for x_external
      real*8, intent(in)    :: force(ndim)       ! force  for x_external
      real*8, intent(in)    :: initial_step      ! initially: step == initial_step
      real*8, intent(in)    :: safe_step         ! problems:  step -> safe_step
      real*8, intent(out)   :: dx_external(ndim) ! proposed move, wrt. x_external

! local constants : --- for finetuning of heuristics ---

!!      integer, parameter :: maxtry    = 1    ! max iter history in a searchdir.
      real*8, parameter  :: smovemax  = 4.0  ! trustlength [step] of forecast 
      real*8, parameter  :: sdownhill = 2.0  ! maximal smove for neg. curv. parabola
      real*8, parameter  :: stepmix   = 0.2  ! mixing factor for step adaptation (obsolete)
      real*8, parameter  :: e_goodness_max = 0.5 ! maximum value of DE(fit)/DE(correctorstep) - 1
      real*8, parameter  :: f_goodness_max = 0.5 ! maximum value of  |searchdir_hat*frce|/|frce|
                                                 ! where frce = force(correctorstep)



! numerical tolerences, etc:
           
      real*8, parameter  :: eps_zero           = 1.0d-20
      real*8, parameter  :: eps_small_energy   = 1.0d-3    
      real*8, parameter  :: eps_skip_corrector = 0.2   
      real*8, parameter  :: eps_reject_trial   = 0.03     ! minimal accpptable smove - else new trial
      real*8, parameter  :: eps_xdiff2_tol     = 1.0d-8   ! per degree of freedom

! local variables :
   
      real*8             :: xdiff2, sq_frc_prev, flen, smove, all_fits(2, 0:3)  
      real*8             :: coeff(0:3), flen_try, tmove, e_goodness, f_goodness
      logical            :: ill_conditioned, invalid_fit, accept_linemin
      character*3        :: block_label

! local resident variables :

      logical, save             :: init, x_was_tampered_with
      logical, save             :: pickup_base_at_entry, pickup_trial_at_entry
      logical, save             :: conjugate_next_trial_vector
      logical, save             :: validate_corrector_move
      integer, save             :: inexttask, icase, icount, bad_steps
      integer, save             :: conjug_counter, active_DOFs
      real*8, save              :: gamma, f, df, step, sdir_rotation, DE, f_last
      real*8, allocatable, save :: sdir(:), sdir_prev(:)
      real*8, allocatable, save :: hdir(:), hdir_prev(:)
      real*8, allocatable, save :: x(:), frc(:), frc_prev(:)
      real*8, allocatable, save :: x_try(:), frc_try(:), x_expected(:)
      real*8, save              :: f_try, df_try

! compile time initialization:

      data init /.false./          ! detect first call
      
!================================================
! --- Misc initialisations - only on start ---
!================================================

      if (.not.init) then 

        allocate ( sdir(ndim)     )
        allocate ( sdir_prev(ndim))
        allocate ( hdir(ndim)     )
        allocate ( hdir_prev(ndim))
     
        allocate ( x(ndim)   )
        allocate ( frc(ndim) )
        allocate ( frc_prev(ndim) )

        allocate ( x_try(ndim)  )
        allocate ( frc_try(ndim))
  
        allocate ( x_expected(ndim)    )

        sdir      = force        ! pickup data from dacapo
        sdir_prev = force
        hdir      = sdir        
        hdir_prev = sdir_prev 

        x         = x_external
        frc       = force     
        frc_prev  = force        ! induces SD step (for Polak-Ribiere)
        f         = etot
        step      = initial_step     
         
        active_DOFs = count(dabs(frc) > eps_zero)  ! readjusted during run

! --- log header / format def : ---

        write(nconso,330)

        write(nconso,331) "The conjugate gradient algorithm reports its qualitative"
        write(nconso,331) "using a letter+number code in column: exit              "
        write(nconso,331) "Meaning of letter codes:                                "
        write(nconso,331) "  pc : exited with conjugated predictor step                  "
        write(nconso,331) "  ps : exited with steepest descent predictor step            "
        write(nconso,331) "  pr : exited with predictor step (conjugation seq. was reset)"
        write(nconso,331) "  pn : exited with predictor step in same direction           " 
        write(nconso,331) "  c- : exited with a corrector step                     "
        write(nconso,331) "  **!: coordinates was tampered outside ICGRAD          "       
        write(nconso,331) "Step in configuration space is: step * smove * |sdir|   "
        write(nconso,331) "where sdir is teh conjugated force vector.              "
        write(nconso,331) "Meaning of number codes:                                "   
        write(nconso,331) "  00 : at start                                         " 
        write(nconso,331) "  01 : predictor step, following a corrector move       " 
        write(nconso,331) "  1* : step was a corrector move                        "        
        write(nconso,331) "  12 :   parabolic projected min with accept. stepsize  "  
        write(nconso,331) "  13 :   cubic projected min with accept. stepsize      " 
        write(nconso,331) "  14 :   parabolic limit of cubic projected min with accept. stepsize" 
        write(nconso,331) "  2* : step was constrained by maximal step size        "
        write(nconso,331) "  22 :   parabolic fit, but step constrained            "
        write(nconso,331) "  23 :   cubic fit, but step constrained                "
        write(nconso,331) "  24 :   parabolic limit of cubic projected, but step constrained " 
        write(nconso,331) "  3* : step was too large - new trial step              "
        write(nconso,331) "  32 :   parabolic fit, but found too small smove       "
        write(nconso,331) "  33 :   cubic fit, but found too small smove           "
        write(nconso,331) "  34 :   parabolic limit of cubic projected, but found too small smove" 
        write(nconso,331) "  4* : force/fit was inconsistent - generate extra trial"
        write(nconso,331) "  41 :   fit invalid; present x was preferred over x_try; step adjusted" 
        write(nconso,331) "  42 :   fit invalid; x_try was preferred over x;         step adjusted" 
        write(nconso,331) "  -1 : ..not so good                                    " 

        write(nconso,330)
        write(nconso,331) "Key quantities at initialization:"
        write(nconso,334) "+ initial number of nonzero force components =", active_DOFs

        write(nconso,330)
        write(nconso,332) "accepted", "trial step", "validate corrector", &
                          "step","energy","|force|","energy","|force|","exit","step","smove", &
                          "sdir_rot", "DE(fit/real)-1", "|s*f/f|"
        write(nconso,331)

 330 format("ICGRAD: ", 128("-")) 
 331 format("ICGRAD: ", a)                     
 332 format("ICGRAD: ", 4x,  1x,a27, 1x,a33, 38x,a20,/, &
            "ICGRAD: ", a4,  1x,     a16,  1x,   a14,   &
                             1x,     a16,  1x,   a14,   &
                            1x, a6   ,   1x,  a8 , 1x,  a7,  1x,  a9 , 3x, a9, 1x, a9)      
 333 format("ICGRAD: ", 1x,i4, 1x, f16.8, 1x, f14.6,   &
                               1x, f16.8, 1x, f14.6,   &
                               1x, a4, i2.2, 1x, f8.4, 1x, f7.4, &
                               1x, f9.4, 1x, f9.4, 1x, f9.4) 
 
 334 format("ICGRAD: ", a,i6)

        init                        = .true.
        pickup_base_at_entry        = .false.
        pickup_trial_at_entry       = .false.
        validate_corrector_move     = .false.
        conjugate_next_trial_vector = .true.
        x_was_tampered_with         = .false.

        bad_steps = 0    
        icase     = 0   ! start
        inexttask = 50  ! generate a trial vector next

        icount         = 0   ! set internal iteration counter
        conjug_counter = 0   ! internal conjugation counter (reset when conjug_counter > DOF)
  
#ifdef MODEL
        icount    = 10  ! to make step coincide with fortram IO unit
#endif

      else   

! -----------------------------------------------------------------
! Serve posted pickup requests of (config, energy, force). 
! Some - outgoing - external constraints may interfere 
! with this algorithm, so that they have to be resyncronized
! flag, if the configuration (x_expected) asked the energy for
! was modified
! -----------------------------------------------------------------

         xdiff2 = sum((x_external-x_expected)*(x_external-x_expected))
         if (xdiff2 > ndim*eps_xdiff2_tol) then
             x_was_tampered_with = .true.
         else
             x_was_tampered_with = .false.
         endif

         if (pickup_base_at_entry .and. pickup_trial_at_entry) &
            call abort_calc(nconso,  &
                 "ion_conjgrad: internal error (multiple pickup requests)")

         if (pickup_base_at_entry) then
            x     = x_external
            frc   = force            
            f     = etot
            pickup_base_at_entry = .false. ! request served
         endif

! --- an auxillary pickup_ variable is needed, if multiple
!     trial steps shoud be supported !!

         if (pickup_trial_at_entry) then
            x_try     = x_external
            frc_try   = force            
            f_try     = etot
            pickup_trial_at_entry = .false. ! request served
         endif

         active_DOFs = max(active_DOFs, count(dabs(frc) > eps_zero))  ! readjusted during run

      endif  !    if (.not.init...
     
      icount = icount + 1     ! internal step counter 

! --- brach out according to task ---

      if (inexttask == 50)  goto  50     ! jump to Block  50
      if (inexttask == 100) goto 100     ! jump to Block 100
      if (inexttask == 400) goto 400     ! jump to Block 400


!======================================================================
!   Block 50 : generate a first trial vector 
!              following must be set:    x, f, frc 
!                                        conjugate_next_trial_vector 
!                                        icase
!======================================================================
 50   continue
      block_label      = ""      ! clear in all cases
      block_label(1:1) = "p"     ! block title  
      if (x_was_tampered_with) block_label(3:3) = "!"


! ... serve request: validate_corrector_move before sdir is updated
!     result = (e_goodness, f_goodness) => accept_linemin = true/false

    
      if (validate_corrector_move) then
         accept_linemin = check_corrector_move(sdir,frc,f,f_last,DE, e_goodness, f_goodness, &
                                               e_goodness_max, f_goodness_max)
         validate_corrector_move = .false.     ! request served
      else
         accept_linemin = .true.
         e_goodness     = 0.0d0    ! set "the ideal value"
         f_goodness     = 0.0d0    ! set "the ideal value"
      endif

        x_try  = 0.0    ! clear iteration history for this search direction
        f_try  = 0.0
        df_try = 0.0

!.......Polak-Ribiere conjugation : also OK for first iteration
!       do not conjugate, if we didn't go to a minimum
!       but rather last step was limited by maximum step size
!       Keep sdir independent of hdir, due to possibility of signflip
!       hdir is internal to the Polak-Ribiere conjugation part
 

        if ((conjugate_next_trial_vector).and.(accept_linemin)) then      
          
          conjug_counter = conjug_counter + 1

          sq_frc_prev = sum( frc_prev * frc_prev )
          if (sq_frc_prev > eps_zero**2) then
            gamma     = sum( (frc-frc_prev) * frc ) / sq_frc_prev
          else
            gamma     = 0.d0 ! give up conjugation (=> SD step)
          endif

          if (dabs(gamma) > eps_zero) then
              block_label(2:2) = "c"   ! signal conjugation
          else
              block_label(2:2) = "s"   ! signal an SD step 
          endif  

          if (conjug_counter > active_DOFs) then   ! reset conjugation sequence
            conjug_counter = 0
            gamma          = 0.0d0
            block_label(2:2) = "r"    
          endif
          
          hdir_prev   = hdir           
          hdir        = frc + gamma*hdir
          
          sdir_prev   = sdir 
          sdir        = hdir 

        else          ! ---- maintain present trial direction ----
                                    
          block_label(2:2) = "n"       ! signal no update of trial direction

	endif   

!.......Maintain the convention that the positive direction of sdir
!       is downhill energy wise (compare to strict zero)   

        if (sum(sdir*frc) < 0.0d0)  sdir = -sdir

        sdir_rotation = angle(sdir, sdir_prev)
 
        df        = sum( step * frc * sdir ) ! possibly oevrwritten in block 100
        if (df < 0) call abort_calc(nconso,  &
                         "ion_conjgrad: step is rotten")

        x_try   = x + step * sdir         ! proposed trial move

        flen = sum(frc*frc)
        if (flen > eps_zero**2) then
            flen = sqrt(flen)
        else
            flen = eps_zero       ! very small anyway
        endif
        write(nconso,333) icount, f, flen, 0.0d0, 0.0d0, block_label, icase, step, &
                          1.0, sdir_rotation, e_goodness, f_goodness

        pickup_trial_at_entry = .true.
        dx_external = x_try - x_external 
        x_expected  = x_try       ! get energy for this config
        inexttask   = 100         ! interpret results for trial vector
        return                    ! EXIT_POINT


!==================================================
!   Block 100  : interpret results for trial vector
!                no exit points in this block
!                syncronize sdir (and dependences) with x_try 
!==================================================
 100    continue                                  ! landing point 

! --- syncronize sdir with x_try (in case of external constraints) ---
      
        sdir      = (x_try - x)/step         ! accepted by external constraints

        sdir_rotation = angle(sdir, sdir_prev)

        df        = sum( step * frc     * sdir ) ! reset derivative at x, x_try
        df_try    = sum( step * frc_try * sdir ) ! and multiply by local length scale

! --- sdir should always be downhill - else virtually switch sdir -> -sdir

        if (df > 0.0d0) then                     ! normal case (step always > 0)

           call cubfit_2pt(nconso, f, df, f_try, df_try, sdownhill, smovemax, &
                           icase, smove, DE, coeff, ill_conditioned, invalid_fit, all_fits)

        elseif (df_try < 0.0d0) then  ! coeff/all_fits refers to reverse frame 
                                      ! i.e. origin at x_try, sdir points from x_try to x

           call cubfit_2pt(nconso, f_try, -df_try, f, -df, sdownhill, smovemax, &
                           icase, tmove, DE, coeff, ill_conditioned, invalid_fit, all_fits)

           smove = 1.0d0 - tmove    
           call reverse_sdir(coeff)
           call reverse_sdir(all_fits(1,:))
           call reverse_sdir(all_fits(2,:))

        else

           call abort_calc(nconso,  &
                "ion_conjgrad: fatal external interference happened")

        endif

!----------------------------------------------
! --- Now decide on what to do from aa, bb, cc ---
!----------------------------------------------

! - 1/2 : normal step / down hill accelerated glide -----------


        if (.not.(ill_conditioned.or.invalid_fit)) goto 400


! - 3 : Ending here means there is an inconsistency between force 
!       direction and parabolic/cubic fit. Step migth be too long 
!       or there is noise on the force (possibly too small step)
!       Pick the point (x,x_try) lowest in energy and generate a new 
!       conjugated trial step from there

        bad_steps = bad_steps + 1      ! keep track of how many times we end up here

        if (f > f_try) then            ! prefer trial step
           x        = x_try
           f        = f_try
           frc_prev = frc
           frc      = frc_try 
           icase    = 42          
        else                             ! prefer present step
           icase    = 41                 ! maintain frc_prev, to get conjugation
        endif                            ! i.e. ommit "frc_prev = frc"

        conjugate_next_trial_vector = .true.

! --- it is assumed that the inconsistency may be resolved
!     by adjusting the trial step size. Ansatz: half the 
!     distance between actual step and step = safe_step (supposedly
!     stabilizing move)

      if (bad_steps == 1) step = 0.5d0*(step + safe_step)

      if (bad_steps > 2)  call abort_calc(nconso,  &
                               "out of good ideas to resolve") 

      goto 50                          ! generate a new trial vector

!=====================================================
!   Block 400  : print out status and update according
!                to the smove determined above
!                If the trial step is sufficiently close to the 
!                minimum, we may skip the corrector move and
!                proceed directly to the predictor move
!                icase label is passed from cubfit_2pt()
!=====================================================
  400   continue
        block_label      = ""      ! clear in all cases
        block_label(1:2) = "c-"
        if (x_was_tampered_with) block_label(3:3) = "!"
        
        bad_steps = 0         ! reset bad steps counter at successful moves

        if ((dabs(smove-1.0d0) < eps_skip_corrector).and. &   ! we are essentially
                     (f_try    < f))                 then     ! at the minimum

           x        = x_try
           f        = f_try
           frc_prev = frc
           frc      = frc_try 
           conjugate_next_trial_vector = .true.

           goto 50                 ! proceed directly to a predictor step

        elseif (smove < eps_reject_trial) then

           icase = 40 + mod(icase, 10)                            ! inherit fit info from cubfit
           conjugate_next_trial_vector = .false.
           step                        = step * eps_reject_trial  ! this may cause starvation 

           goto 50                       ! proceed with a very small predictor step

        else   ! ----------  make a corrector move -----------------------

           x          = x + smove * step * sdir  ! corrector move
           frc_prev   = frc
        
           f_last                  = f         ! save it for check_corrector_move()
           validate_corrector_move = .true.    ! post check_corrector_move request (only place)

           flen = sum(frc*frc)
           if (flen > eps_zero**2) then
            flen = sqrt(flen)
           else
            flen = eps_zero       ! very small anyway
           endif

           flen_try = sum(frc_try*frc_try)
           if (flen_try > eps_zero**2) then
             flen_try = sqrt(flen_try)
           else
             flen_try = eps_zero  ! very small anyway
           endif

           write(nconso,333) icount, f, flen, f_try, flen_try, block_label, icase, step, &
                             smove, sdir_rotation, 0.0d0, 0.0d0
 

!.....finalize, and adjust step for next move

           if ((icase < 22).or.(icase > 24)) then
               conjugate_next_trial_vector = .true.   ! got to min; new direction
           else
               conjugate_next_trial_vector = .false.  ! step length constrained cases
           endif
 
           step = set_step(step*smove)
           icase = 1            ! signal new direction for predictor printout

           pickup_base_at_entry = .true.        ! post request
           dx_external = x - x_external
           x_expected  = x                      ! get energy for this config 
           inexttask   = 50                     ! start new linesearch 
        
           return                       ! EXIT_POINT

        endif  ! ... skip_corrector ...

        ! bouncing point for SUBROUTINE ion_conjgrad 

!===============================================================
!###############################################################
      contains                ! local subroutines
!###############################################################
        function set_step(step)
!===============================================================
!       Consider smove set an random set within a 
!       given distribution. Enforce step > 0
!===============================================================
        implicit none
        real*8             :: set_step, step

        integer, parameter :: max_hist = 3
        logical, save      :: init =  .false.
        real*8,  save      :: step_history(max_hist) 

        real*8             :: smin, smax, savg
        integer            :: i
!---------------------------------------------------
        if (.not.init) then           ! cold start

          step_history = dabs(step)   ! set history
          init = .true.
!
        else                          ! hot start

! --- roll history buffer ---

          do i=2,max_hist
            step_history(i-1) = step_history(i)
          enddo
          step_history(max_hist) = dabs(step)  
        
        endif

    
! --- produce estimators ---

        smin = minval(step_history)
        smax = maxval(step_history)
        savg = sum(step_history)/dble(max_hist)
        
! --- propose new step size ---

        set_step = savg
        
!inefficient/not nescessary:  if (set_step > 4*smin) set_step = 4*smin

!===============================================================
        end function set_step

        
        real*8 function angle(x1,x2)
!===============================================================
!       Return angle (in degrees) between x1,x2
!       Return angle = 90 (orthogonal) if either/both x1,x2 = 0
!===============================================================
        implicit none
        real*8, intent(in)  :: x1(:), x2(:)
        real*8              :: sq_x1, sq_x2, x1x2, arg
        real*8, parameter   :: pi = 3.14159265358979323844d0
        real*8, parameter   :: lowlim = -1.0d0 + eps_zero
        real*8, parameter   :: upplim =  1.0d0 - eps_zero
!--------------------------------------------------------------
      
        if (size(x1) /= size(x2)) &
           call abort_calc(nconso,  &
                "ion_conjgrad->angle: misalligned vectors")
        sq_x1 = sum(x1*x1)
        sq_x2 = sum(x2*x2)
        x1x2  = sum(x1*x2)

        if ((sq_x1 > eps_zero**2).and.(sq_x2 > eps_zero**2)) then
           
! --- handle boundaries gracefully ---

           arg   =  x1x2/sqrt(sq_x1)/sqrt(sq_x2)
           if (arg < lowlim) arg =  lowlim
           if (arg > upplim) arg =  upplim   
           angle =  (180.0d0/pi)*acos(arg)

        else
           angle =  90.0d0   ! ill conditioned angle (orthogonal)
        endif
        
!--------------------------------------------------------------
        end function angle


     logical function check_corrector_move(searchdir,frce,f_corr,f_base,DE_proj, &
                                           e_goodness, f_goodness, e_goodness_max, f_goodness_max) 
!=================================================================================
!    Perform check of a proposed move (f_base -> f_corr):
! 
!     1) The energy: e_goodness = DE(fit)/DE(real) - 1    (ideal value == 0)
!                                 where DE(real) = f_corr - f_base
!     2) The force : f_goodness = |searchdir_hat*frce|/|frce|    (ideal value == 0)
!
!    "conclusion": accept_linemin = (dabs(e_goodness) < e_goodness_max) . and.
!                                   (dabs(f_goodness) < f_goodness_max) 
!  
!=================================================================================
     real*8, intent(in)  :: searchdir(:), frce(:), f_corr, f_base, DE_proj
     real*8, intent(out) :: e_goodness,     f_goodness
     real*8, intent(in)  :: e_goodness_max, f_goodness_max
! --- locals ---
     real*8              :: snorm2, fnorm2
! ---------------------------------------------------------------------
     snorm2 = sum(searchdir*searchdir)
     fnorm2 = sum(   frce   *   frce   ) 
     if ((snorm2 > eps_zero).and.(fnorm2 > eps_zero)) then
        f_goodness = sum(searchdir*frce)/sqrt(snorm2)/sqrt(fnorm2)
     else
        f_goodness = 0.0d0          ! f_goodness is illconditioned
     endif

     if (dabs(f_corr - f_base) > eps_zero) then
        e_goodness = DE_proj/(f_corr - f_base) - 1.0d0
     else
        e_goodness = 0.0d0          ! e_goodness is illconditioned
     endif

     check_corrector_move = (dabs(e_goodness) < e_goodness_max) .and. &
                            (dabs(f_goodness) < f_goodness_max) 
! ---------------------------------------------------------------------
     end function check_corrector_move

!###############################################################
!###############################################################
     subroutine cubfit_2pt(nconso, f0, df0, f1, df1, sdownhill, smovemax, &
                           icase, smove, DE, coeff, ill_conditioned, invalid_fit, &
                           all_fits)                 
!======================================================================
!    Cubic fit to function and derivative
!    in x = 0 and x = 1. 
!         f_fit(x)  =  coeff(3) x^3 + coeff(2) x^2  +  coeff(1) x  +  coeff(0)
!    in : f0  = f(0)
!         df0 = -df(0)/dx (NB!)    (df0 > 0) MUST be obeyed
!         f1  = f(1)
!         df1 = -df(1)/dx (NB!)
!
!  icase subset flagged by cubfit_2pt:
! 
!     icase = 12 : parabolic projected min with accept. stepsize
!     icase = 13 : cubic projected min with accept. stepsize
!     icase = 14 : parabolic limit of cubic projected min with accept. stepsize
!     icase = 22 : parabolic fit, but step constrained
!     icase = 23 : cubic fit, but step constrained
!     icase = 24 : parabolic limit of cubic projected, but step constrained 
!
!======================================================================    
        implicit none
        integer, intent(in)   ::  nconso
        real*8,  intent(in)   ::  f0, df0, f1, df1, sdownhill, smovemax
        integer, intent(out)  ::  icase
        real*8,  intent(out)  ::  smove, DE, coeff(0:3)
        logical, intent(out)  ::  ill_conditioned, invalid_fit
        real*8,  intent(out)  ::  all_fits(2, 0:3)               ! for debugging only

        real*8, parameter     ::  accept_cubic_corr = 0.01d0     ! limit for use cubic fit 

        integer  ::  icase2, icase3
        real*8   ::  det, smove2, smove3,  ext_plus, ext_minus
        real*8   ::  DE_2slope, DE_cub, cubic_coor_fac
        real*8   ::  a_2slope(0:3), a_cub(0:3)
        logical  ::  ill_conditioned_par, ill_conditioned_cub
        logical  ::  invalid_fit_par,     invalid_fit_cub  

        logical, save :: init 
        data             init /.false./   ! detect first call
        
        
!--------------------------------------------------------------
!       print header at first invokation
!--------------------------------------------------------------
        if (.not.init) then
           write(nconso, 600) "f0","df0","f1","df1","parfit","cubfit","icase","smove","DE"
           init = .true.
        endif
 600 format("cubfit_2pt:",  a9,   1x,   a12, 1x,   a15, 1x,   a12, 6x, " :: ", &
                            a20, 18x, a18,  24x, a5,   1x, a5,   1x, a5)
 610 format("cubfit_2pt:", f15.8, 1x, f12.6, 1x, f15.8, 1x, f12.6, " :: ", &
                           f11.4,3f9.4, 2x, f11.4,3f9.4, 2x, i2.2, 1x, f7.4, 1x, f7.4)

!--------------------------------------------------------------
! parabolic fit :  trust slopes / symmetric weigthing of f0/f1
! set coefficient vector a_2slope
! a_2slope(2) ~ 0 => linear regime (not flagged as exception)
!--------------------------------------------------------------
        ill_conditioned_par = .false. 
        invalid_fit_par     = .false.       

        icase2 = 0
        a_2slope(0) =  0.5d0*(f0 + f1) + 0.25d0*(df0 + df1) 
        a_2slope(1) = -df0        
        a_2slope(2) =  0.5d0*(df0 - df1)                   
        a_2slope(3) =  0.0d0                  ! cubic term

        if (a_2slope(1) > eps_zero) invalid_fit_par = .true. 

        if (a_2slope(2) > eps_zero) then     ! concave => minimum
           smove2 = -0.5d0 * a_2slope(1) / a_2slope(2)  
           if (smove2 > smovemax) then
             smove2 = smovemax
             icase2 = 22        ! step too long
           else
             icase2 = 12        ! step OK
           endif   
        else                    ! linear/convex regime => downhill
           smove2 = sdownhill    
           icase2 = 22
        endif

        if (smove2 > smovemax) then   ! in case user specified 
            smove2 = smovemax         ! sdownhill > smovemax ...
            icase2 = 22
        endif

        DE_2slope= eval_polynomium(smove2, a_2slope) - a_2slope(0)

!--------------------------------------------------------------
! cubic fit : exact to values and slopes
!    cubic_coor_fac: the relative importance of cubic corrections 
!    f_fit(x) = a_cub(3) x^3 + a_cub(2) x^2 + a_cub(1) x + a_cub(0)
!--------------------------------------------------------------
        ill_conditioned_cub = .false. 
        invalid_fit_cub     = .false.       
         
        a_cub(0)  = f0                              ! cubic coefficients
        a_cub(1)  = -df0
        a_cub(2)  = 3.0*(f1-f0) + df1 + 2.0*df0
        a_cub(3)  = 2.0*(f0-f1) - df1 -     df0
        det    = a_cub(2)**2  -  3.0 * a_cub(1) * a_cub(3)
       
        if (dabs(a_cub(2)) > sqrt(eps_zero)) then
           cubic_coor_fac = dabs(0.75d0 * a_cub(1) * a_cub(3) / a_cub(2)**2)
        else
           cubic_coor_fac = 1.0d0      ! assume large (but factor ill-conditioned)
        endif

        if (a_cub(1) > eps_zero) invalid_fit_cub = .true.  

        icase3 = -1                                ! signal error
        smove3 = -1.0d0                            ! trigger downhill as default
        if (det > eps_zero) then                   ! cubic extrema exists

          if (cubic_coor_fac > accept_cubic_corr) then 
            if (dabs(a_cub(3)) > eps_zero) then

              ext_plus  = (-a_cub(2) + sqrt(det)) / 3.0/a_cub(3)
              ext_minus = (-a_cub(2) - sqrt(det)) / 3.0/a_cub(3)
              if (a_cub(3) > 0) then
                smove3 = max(ext_plus, ext_minus)  ! select rightmost extrema
              else
                smove3 = min(ext_plus, ext_minus)  ! select leftmost extrema
              endif 
              icase3 = 13

            else
              ill_conditioned_cub = .true.         ! unable to use cubic solution
            endif

          else   ! ... cubic_coor_fac is small ----> use parabolic limit

            if (dabs(a_cub(2)) > eps_zero) then    ! is concave
              smove3 = -0.5d0*a_cub(1)/a_cub(2)              ! else linear/convex 
              icase3 = 14                                    ! => downhill step
            else
              smove3 = sdownhill
              icase3 = 24 
            endif
          endif    ! (cubic_coor_fac ... 
        endif      ! (det > ...

        
        if (smove3 < 0.0)  then                    ! covers situations where
            smove3 = sdownhill                     ! minimum < 0 and not
            icase3 = 23                            ! no extrema exist (det < 0)
        endif

        if (smove3 > smovemax) then                ! curb excessive steps
            smove3 = smovemax
            icase3 = 23
        endif

        DE_cub = eval_polynomium(smove3, a_cub) - a_cub(0)

!--------------------------------------------------------------
! Now compare cubic/parabolic fit and select which set to use
! principle : use cubic solution, if it is flagged OK - else
!             and for small energy changes (DE_2slope) use parabolic fit  
!--------------------------------------------------------------

! --- set global conditions ---
        
        ill_conditioned = ill_conditioned_par .and. ill_conditioned_cub
        invalid_fit     = invalid_fit_par     .and. invalid_fit_cub    

        all_fits(1,:) = a_2slope    ! for debugging only
        all_fits(2,:) = a_cub       ! for debugging only


        if ((icase3 == 13).and.(DE_2slope < -ndim*eps_small_energy) &
                          .and.(.not.ill_conditioned_cub)  ) then
           coeff = a_cub
           icase = icase3
           smove = smove3
           DE    = DE_cub
        elseif  (.not.ill_conditioned_par) then   ! rely on parabolic result
           coeff = a_2slope
           icase = icase2
           smove = smove2
           DE    = DE_2slope
        else                                      ! render outvar defined
           coeff = 0.0d0                          ! although useless
           icase = -1
           smove = 0.0d0
           DE    = 0
        endif

        write(nconso, 610) f0, df0, f1, df1, all_fits(1,:), all_fits(2,:), icase, smove, DE 

        return
        end subroutine cubfit_2pt 
!================================================ 
    function eval_polynomium(x,a)
!--------------------------------------------------------------
!   eval_polynomium = sum[n=0,size(a-1), a(n) * x^n]
!   internal: requires an interface for coefficient vector a
!--------------------------------------------------------------
    real*8            :: eval_polynomium
    real*8,intent(in) :: x, a(0:)   ! apparently OK declaration
    integer           :: i
!--------------------------------------------------------------
    eval_polynomium = a(0)       !  0**0 is an exception
    do i = 1, size(a)-1
      eval_polynomium = eval_polynomium + a(i) * x**i 
    enddo
!---------------------------------------
    end function eval_polynomium
  
!================================================ 
    subroutine reverse_sdir(coef)
! ---------------------------------------------------------------------------
!  Expand[a3 (1-t)^3 + a2(1-t)^2 + a1 (1-t) + a0]    
!                                                         2         2       3
!      = a0 + a1 + a2 + a3 - a1 t - 2 a2 t - 3 a3 t + a2 t  + 3 a3 t  - a3 t       
! ---------------------------------------------------------------------------
   real*8,intent(inout) :: coef(0:3)   
   real*8               :: a(0:3)
   a       =  coef
   coef(0) =   a(0) +       a(1) +       a(2) + a(3)
   coef(1) = - a(1) - 2.d0* a(2) - 3.d0* a(3)
   coef(2) =   a(2) + 3.d0* a(3) 
   coef(3) = - a(3) 
! ---------------------------------------------------------------------------
    end subroutine reverse_sdir

    end  subroutine ion_conjgrad  
