!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
!                                                                      C
!     This module takes care of the electro-static decoupling between  C 
!     the periodic images.                                             C
!     This is done using a fit to atom-centered Gaussians which re-    C
!     produces the long-range electrostatic potential of the original  C 
!     density.                                                         C
!     This can be used to do cluster calculations witha a plane-wave-  C
!     expanded basis.                                                  C
!     Jan Rossmeisl  17/11-2000                                        C
!     Based on (P.E. Blochl J. Chem. Phys. 103, 17 1995)               C
!                                                                      C
!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      module blochlmodule

      use Madelung_module
#include "definitions.h"

!     Saved initilization of the g_vectors

      logical, save, private :: init = .true. ! initilize only once
      logical, save, private :: lstop = .false. ! true if no decoupling
      logical, save, private :: onfit = .false.
      logical, save, private :: ldam = .true. ! if false damconv lt dammax
      real*8,  save, private :: dammax,gcutoff,width
      integer, save, private :: num
      integer, save, private :: ngmax ! The number of used g-vec 
      real*8,  save, private, pointer :: g_vec(:,:) ! g-vector(ngmax,3)
      real*8,  save, private, pointer :: sqgnorm(:) ! ngmax

!     The cordinates on the fft-grid 
      integer,    save, private, pointer :: g_cor(:,:) ! (ngmax,3)
      complex*16, save, private, pointer :: a(:,:),ainv(:,:)

      real*8,     save, private, pointer :: c_ainv(:)
      real*8,     save, private :: c_ainv_c
      real*8,     save, private, allocatable :: w(:)

      contains
      subroutine blochl(volume,nions,dirc,recc,rdens,posion,&
           ngx,ngy,ngz,nspec,nionsp,nelec,nconso,&
           lpctx,lpcty,lpctz,icharg,nplwv,nspin,&
           damconv,ffield,deltapot,deltae,timer,lupdio,lmastr) 

      use basicdata
      implicit none

!     input

      logical, intent(in) :: lupdio
      integer, intent(in) :: ngx,ngy,ngz,nions
      integer, intent(in) :: nspec,nionsp(:),nconso ! nionsp(nspec) 
      real*8,  intent(in) :: nelec
      integer, intent(in) :: lpctx(:),lpcty(:),lpctz(:) ! lpctx(ngx)
      integer, intent(in) :: icharg(:),nspin,nplwv ! nspec
      real*8              :: rdens(ngx,ngy,ngz,nspin),volume ! ngx,ngy,ngz
      real*8, intent(in)  :: recc(:,:) ! 3x3
      real*8, intent(in)  :: dirc(:,:) ! 3x3
      real*8, intent(in)  :: posion(:,:,:) !  3,nions,nspec
      real*8, intent(in)  :: damconv
      real*8  ffield(3,nions,nspec) 
      logical*4,intent(in) :: lmastr

!     locals

      integer i,j,k,ng,nn
      integer ngaus,icharge(nions)
      real*8 nb
      real*8 rions(3,nions),rion(3)
      real*8 enpot
      real*8 vmadl(nions,nions)
      real*8 center(3)
!      real*8 gcutoff            ! the cutoff of the squared norm of g-vector
      real*8 xrdens(ngx)
      real*8 fev(3,nions)
      complex*16 workdens(ngx,ngy,ngz)

!     TIMERS
      real*8 timer(*)
#include "etime.h"

!     allocatables & pointer
     
      real*8, allocatable :: my(:) ! ngaus
      real*8, allocatable :: rcutoff(:),cgaus(:,:) ! ngaus
      complex*16, allocatable :: gdens(:) ! ngmax
      complex*16, allocatable :: ggaus(:,:) ! ngmax x n
      complex*16, allocatable :: b(:) ! n x n

!     results of this module NB! not all is used as output

      real*8, allocatable :: q(:) ! the weight of the gaussians
      real*8 deltaf(3,nions)    ! the correction to the force
      real*8 ew_energy
      real*8, intent(out) :: deltae ! the energy correction
      real*8 ew_force(3,nions)  
      complex*16 potcor(ngx,ngy,ngz) !the correction to the potential
      complex*16, intent(out) :: deltapot(nplwv) ! ntot
      real*8 ptch(nions)        ! the partial charges of the ions
      real*8 f_gaus(3,nions)    ! a "pulay-force" 
      deltapot(:) = (0.d0,0.d0)
      deltae = 0.d0

!     =================================
!     IF NO DECOUPLING lstop = .true.
!     =================================

      
      if(lstop)return

!     ================================================================
!     For first run (init= true) read the input parameters form netcdf
!     The g-vectors with a norm less than g-cutoff
!     ================================================================

      if(init)then
         call read_input(num,gcutoff,dammax,width,lstop,onfit)
         if(lstop)return
         call g_vectors(gcutoff,ngx,ngy,ngz,ngmax,&
              lpctx,lpcty,lpctz,recc,g_vec,g_cor,sqgnorm,nconso)
         allocate(w(ngmax))
         do ng = 2,ngmax
            w(ng) = exp(-sqgnorm(ng)/gcutoff)*& !the weight function 
                 4*pi*(sqgnorm(ng)-gcutoff)**2/&
                 (sqgnorm(ng)*gcutoff)
         enddo 
      endif
     
      ngaus = num*nions           ! the number of gaussians

      allocate (rcutoff(ngaus))
      allocate (b(ngaus),q(ngaus),cgaus(3,ngaus))

!     ========================================================
!     Rewrite the posistions and charges of the ions in arrays 
!     over nions
!     ========================================================

      nn = 0
      do i = 1,nspec
         do j = 1,nionsp(i)
            nn = 1 + nn
            icharge(nn)=icharg(i)
            call xyzpos(posion(1,j,i),dirc,rion)
            rions(:,nn) = rion(:)
         enddo
      enddo 
      
!     ==========================================
!     Reset the correction to the forces ffield
!     ==========================================
      
      ffield(:,:,:) = 0.d0



      allocate(ggaus(ngmax,ngaus))
      allocate(gdens(ngmax))

!     =============================================
!     Setup the gaussian basis in reciprocal space
!     =============================================

      call gaus_basis_set(nplwv,rions,g_vec,ngmax,&
           num,width,ngaus,nions,rcutoff,cgaus,ggaus,&
           nconso,sqgnorm)

!     ==============================================
!     The fourier transform of the electron density
!     ==============================================

      if(nspin .eq. 2)then
         workdens(:,:,:) = (rdens(:,:,:,1)+rdens(:,:,:,2))&
              /dble(ngx*ngy*ngz) ! copy and norm density
      else
         workdens(:,:,:)=rdens(:,:,:,1)/dble(ngx*ngy*ngz)
      endif

      call fft3d(workdens,ngx,ngy,ngz,-1)
      do ng = 1,ngmax
         gdens(ng) =  workdens(g_cor(ng,1),g_cor(ng,2),g_cor(ng,3))
      enddo

!     ============================================================
!     The fit of the gaussian weights to the electron density, q(i)
!     The model electron density is returned 
!     ============================================================

      call model_density(nions,nelec,icharge,ggaus,gdens,&
           sqgnorm,ngaus,num,ngmax,gcutoff,w,a,ainv,b,c_ainv,&
           c_ainv_c,q,ptch,nconso,lupdio,init)
      

      
      init = .false.
      if(onfit)then
         write(nconso,*)'GAUSSIAN-PARTIAL-CHARGE(-e)'
         do i = 1,nions
            write(nconso,*)'PARTIAL-CH',i,ptch(i)
         enddo
         if (lmastr) then 
           call write_ptch(ptch,nions)
         endif
         deallocate(rcutoff,cgaus,ggaus,gdens,b,q)
         return
      endif 

!     ==========================================================
!     The madelung matrix only dependent on the positions of the 
!     ions is calculated (vmadl)    
!     ==========================================================

      
      if (lmastr) then 
         call write_ptch(ptch,nions)
      endif

      call ewald_madl(posion,volume,nions,recc,dirc,&
           nspec,nionsp,ngx,ngy,ngz,lpctx,lpcty,lpctz,&
           vmadl,ew_force,ew_energy,nconso,partial=ptch)

      deltae = -ew_energy

      deltaf(:,:) = ew_force(:,:)


!     ==========================================================
!     The correction to the energy (deltae) and force due to the 
!     decoupling, also the partial charges to the ions is calculated
!     here (ptch).
!     ==========================================================

      call energycor(ptch,q,num,nions,ngaus,rcutoff,&
           volume,dirc,recc,posion,cgaus,nspec,nionsp,icharge,&
           rions,nb,deltae,deltaf,nconso)


!     ==========================================================
!     The correction to the effective potential (potcor)
!     ==========================================================

      allocate(my(ngaus))

      call pot_cor(vmadl,nions,num,ngaus,ngmax,nplwv,&
           ainv,c_ainv,c_ainv_c,ggaus,g_vec,g_cor,w,ptch,&
           icharge,nb,rcutoff,q,ngx,ngy,ngz,rions,potcor,&
           deltapot,fev,my,nconso)   


!      deltapot(:) = (0.d0,0.d0)
!     ==========================================================
!     An additional correction to the forces on the ions (f_gaus)
!     This term originates from the non-completeness of the 
!     Gaussian basis, it is analogous to the Pulay-forces
!     ==========================================================
      call gaus_for_cor(ainv,a,w,c_ainv_c,b,rions,g_vec&
           ,ggaus,gdens,q,num,ngmax,nions,ngaus,my,f_gaus&
           ,nconso)
      
      do i = 1,nions
         write(nconso,*)'for_pulay',i,f_gaus(:,i)
      enddo


      nn = 0
      do i = 1,nspec
         do j = 1,nionsp(i)
            nn = 1 + nn
            ffield(:,j,i)=-deltaf(:,nn)+f_gaus(:,nn)
!     &           +fev(:,nn))
         enddo
      enddo


      
!     =================================================
!     add the correction to the effective one electron 
!     potential
!     =================================================

      call sumpot(enpot,nplwv,rdens,deltapot,1)
      write(nconso,*)'ene_pre',deltae
      deltae = deltae-enpot
!      deltae=.5*(eneion-enpot)
      write(nconso,*)'energy_dueto_pot_e',enpot
      write(nconso,*)'ene_total',deltae
      call uttime(time)
      timer(TFORCOR)=timer(TFORCOR)+time(1)

      deallocate(my,b,cgaus,rcutoff,gdens,ggaus,q)    

      return      
      end subroutine blochl


!     ===========================================================
!     ===========================================================
!     ===========================================================
!     SUBROUTINES SUBROUTINES SUBROUTINES SUBROUTINES SUBROUTINES
!     ===========================================================
!     ===========================================================
!     ===========================================================

!     ======================================================
!     Read the input from python script
!     ======================================================

      subroutine read_input(num,gcutoff,dammax,width,lstop,onfit)
      use netcdfinterface
      use run_context
      use basicdata
      
      implicit none
      logical lstop,onfit
      integer status,ncid,nn
      integer, intent(out) :: num
      real*8 dc,dammax,gcutoff,gg,ww,width
      character cdummy

!     -----------------------------------------------------
!     Open netCDF dataset
!     -----------------------------------------------------

!     parallel send should be implemented

      status =  nf_open(netCDF_input_filename, NF_NOWRITE, ncid)  
!     if (status /= nf_noerr) call abort_calc(nconso, &
!                "blochl read input-> nf_open : error opening nc-file")
      if (status /= nf_noerr) then 
           write(nconso,*) &
           "blochl read input-> nf_open : error opening nc-file" 
         lstop = .true. 
         return
      endif

      status = nfget(ncid, "Decoupling", cdummy)  
      if (status == nfif_OK) then


         status = nfget(ncid, "Decoupling%NumberOfGaussians", nn)
         if (status == nfif_OK) then 
            num = nn
         else
            num = 3             !default number of gaussian pr. atom
         endif

         status = nfget(ncid, "Decoupling%WidthOfGaussian", ww)
         if (status == nfif_OK) then 
            width = ww
         else
            width = 0.35             !default number of gaussian pr. atom
         endif

         status = nfget(ncid, "Decoupling%ECutoff", gg)
         if (status == nfif_OK) then 
            gcutoff = gg/hsqdtm 
         else
            gcutoff = 100/hsqdtm !default squaregcutoff 100eV
         endif

         status = nfget(ncid, "Decoupling%DensityConv", dc)
         if (status == nfif_OK) then 
            dammax = dc
         else
            dammax = 1000.0     !default maximum value of DAM before decoupling
         endif
      else
         status = nfget(ncid, "CalculatePartialCharges", cdummy)  
         if (status == nfif_OK) then
            onfit = .true.      !no-decoupling calaulate partial charges
            
            status = nfget(ncid, "CalculatePartialCharges%NumberOfGaussians", nn)
            if (status == nfif_OK) then 
               num = nn
            else
               num = 3          !default number of gaussian pr. atom
            endif
            
            status = nfget(ncid, "CalculatePartialCharges%WidthOfGaussian", ww)
            if (status == nfif_OK) then 
               width = ww
            else
               width = 0.35     !default number of gaussian pr. atom
            endif
            
            status = nfget(ncid, "CalculatePartialCharges%ECutoff", gg)
            if (status == nfif_OK) then 
               gcutoff = gg/hsqdtm 
            else
               gcutoff = 100/hsqdtm !default squaregcutoff 100eV
            endif
            
            status = nfget(ncid, "CalculatePartialCharges%DensityConv", dc)
            if (status == nfif_OK) then 
               dammax = dc
            else
               dammax = 1000.0  !default maximum value of DAM before decoupling
            endif
         else
            lstop = .true.  
         endif
      endif

      status=nf_close(ncid)
      end subroutine read_input

!     ======================================================
!     The Fourier transform of the gaussian basis calculated
!     analytic, the Gaussians is only setup in reciprocal space
!     ======================================================

      subroutine gaus_basis_set(nplwv,rions,g_vec,&
           ngmax,num,width,ngaus,nions,rcutoff,cgaus,ggaus,&
           nconso,sqgnorm)

      implicit none

!     input 
      integer, intent(in) :: ngmax,ngaus
      integer, intent(in) :: nconso,nplwv,num,nions
      real*8, intent(in) :: rions(:,:),g_vec(:,:)
      real*8,intent(in) :: sqgnorm(:),width

!     locals
      integer i,n,j,ng
      real*8 gdotri,cgaus(3,ngaus),norm
      complex*16 com

!     output
      real*8, intent(out) :: rcutoff(ngaus)
      complex*16, intent(out) :: ggaus(ngmax,ngaus)

      
      n = 0
      do i = 1,nions
         do j = 1,num
            n = 1 + n
            rcutoff(n)  = width * 1.5d0**(j-1) !the cutoffradius for the gaussians
            cgaus(:,n) = rions(:,i) !center of the gaussians
         enddo
      enddo

!     ggaus(ng,ngaus) is the gaussian basis i g-space
 
      norm=1.d0
      do i=1,ngaus  
         ggaus(1,i)=norm
         do ng =2,ngmax       
            gdotri=dot_product(g_vec(ng,:),cgaus(:,i))
            com=(0.d0,-1.d0)*gdotri&
                 -1.d0/4.d0*sqgnorm(ng)*(rcutoff(i))**2
            ggaus(ng,i) = norm*exp(com)
         enddo
      enddo
   
      

      return
      end subroutine gaus_basis_set

      
!     =======================================================
!     This subroutine calculates the g-vectors and saves the 
!     g-vectors with a sqare-norm less than g_cut_off,
!     This is only done once for one job step
!     =======================================================
      
      subroutine g_vectors(gcutoff,ngx,ngy,ngz,ngmax,&
           lpctx,lpcty,lpctz,recc,g_vec,g_cor,sqgnorm,nconso)
  
      implicit none
      
!     input 
      integer ngx,ngy,ngz,lpctx(ngx),lpcty(ngy),lpctz(ngz)
      integer nconso
      real*8  gcutoff,recc(3,3)
      
!     locals
      integer i,j,k,ng,nplwv
      integer, allocatable :: corwork(:,:)
      real*8, allocatable :: gwork(:,:) ! nplwv x 3
      real*8, allocatable :: gn(:) ! nplwv
      real*8 zero,gnorm,g(3,3),sqg(3)

!     output
      integer ngmax
      integer, pointer :: g_cor(:,:)
      real*8, pointer :: g_vec(:,:) ! ngmax x 3
      real*8, pointer :: sqgnorm(:) ! ngmax


      nplwv = ngx*ngy*ngz

      allocate (gwork(nplwv,3))
      allocate (gn(nplwv))
      allocate(corwork(nplwv,3))
      
      zero = 1E-15
      ng = 1
      do i = 1,ngx
         g(1,:) = recc(1,:)*lpctx(i)
         do j = 1,ngy
            g(2,:) = recc(2,:)*lpcty(j)
            do k = 1,ngz
               g(3,:) = recc(3,:)*lpctz(k)
               sqg(1)=(sum(g(:,1))**2)
               sqg(2)=(sum(g(:,2))**2)
               sqg(3)=(sum(g(:,3))**2)
               gnorm = sum(sqg(:))
               if (gnorm .gt. gcutoff)cycle
               if (gnorm .lt. zero)then
                  gwork(1,1) = sum(g(:,1)) ! the gvector with norm=0
                  gwork(1,2) = sum(g(:,2))
                  gwork(1,3) = sum(g(:,3))
                  corwork(1,1)=i
                  corwork(1,2)=j
                  corwork(1,3)=k
               else  
                  ng = ng + 1
                  gn(ng) = gnorm ! the norm of the reci-vectors less than gcutoff
                  gwork(ng,1) = sum(g(:,1)) ! the g vector of relavance
                  gwork(ng,2) = sum(g(:,2))
                  gwork(ng,3) = sum(g(:,3))
                  corwork(ng,1)=i ! the points of relavance
                  corwork(ng,2)=j
                  corwork(ng,3)=k
               endif 
            enddo
         enddo
      enddo
      ngmax = ng
      

      allocate(g_vec(ngmax,3),g_cor(ngmax,3))
      allocate(sqgnorm(ngmax))

!     ---------------------------------------------------
!     array of only the relevant g-vectors 
!     --------------------------------------------------

      do ng = 1,ngmax
         g_vec(ng,:) = gwork(ng,:)
         g_cor(ng,:) = corwork(ng,:) ! array of points only of the needed size
         if(ng == 1)cycle
         sqgnorm(ng) = gn(ng)
      enddo


      deallocate( gwork,gn,corwork )
      return
      end subroutine g_vectors      

!     ============================================================
!     Fit the weight of the gaussians (q(ngaus)) to the electron density
!     using the scheme made by P.E. Blochl (1995) and make a model 
!     electron density
!     ============================================================
      
      subroutine model_density(nions,nelec,icharge,ggaus,gdens,&
           sqgnorm,ngaus,num,ngmax,gcutoff,w,a,ainv,b,c_ainv,&
           c_ainv_c,q,ptch,nconso,lupdio,init)

      implicit none

!     input
      logical, intent(in) :: init
      logical, intent(in) :: lupdio
      integer, intent(in) :: ngaus,ngmax,num,nions
      real*8,  intent(in) :: nelec
      integer, intent(in) :: nconso,icharge(:)
      real*8, intent(in) :: gcutoff,sqgnorm(:),w(:)
      complex*16, intent(in) :: ggaus(:,:),gdens(:)

!     locals
      integer i,j,k,ng,control
      real*8 c_ainv_b,elch(nions)
      integer yt(ngaus)
      complex*16 e(ngaus,ngaus),lamda

!     output
      complex*16, pointer     :: a(:,:)
      complex*16, intent(out) :: b(ngaus)
      complex*16, pointer     :: ainv(:,:)
      real*8, intent(out)     :: q(ngaus)
      real*8, pointer         :: c_ainv(:)
      real*8                     c_ainv_c
      real*8, intent(out)     :: ptch(nions) ! the partial charges of the ions


!     ============================
!     Weight the small  g-vectors
!     ============================

!     =============================================================
!     calculate the b(i) vector eq. 16(a)
!     calculate the a(i,j) matrix and the inverse of a that is ainv
!     eq 16(b)
!     =============================================================

      b(:) = 0.d0
      do i = 1,ngaus
         do ng=2,ngmax
            b(i)=w(ng)*real(conjg(gdens(ng))*ggaus(ng,i))+b(i)
         enddo
      enddo

 
      if(lupdio .or. init)then
         if(init)then
            allocate(c_ainv(ngaus))
            allocate(a(ngaus,ngaus),ainv(ngaus,ngaus))
         endif
         e(:,:) = dcmplx(0.d0,0.d0)
         a(:,:) = dcmplx(0.d0,0.d0)
         do i = 1,ngaus
            e(i,i)=dcmplx(1.d0,0.d0)
            do j = 1,ngaus
               do ng=2,ngmax
                  a(i,j)=w(ng)*conjg(ggaus(ng,i))*ggaus(ng,j)&
                       +a(i,j)
               enddo
            enddo
         enddo

!     ===================================================
!     Find the inverse of a(i,j)
!     the output e(i,j) is the inverse of a(i,j)
!     ===================================================


!     yt is a dummie matrix only needed to call zgesv

         call zgesv(ngaus,ngaus,a,ngaus,yt,e,ngaus,control) 
     

!     ==================================
!     control of the inversion of a(i,j)
!     ==================================


         if(control .ne. 0)then
            write(*,*)'control BLOCHL:ainv wrong',control
            call clexit(nconso)
         endif


         ainv(:,:) = e(:,:)

!     ==================================================
!     Calculate the lagrange multipler lamda in eq. (18)
!     ================================================== 

         c_ainv_c = 0.d0
!     c_ainv(i)=c(i)*ainv(i,j) sum over the rows
         c_ainv(:)=0.d0
         do j = 1,ngaus
            c_ainv(j)=real(sum(ainv(:,j))) 
         enddo
         
      
!     The norminator c_ainv_c= c(i)*ainv(i,j)*c(j) sum over the collums

         c_ainv_c=sum(c_ainv(:))

      endif 


!     c_ainv_b = c(i)*ainv(i,j)*b(j)

      c_ainv_b = 0.d0
      c_ainv_b=dot_product(c_ainv(:),b(:)) 
      
      
      lamda = (c_ainv_b - nelec)/c_ainv_c
      

!     =============================================================
!     The weight of each gaussian is calculated eq. 18, q(i) vector
!     =============================================================
         
      q(:) = 0.d0
      do i = 1,ngaus
         q(i) = sum(ainv(i,:)*(b(:)-lamda))
!         write(nconso,*)'q_gaus_weight',i,q(i)
      enddo

      i = 0
      do j = 1,nions
         elch(j) = 0
         do k = 1,num
            i=i+1
            elch(j) = q(i)+elch(j)
         enddo
         ptch(j) = elch(j) - icharge(j) ! point charge
      enddo


      return
      end subroutine model_density

!     ============================================================
!     Here is the correction to the energy due to the elctrostatic 
!     decoupling calculated
!     ============================================================

       subroutine energycor(ptch,q,num,nions,ngaus,rcutoff,&
           volume,dirc,recc,posion,cgaus,nspec,nionsp,icharge,&
           rions,nb,deltae,deltaf,nconso)

      use basicdata
      implicit none

!     input
      integer, intent(in) :: num,nions,ngaus,nconso
      integer, intent(in) :: nionsp(:),nspec,icharge(:)
      real*8, intent(in) :: q(:),rcutoff(:),volume,rions(:,:)
      real*8, intent(in) :: posion(:,:,:),dirc(:,:),recc(:,:)      
      real*8, intent(in) ::cgaus(:,:),ptch(:)

!     locals 
      integer k,i,j
      real*8 en2,en3,en4,c,for2(3,nions,nspec)
      real*8 sum4,elch(nions)
      real*8 disr,worken3,for3(3,nions)

!     output
      real*8  deltae,nb
      real*8  deltaf(3,nions)

      c = edeps/(4.0d0*pi)             ! 1/(4pi*e0) units are A,eV,e  

!     ==========================================================
!     The point-charge on the atoms (ptch) calculated as sum of the 
!     weight of the gaussians minus the ionic charge
!     The compensating constant charge background (nb). nb = 0 if
!     the supercell is charge neutral

!     ===========================================================
!     The enternal interaction energy between the partial charges
!     in one supercell e3 eq. 8
!     ===========================================================
      nb = 0.d0
      for3(:,:) = 0.d0     
      worken3 = 0.d0
      write(nconso,*)'GAUSSIAN-PARTIAL CHARGE(-e)'
      do i=1,nions
         nb = nb + ptch(i)/volume
         do j=1,nions
            if(i == j)cycle ! if same ion
            disr = sqrt(sum((rions(:,i)-rions(:,j))**2))
            worken3 = c*ptch(i)*ptch(j)/disr + worken3
            for3(:,i)=c*ptch(i)*ptch(j)*&
                 (rions(:,i)-rions(:,j))/(disr)**3+for3(:,i)
         enddo
         write(nconso,*) 'Partial_charge',i,ptch(i)
      enddo
      en3 = 0.5d0*worken3

!     =======================================================
!     The partial charge density interactions over all space 
!     that includes interactions between different supercells E2
!     =======================================================
!     =======================================================
!     The difference in the interaction energy between the 
!     partial charges and the real charge density and the 
!     uniform background charge E4  eq. 11
!     Notice that this term is different from eq. 12
!     because the partial charge is used in this code
!     =======================================================

      en4 = 0.d0
      en4=c*nb*pi*sum(q(:)*(rcutoff(:)**2)) 
!     ======================================================
!     Total energy correction en2+en3+en4
!     ======================================================
      write(nconso,*)'energy_correction2',deltae

      deltae =deltae+en3-en4

!     ======================================================
!     The force correction without the Pulay-forces
!     ======================================================

      do k=1,nions
!         write(nconso,*) 'force_corrrection2',k,deltaf(:,k)

         deltaf(:,k) = deltaf(:,k)-for3(:,k)
         write(nconso,*) 'force_decoupling',k,deltaf(:,k)
!         write(nconso,*) 'force_corrrection3',k,for3(:,k)
      enddo
     
!      write(nconso,*) 'force_corrrection',deltaf(:,:) 
!      write(nconso,*) 'energy_correction3',en3
!      write(nconso,*) 'en_cor4del',en4,deltae


      return
      end subroutine energycor

!     =====================================================
!     The corection to the Hellmann-Feymann force due to the 
!     not compleed basis set of Gaussians: Blochl (22)-(23)
!     sum over i of -dq(i)/dR*my(i)
!     =====================================================
      subroutine gaus_for_cor(ainv,a,w,c_ainv_c,b,rions,gvec&
           ,ggaus,gdens,q,num,ngmax,nions,ngaus,my,f_gaus&
           ,nconso)

      implicit none
      
     
!     input           
      integer, intent(in) :: num,ngaus,nions,ngmax,nconso
      real*8, intent(in) :: gvec(:,:)      
      real*8, intent(in) :: rions(:,:)
      real*8, intent(in) :: w(:),q(:),my(:),c_ainv_c
      complex*16, intent(in) :: ainv(:,:),a(:,:)
      complex*16, intent(in) :: ggaus(:,:),b(:),gdens(:)

!     locals
      integer h,i,j,k,ng,kn,jn,l,n
      real*8 q0(ngaus),db(3,ngaus),dq0(3,ngaus,nions)
      real*8 distr1r2(3),gdotr,dq(3,ngaus,nions)
      complex*16 da(3,ngaus,ngaus)

!     output
      real*8, intent(out) :: f_gaus(3,nions)

!     ===================================================================
!     Calculate dA(i,j)/dR and db(i)/dR NOTE that n(G)/R_i =0 for the self
!     consistent density
!     ===================================================================

      
      da(:,:,:) = dcmplx(0.d0,0.d0)
      do i = 1,ngaus
         db(:,i) = 0.d0
         do ng = 2,ngmax
            db(:,i) = db(:,i)+w(ng)*real(conjg(gdens(ng))&
                    *(0.d0,1.d0)*gvec(ng,:)*ggaus(ng,i)) ! NB NB
            do j = 1,ngaus 
               da(:,i,j)=da(:,i,j)+w(ng)*conjg(ggaus(ng,i))&
                    *ggaus(ng,j)*(0.d0,1.d0)*gvec(ng,:)
            enddo
         enddo
         q0(i) = sum(ainv(i,:)*b(:))
      enddo

!     ====================================================================   
!     (23 a) dq0(i) the derivative of q(i) without the constraint of charge
!     conservation
!     ====================================================================

      n=0
      dq0(:,:,:)= 0.d0
      do l=1,nions
         do k = 1,num
            n=n+1
            do i = 1,ngaus
               do h=1,3
                  dq0(h,i,l)=-ainv(i,n)*(db(h,n)+sum(da(h,n,:)*&
                       q0(:)))+sum(ainv(i,:)*da(h,:,n)*q0(n))&
                       +dq0(h,i,l)
               enddo
            enddo
         enddo
      enddo


!     =================================================================
!     (23 b)dq(i) the derivative of q(i) with the constraint of charge
!     conservation and f_gaus is the "pulay" correction to the force 
!     =================================================================

      f_gaus(:,:)=0.d0
      dq(:,:,:)=0.d0
      do l=1,nions
         do i=1,ngaus
            do j=1,ngaus
               do k=1,ngaus
                  dq(:,i,l)=-ainv(i,j)/c_ainv_c*dq0(:,k,l)+dq(:,i,l)
               enddo
            enddo
            dq(:,i,l)=dq(:,i,l)+dq0(:,i,l)
            f_gaus(:,l)=my(i)*dq(:,i,l)+f_gaus(:,l)
         enddo
      enddo
      

      return
      end subroutine gaus_for_cor


!     ==========================================
!     In this routine the correction to the electron potential
!     origin from the electrostatic decoupling is calculated 
!     the equations are taken from Blochl (19)-(21)
!     Notice that here all is calculated in terms of the 
!     partial charge
!     ==========================================

      subroutine pot_cor(vmadl,nions,num,ngaus,ngmax,ntot,&
           ainv,c_ainv,c_ainv_c,ggaus,g_vec,g_cor,w,ptch,icharge&
           ,nb,rcutoff,q,ngx,ngy,ngz,rions,potcor,deltapot,&
           fev,my,nconso)

      use basicdata
      implicit none

     
!     input
      integer, intent(in) :: nions,num,ngaus,ngx,ngy,ngz
      integer, intent(in) :: ngmax,nconso,g_cor(:,:),icharge(:)
      real*8, intent(in) :: nb,g_vec(:,:),c_ainv_c,w(:)
      real*8, intent(in) :: ptch(:),q(:),rcutoff(:),c_ainv(:)
      real*8, intent(in) :: vmadl(:,:),rions(:,:)
      complex*16, intent(in) :: ainv(:,:),ggaus(:,:)

!     locals
      integer i,j,ng,ntot,n,t,k
      real*8 my(ngaus),c,deltany0
      real*8 m(nions,nions),factor,gdotri
      real*8 disr,bc,myi
      complex*16 deltapot(ntot)
      complex*16 gainv,com
      complex*16 deltany(ngmax)
      real*8 fev(3,nions)

!     output 
      complex*16, intent(out) :: potcor(ngx,ngy,ngz)


 
      c = edeps/(4.0d0*pi)             ! 1/(4pi*e0) units are A,eV,e  

!     ===================================================
!     the "my" is the derivative of the energy correction
!     with respect to the partial charges (20)
!     ===================================================


!     The Madelung matrix  

      do i=1,nions
         do j=1,nions
            m(j,i) = vmadl(j,i)
            if(i == j)cycle
            disr = sqrt(sum((rions(:,i)-rions(:,j))**2))
            m(j,i) = m(j,i)-1/disr           
         enddo
      enddo
      

      factor = pi*nb     
      t = 0
      do i = 1,nions
         myi = sum(m(i,:)*ptch(:))
         do j = 1,num
            t=t+1
            bc=factor*(rcutoff(t)**2)
            my(t)=c*(bc+myi)
         enddo
      enddo
           

!     =========================================
!     "deltany0" is the correction to the electron 
!     potential in G=0 (21)
!     =========================================

!     c_ainv(j) =c(i)*ainv(i,j) sum over the rows
!     c_ainv_c =c(i)*ainv(i,j)*c(j) sum over the collums
      
      deltany0 = 0.d0
      do i=1,ngaus
         deltany0 = (c_ainv(i)/c_ainv_c)*my(i) + deltany0         
      enddo
      
!    =========================================
!     "deltany(ng)" is the correction to the electron 
!     potential for G#0 (21) (remember to sum over j as well)
!     =========================================

!     w(ng) = ! the weight function small g norms (ng)

      
      potcor(:,:,:) = dcmplx(0.d0,0.d0)
      deltany(:) = 0.d0
      do ng =2,ngmax
         do j=1,ngaus
            gainv=w(ng)*sum(ggaus(ng,:)*ainv(:,j))
            deltany(ng)=deltany(ng)+gainv*(my(j)-deltany0)
         enddo 
!     Copy the potential correction back on the fourier grid
         potcor(g_cor(ng,1),g_cor(ng,2),g_cor(ng,3))=deltany(ng)
      enddo

      
    

!     The pulay force calculated from the potential correction


      do i=1,nions
         fev(:,i)=0.d0
         do j=1,num
            t=j+(i-1)*num
            do ng=2,ngmax
               fev(:,i)=fev(:,i)+deltany(ng)*&
                    (0.d0,1.d0)*g_vec(ng,:)*&
                    conjg(ggaus(ng,t))*q(t)
            enddo
         enddo
         write(nconso,*)'force_pulay2',fev(:,i)
      enddo
      


!     The potential correction for G=0
 
      potcor(g_cor(1,1),g_cor(1,2),g_cor(1,3))= deltany0
  

      call fft3d(potcor,ngx,ngy,ngz,1)


!     write the potential correction on the same form as cveff

      t=0
      do i=1,ngz
         do j=1,ngy
            do k=1,ngx
               t=t+1
               deltapot(t)=-1*potcor(k,j,i)    
            enddo
         enddo
      enddo

      end subroutine pot_cor

!     ======================================================
!     Write the partial charges to NetCDFFile
!     ======================================================

      subroutine write_ptch(ptch,nions)
      use netcdfinterface
      use run_context

      implicit none
      integer, intent(in):: nions
      real*8, intent(in) :: ptch(nions)
      real*8 netcdf_ptch(nions)
      integer status,ncid_out,nstep
      
      netcdf_ptch(:)=ptch(:)
      call Reorder_atomvector(nconso, netcdf_ptch(:), & 
           "internal_to_netCDF")

!     -----------------------------------------------------
!     Open netCDF dataset
!     -----------------------------------------------------
      status =  nf_open(netCDF_output_filename, NF_WRITE, ncid_out)
      if (status /= nf_noerr) call abort_calc(nconso, &
      "write_ptch -> nf_open : error opening nc-outfile")

      status = nfgetglobaldim(ncid_out,'number_ionic_steps', nstep)
      status = nfput(ncid_out,"PartialCharge",netcdf_ptch,&
            dim_name1='number_of_dynamic_atoms',& 
            dim_name2='number_ionic_steps', &
            startnf=(/1,nstep/),countnf=(/nions,1/))

      if (status /= nf_noerr) call abort_calc(nconso, &
      "write_ptch -> nf_open : error writing partial charges")

      status=nf_close(ncid_out)
      if (status /= nf_noerr) call abort_calc(nconso, &
      "write_ptch -> nf_open : error closing netcdf file")
      
      end subroutine write_ptch
      end module blochlmodule
