#!/usr/bin/env python

''' Python module for stripping Dacapo netcdf files. Mainly used for
the stripnetcdf utility in Tools.  '''

__docformat__ = 'reStructuredText'

# Default options
optiondict={'keepwf':False,
	    'keepfft':False,
	    'keepchd':False,
	    'keepnlp':False,
	    'keepep':False,
            'keeppcd':False,
	    'recurse':False}

variabledict={'keepwf':'WaveFunction',
	      'keepfft':'WaveFunctionFFTindex',
	      'keepchd':'ChargeDensity',
	      'keepnlp':'NLProjectorPsi',
	      'keepep':'ElectrostaticPotential',
              'keeppcd':'PartialCoreDensity'}

import fnmatch,os,sys
from Dacapo import Dacapo

def FindFiles(rootdir='.',patternlist=['*.nc'],recurse=False,
	      returnfolders=False):
    '''Adapted from recipe 4.18 in Python Cookbook'''

    class Bunch:
	def __init__(self,**kwds):
	    self.__dict__.update(kwds)

    arg = Bunch(recurse=recurse,patternlist=patternlist,
		returnfolders=returnfolders,results=[])

    def visit(arg,dirname,files):
	for name in files:
	    fullname = os.path.normpath(os.path.join(dirname,name))
	    if arg.returnfolders or os.path.isfile(fullname):
		for pattern in arg.patternlist:
		    if fnmatch.fnmatch(name,pattern):
			arg.results.append(fullname)
			break
	if not arg.recurse:
	    files[:]=[]

    os.path.walk(rootdir,visit,arg)

    return arg.results

def stripnetcdf(filename,options=[]):
    '''
    Strips big netcdf variables from filename.

    Usage:
    >>> from stripnetcdf import stripnetcdf
    >>> stripnetcdf('out.nc',['keepwf'])
    '''
    #parse options
    for o in options:
	optiondict[o]=True

    try:
	loa = Dacapo.ReadAtoms(filename)
	calc = loa.GetCalculator()
    except:
	print "couldn't read %s, maybe a trajectory. skipping it." % filename
	return

    print filename+":"
    for option in optiondict.keys():
	if option != 'recurse' and not optiondict[option]:
	    try:
		calc.DeleteNetCDFEntry(variabledict.get(option))
		print "stripped %s" % variabledict.get(option)
	    except AttributeError:
		pass

    calc.WriteAsNetCDFFile(filename)
    print '\n'
	
def StripNetCDF(rootdir='.',patternlist=['*.nc'],options=[]):
    '''
    This is the main function that strips everything big from the
    netcdf files.
    
    rootdir is the directory to start in, normally the one you are
    in.
    
    patternlist is a list of patterns to match ['*box.nc','vib*.nc'],
    a single string can also be given.
    
    options is a tuple of strings with following possible values:
    ['recurse','keepwf','keepfft','keepchd','keepnlp','keepep']
    
    recurse tells stripnetcdf to recursively strip files in all
    directories in the rootdir that match the patterns.  the other
    options indicate that the corresponding variables should be kept.

    this will strip all the big variables from all files matching
    *vib*.nc recursively starting in the current directory.

    >>> from stripnetcdf import StripNetCDF
    >>> StripNetCDF(rootdir='.',patternlist='*vib*.nc',options=['recurse'])
    '''

    if isinstance(patternlist,str):
	patternlist=[patternlist]

    if 'recurse' in options:
	optiondict['recurse']=True
	
    files = FindFiles(rootdir=rootdir,patternlist=patternlist,
		      recurse=optiondict['recurse'])
    
    for filename in files:
	stripnetcdf(filename,options)
