#!/usr/bin/env python
# =======================================
#    STM of Al(fcc)100(N x M unit cell)
# =======================================
from Dacapo import Dacapo
from ASE import Atom,ListOfAtoms
from Numeric import sqrt,array
from ASE.Visualization.VMD import VMD

a0     = 4.05         # cubic fcc lattice constant
N      = 2            # repetition along x
M      = 2            # repetition along y
layers = 2            # slab layers
electronsperatom = 3 
vaclay = 5            # interlayer dist = a0/2


# ---------------------------------------------------
# ------- Setup the structure/calculation -----------
# ---------------------------------------------------

atoms   = ListOfAtoms([])
for n in range(layers):
    for i in range(N):
        for j in range(M):
            scaledpos = [(i+(n%2)/2.)/sqrt(2.),(j+(n%2)/2.)/sqrt(2.),-n/2.]
            atoms.append(Atom('Al', a0*array(scaledpos)))
                                       
                                       
unitcell = [[N/sqrt(2.), 0.0,        0.0],
            [0.0,        M/sqrt(2.), 0.0],
            [0.0,        0.0,        (vaclay+layers)/2.]]

atoms.SetUnitCell(a0*array(unitcell),fix=True)

# VMD(atoms)
                            
calc = Dacapo(planewavecutoff = 150, 
              nbands = 10 + len(atoms)*electronsperatom/2, 
              kpts=(4,4,1),
              xc = 'LDA',
              usesymm=True,
              out = 'Al100.nc',txtout = 'Al100.txt') 

atoms.SetCalculator(calc) 
energy = atoms.GetPotentialEnergy()

# =======================================
#    STM of Al(fcc)100(N x M unit cell)
# =======================================
# --------------------------------------------------
# ------------ STM image generation part -----------
# --------------------------------------------------

from Dacapo.ElectronicStates import ElectronicStates
from Dacapo import Dacapo
from ASE.Utilities.STMTool import STMTool
from ASE.Visualization.VTK import VTKPlotSTM,VTKPlotAtoms

loe = ElectronicStates(filename='Al100.nc')
stm = STMTool(loe,contourvalue=1.0e-4, channel="s",
             normalaxis=2, smoothfactor=0.1)    # available channels: s,px,py,pz

stmplot = VTKPlotSTM(stm)
stmplot.SetColorRange((3.1, 3.6))        # [3.1 AA; 3.6 AA] from surface span color range

# uncomment the next two lines to add the atoms to the plot
atoms = Dacapo.ReadAtoms('Al100.nc')
atomplot = VTKPlotAtoms(atoms,parent=stmplot)

# Change the radius to 2.0 AA and set the color to red
# The color is given in an rgb (red,green,blue) scale
# p1=atomplot.GetDictOfSpecies()['Al']
# p1.SetRadius(2.0)
# p1.SetColor((1,0,0))

stmplot.Render()                        # updates picture
stmplot.SaveAsBMP("Al100_stm.bmp")      # save as bitmap plot




