# Copyright (C) 2003  CAMP
# Please see the accompanying LICENSE file for further information.

""" Dacapo calculator
The class ``Dacapo`` implements the ASE calculator interface for the
DFT pseudo-potential code Dacapo.

Dacapo(...) -> DFT Dacapo calculator.

The calculator can be constructed from a NetCDF Dacapo file, from another
calculator, or from scratch giving all the necessary
parameters  (or a combination)
        
"""

__docformat__ = 'reStructuredText'

import Numeric as num

from ASE.Calculators.Calculator import Calculator
from ASE.Utilities.MonkhorstPack import MonkhorstPack
from Dacapo.Simulation import Simulation
from ASE.Utilities.Subject import Subject
import LinearAlgebra as LinAlg
import weakref
import copy
import os,time

class Dacapo(Simulation):
    """Implementation of the Dacapo DFTCalculator interface."""
    
    def __init__(self, 
                 out=None, txtout=None,infile=None,nbands=None,
                 xc=None,kpts=None,spinpol=None,planewavecutoff=None,
                 usesymm=None,dipole=None,densitycutoff=None,
                 use_free_energy=False,scratchdir=None,kT=0.1,
                 fixdensity=False,save_memory=True):

        """Dacapo(...) -> DFT Dacapo calculator.

        The calculator can be constructed from a file, 
        or from scratch giving all the necessary
        parameters  (or a combination)."""

	Simulation.__init__(self)

        # Override kpts, nbands, spinpol, xc:
        if nbands is not None:
            self.SetNumberOfBands(nbands)

        if xc is not None:
            self.SetXCFunctional(xc)
        else:
            self.SetXCFunctional('PW91')
	    
        if kpts is not None:
            self.SetBZKPoints(kpts)
	else:
	    # default gamma point
	    self.SetBZKPoints([(0.0, 0.0, 0.0)])
	    
        if spinpol is not None:
            self.SetSpinPolarized(spinpol)
	else:
	    self.SetSpinPolarized(False)
	    
        if planewavecutoff is not None:
            self.SetPlaneWaveCutoff(planewavecutoff)
	else:
	    self.SetPlaneWaveCutoff(350)
	    
        if densitycutoff is not None:
            self.SetDensityCutoff(densitycutoff)

	if usesymm:
            self.SetSymmetryOn()
	else: 
            self.SetSymmetryOff()
	    
        if dipole is True: 
            self.DipoleCorrection()

        if scratchdir is not None:
            self.GetJobType().SetScratch(scratchdir) 

	self.SetElectronicTemperature(kT)

	self.SetOccupationStatistics('FermiDirac')

	self.SetConvergenceParameters(energy=0.00001,
				      density=0.0001,
				      occupation=0.001)

        if fixdensity: 
            self.SetChargeMixing(False) # Harris calculation
        else:
            self.SetChargeMixing(True)  # uses Pulay algorithm to mix density
            
	self.SetKerkerPreconditioning(False)
	self.SetEigenvalueSolver('eigsolve')

	user = os.environ.get('USER')
	if user is not None:
	    self.SetNetCDFEntry('User',value=user)
	else:
	    self.SetNetCDFEntry('User',value='Unknown')

	self.SetNetCDFEntry('Date',value=time.asctime())
	
	self.SetKeywords('')
	    
        self.use_free_energy = use_free_energy            
	self.SetNetCDFFile(out)
	self.SetTxtFile(txtout)
        self.SetInFile(infile)
        self.ready = False
        self.positions = None

	# set default pseudopotentials database
        from Dacapo.PseudoPotentials import defaultpseudopotentials
        self.SetPseudoPotentials(copy.copy(defaultpseudopotentials))

        self.SaveMemory(save_memory)

	# some default settings
        self.StayAliveOff()

        # set default runscript
        self.SetRunScript()
        

    def __del__(self):
        if hasattr(self,'_dacapo_is_running'): 
            if self._dacapo_is_running == 'true':
	       try: 
                   self.ExecuteExternalDynamics(stopprogram='true')
               except: 
                   pass

    def SetPseudoPotentials(self,pseudopotentials):
        self.pseudopotentials = pseudopotentials 

    def GetPseudoPotentials(self):
        return self.pseudopotentials 

    def SetPseudoPotential(self,Z,path):
        self.GetPseudoPotentials()[Z] = path
        
    def GetPseudoPotential(self,Z):
        return self.GetPseudoPotentials()[Z]

    def SetNetCDFFile(self,ncfile=None): 
        ''' Set calculator output ncfile to ncfile if given
        or to out_timestring.nc if not.'''
        self.StopDacapo()
        if ncfile is None:	
            ncfile = 'out.nc'
        self.ncfile = ncfile
        
        if not hasattr(self,'txtfile'):
            self.SetTxtFile()
            
    def GetNetCDFFile(self):
        return self.ncfile

    def SetTxtFile(self,txtfile=None):
        '''Set txt file to txtfile if given, otherwise
        to the name of the netcdf file but with a .txt
        extension.'''
        self.StopDacapo()
	if txtfile == None:
	    from os.path import splitext
	    base,ext=splitext(self.GetNetCDFFile())
	    txtfile = base + '.txt'
	self.txtfile = txtfile

    def SetInFile(self,infile):
        self.infile = infile

    def CheckCalculation(self): 
        ''' simple check of caculational parameters '''
        if self.GetDensityCutoff() is not None: 
           density = self.GetDensityCutoff() 
           if density < self.GetPlaneWaveCutoff(): 
		raise 'Density cutoff must be >= plane wave cutoff'

    def SaveMemory(self,value):
        self.save_memory = value


    def ReadAtoms(filename,calc=None,index=-1,save_memory=True):
        """ Start from a Dacapo NetCDF file.
	This is a static class method, use like 
		atoms = Dacapo.ReadAtoms(file)
        
        All existing NetCDF variables <variable> will be
        defined through either the default DFT methods,
        or via the GetNetCDFEntry. 
        """
        from Dacapo import NetCDF
        from Scientific.IO.NetCDF import NetCDFFile
	from ASE import ListOfAtoms
        from Simulation import DelayedReadNetCDFEntry

        oldfile=NetCDF.File(filename)
	file = NetCDFFile(filename,'r')

	# get a instance of the Dacapo calculator
	if calc is None: 
	    calc = Dacapo(save_memory=save_memory)

        if calc.GetNetCDFFile()!=filename: 
        	calc.SetNetCDFFile(filename)
        
        calc.DeleteNetCDFEntry('TotalEnergy')
        calc.DeleteNetCDFEntry('TotalFreeEnergy')

	dict = oldfile.__dict__

        if 'WaveFunction' in dict.keys() and calc.save_memory:
            attr = dict['WaveFunction']
            newattr = DelayedReadNetCDFEntry(file,name='WaveFunction',
                                             dimensionnames = attr.GetDimensionNames())
            
                                           
            # get name to add this attr under
            attrname = calc.GetAttrName(newattr.GetName())
            setattr(calc,attrname,newattr)

            del dict['WaveFunction']

        
        for variable in dict.keys():

            if variable[-1]!="_" and variable[0]!="_":#don't take private attr
                	
		# now add this variable
		attr = dict[variable]
                # read information
                attr.ReadFromNetCDFFile(file) 

		# get name to add this attr under
		attrname = calc.GetAttrName(attr.GetName())
		setattr(calc,attrname,attr)

	# remove DynamicAtomVelocities
        calc.DeleteNetCDFEntry('DynamicAtomVelocities')

        # remove Dynamics 
        calc.DeleteNetCDFEntry('Dynamics') 
        
        # define the atoms
	try: 
	    atoms = calc.GetListOfAtoms()
            # read the information for ListOfAtoms
            calc.ReadListOfAtoms(index)       
            calc._AtomsHasChanged()
	except AttributeError: 
        # create the atoms 
            atoms = ListOfAtoms([])
            atoms.SetCalculator(calc)
            # read the information for ListOfAtoms
            calc.ReadListOfAtoms(index)
            # initialize atomsinfo
            calc._AtomsHasChanged()
            calc.positions = atoms.GetCartesianPositions()

        calc.ready = True
        calc.restart_dacapo = False
        calc.SetAtomsCounter(atoms.GetCount())

        if not save_memory:
            file.close()
        del oldfile
        
        return atoms
    # make this a static class method
    ReadAtoms = staticmethod(ReadAtoms)


    def _SetListOfAtoms(self, atoms):
        """Make a weak reference to the ListOfAtoms."""
        self.SetAtomsCounter(atoms.GetCount()-1)
        self.atoms = weakref.ref(atoms)
        self.ready = False
        # self.RestartDacapo()

    def GetListOfAtoms(self):
        """Return the ListOfAtoms."""
        return self.atoms()

    def GetNumberOfBands(self):
        """Return the number of bands."""
        return self.GetNetCDFEntry('ElectronicBands','NumberOfBands')
  
    def SetNumberOfBands(self, nbands):
        self.RestartDacapo()
        self.SetNetCDFEntry('ElectronicBands',att='NumberOfBands',value=nbands)

    def SetPlaneWaveCutoff(self,cutoff):
        """ set plane wavecutoff (eV) """
        self.DeleteNetCDFEntry('softgrid_dim1')
        self.DeleteNetCDFEntry('softgrid_dim2')
        self.DeleteNetCDFEntry('softgrid_dim3')
        self.DeleteNetCDFEntry('hardgrid_dim1')
        self.DeleteNetCDFEntry('hardgrid_dim2')
        self.DeleteNetCDFEntry('hardgrid_dim3')
	self.RestartDacapo()
        self.SetNetCDFEntry('PlaneWaveCutoff',value=cutoff) 

    def GetPlaneWaveCutoff(self):
        return self.GetNetCDFEntry('PlaneWaveCutoff')

    def SetDensityCutoff(self,cutoff):
        """ set the cutoff (eV) for the density """ 
        self.RestartDacapo()
        self.SetNetCDFEntry('Density_WaveCutoff',value=cutoff)

    def GetDensityCutoff(self):
        return self.GetNetCDFEntry('Density_WaveCutoff')

    def SetElectronicTemperature(self,temp):
        self.RestartDacapo()
        self.SetNetCDFEntry('ElectronicBands',
			    att='OccupationStatistics_FermiTemperature',
			    value=temp)

    def GetElectronicTemperature(self):
        value = self.GetNetCDFEntry('ElectronicBands',
				    att='OccupationStatistics_FermiTemperature')
        return value
        
    def SetOccupationStatistics(self,value):
        self.RestartDacapo()
        if value=='FermiDirac' or value=='MethfesselPaxton':
            self.RestartDacapo()
            self.SetNetCDFEntry('ElectronicBands',
                                att='OccupationStatistics',
                                value = value)
        else:
            print 'Occupation statistics must be FermiDirac or'
            print 'MethfesselPaxton'

    def GetOccupationStatistics(self):
        return self.GetNetCDFEntry('ElectronicBands',
                                    att='OccupationStatistics')

    def SetConvergenceParameters(self,energy=None,density=None,occupation=None):
        if energy is not None: 
            self.SetNetCDFEntry("ConvergenceControl",
                                att="AbsoluteEnergyConvergence",value=energy)
        if density is not None: 
            self.SetNetCDFEntry("ConvergenceControl",
                                att="DensityConvergence",value=density)
        if occupation is not None: 
            self.SetNetCDFEntry("ConvergenceControl",
                                att="OccupationConvergence",value=occupation)

    def GetConvergenceParameters(self):
        parameters = {}
        parameters.update({'energy': self.GetNetCDFEntry("ConvergenceControl",
                                                         att="AbsoluteEnergyConvergence")})
        parameters.update({'density': self.GetNetCDFEntry("ConvergenceControl",
                                                          att="DensityConvergence")})
        parameters.update({'occupation': self.GetNetCDFEntry("ConvergenceControl",
                                                             att="OccupationConvergence")})
        return parameters
        
    def SetChargeMixing(self,value):
        self.RestartDacapo()
        if value:
            noyes = 'Yes'
        else:
            if not value:
                noyes = 'No'
            else:
                print 'value must be True (density mixing using the Pulay algorithm)'
                print 'or False (density is fixed at input value,Harris functional) '
                return

        self.SetNetCDFEntry('ChargeMixing',att='UpdateCharge',value=noyes)

    def GetChargeMixing(self):
        return self.GetNetCDFEntry('ChargeMixing',att='UpdateCharge')
 
    def SetKerkerPreconditioning(self,value):
        self.RestartDacapo()
        if value:
            noyes = 'Yes'
        else:
            if not value:
                noyes = 'No'
            else:
                print 'value must be True (Kerker preconditioning is used)'
                print 'or False (Linear mixing (default)'
                return
        self.SetNetCDFEntry('ChargeMixing',
                            att='Pulay_KerkerPrecondition',
                            value=noyes )

    def GetKerkerPreconditioning(self):
        return self.GetNetCDFEntry('ChargeMixing',att='Pulay_KerkerPrecondition')


    def SetEigenvalueSolver(self,value):
        self.RestartDacapo()
        if value=='eigsolve' or value=='rmm-diis':
            self.SetNetCDFEntry('ElectronicMinimization',
                                att='Method',
                                value=value)
        else:
            print 'Eigenvalue solver must be eigsolve or rmm-diis'

    def GetEigenvalueSolver(self):
        return self.GetNetCDFEntry('ElectronicMinimization',att='Method')


    def GetXCFunctional(self):
        """Return the XC-functional identifier.        
        'LDA', 'PBE', ..."""

	return self.GetNetCDFEntry('ExcFunctional')
 
    def SetXCFunctional(self, xc):
        """Set the XC-functional. 
        ``xc`` must be one of: 'LDA', 'PBE', ..."""

        xclist = ['LDA','PW91','PBE','RPBE','revPBE']
        if xc not in xclist:
            print 'Exchange-correlation functional must be one of ',xclist
            return 
            
	if xc=='LDA':
            xcdacapo = 'VWN'
        else:
            xcdacapo = xc

	self.RestartDacapo()
        self.SetNetCDFEntry('ExcFunctional',value=xcdacapo)

    def GetBZKPoints(self):
        """Return the k-points."""
        return self.GetNetCDFEntry('BZKpoints')
 
    def SetBZKPoints(self, bzkpts):
        """Set the scaled k-points. 
        ``kpts`` should be an array of scaled k-points."""

        self.RestartDacapo()   
        if type(bzkpts[0]) is int:
            bzkpts = MonkhorstPack(bzkpts)

        self.DeleteNetCDFEntry('BZKpoints')
        self.DeleteNetCDFEntry("number_BZ_kpoints")
        self.SetNetCDFEntry('BZKpoints',value=bzkpts,
                             dim=["number_BZ_kpoints"] )
                             
    def GetSpinPolarized(self):
        """Is it a spin-polarized calculation?"""
	if self.GetNetCDFEntry('ElectronicBands','SpinPolarization') == 2: 
	    return True 
        else: 
	    return False
   
    def SetSpinPolarized(self, spinpol,fixmagmom=None):
        """Is it a spin-polarized calculation?"""
        self.RestartDacapo()
        if spinpol: 
            self.SetNetCDFEntry('ElectronicBands',
				att='SpinPolarization',value=2)
        else: 
            self.SetNetCDFEntry('ElectronicBands',
				att='SpinPolarization',value=1)
        
	if fixmagmom is not None:
	    self.SetNetCDFEntry('ElectronicBands',
				att='FixedMagneticMoment',value=fixmagmom)
	    
    def GetIBZKPoints(self):
        """Return k-points in the irreducible part of the Brillouin zone.

        These are created at runtime, so asking for them may result in a
        calculation if they don't exist!
        """
        self.Calculate()
        ibzkpoints = self.GetNetCDFEntry('IBZKpoints')
        return ibzkpoints
 
    def GetIBZKPointWeights(self):
        """Weights of the k-points. 
        
        The sum of all weights is one."""
        return self.GetNetCDFEntry('KpointWeight')

    def GetEigenvalues(self, kpt=0, spin=0):
        """Return the eigenvalues for all bands."""  
        # absolute scale at the moment
        self.Calculate()
        return self.GetNetCDFEntry('EigenValues')[-1][kpt,spin]

    def GetFermiLevel(self):
        return self.GetNetCDFEntry('FermiLevel')[-1]

    def GetOccupationNumbers(self, kpt=0, spin=0):
        """Return the occupation numbers for all bands.
 
        These numbers fall in the range [0,1] for a spin-polarized
        calculation, and in the range [0,2] otherwise."""        
        self.Calculate()
        return self.GetNetCDFEntry('OccupationNumbers')[-1][kpt,spin]

    def GetMagneticMoment(self):

        if not self.GetSpinPolarized():
            return 0.0

        weights = self.GetIBZKPointWeights()
        e0 = 0.0
        e1 = 0.0
        for kpt in range(len(self.GetIBZKPoints())):
            w = weights[kpt]
            e0 += num.sum(self.GetOccupationNumbers(kpt=kpt,spin=0))*w
            e1 += num.sum(self.GetOccupationNumbers(kpt=kpt,spin=1))*w

        return abs(e0-e1)
        

    def GetWaveFunctionClass(self):
        """  setup class for handling wavefunctions.
        Since wavefunction information is only written at end of dacapo
        calculation, dacapo must be stopped  
        """
        from Dacapo.WaveFunction import WaveFunction

        self.Calculate()
        self.StopDacapo()

        if not hasattr(self,'wavefunction'):
            self.wavefunction = WaveFunction(calculator=self)
        
        self.wavefunction.ReadFromNetCDFFile(self.ncfile)

        return self.wavefunction
 
    def GetWaveFunctionArray(self,band=None,kpt=None,
			     spin=None,wavefunction=None,kpoint=None ):
        """Return array of wavefunction values."""
        wfct = self.GetWaveFunctionClass()
        return wfct.GetWaveFunction(band,kpt,spin,wavefunction,kpoint)

    def GetWaveFunction(self,band,kpt=0,spin=0):
        """ get wavefunction in reciprocal space """
        wfct = self.GetWaveFunctionClass()
        wfct.SetBand(band)
        wfct.SetKPointNumber(kpt)
        wfct.SetSpin(spin)
	return  wfct.GetReciprocalBlochFunction()

    def GetDensityArray(self, spin=0):
        """Return array of density values."""
        from ASE.Utilities.VectorSpaces import BravaisLattice
        self.Calculate()
        self.StopDacapo()
        
        densgrid = self.GetNetCDFEntry('ChargeDensity')[spin]

        grid = BravaisLattice(basis=self.GetListOfAtoms().GetUnitCell())
        vol = grid.GetVolumeOfCell()
    
        # Transposing to get grid on the form: grid(1,2,3)
        # By copying the grid it is made contiguous
        tempgrid=copy.copy(num.transpose(densgrid))

        return tempgrid/vol

    def GetElectrostaticPotential(self,spin=0):
        """Return array for electrostativ potential"""
        self.Calculate()
        self.StopDacapo()

        grid = self.GetNetCDFEntry('ElectrostaticPotential')[spin]

        # Transposing to get grid on the form: grid(1,2,3)
        # By copying the grid it is made contiguous
        tempgrid=copy.copy(num.transpose(grid))

        return tempgrid
    
    def SetDensityArray(self,density,spin=0):
        """ set array of density values for the given spin."""
        from ASE.Utilities.VectorSpaces import BravaisLattice

        grid = BravaisLattice(basis=self.GetListOfAtoms().GetUnitCell())
        vol = grid.GetVolumeOfCell()
        
                
        N1,N2,N3=num.shape(density)
        if self.GetSpinPolarized()==True:
            nspin = 2
        else:
            nspin = 1

        # check spin
        if spin not in [0,1]:
            raise "spin should be 0 or 1"

        if spin==1 and nspin==1: 
            raise "Calculation not spin polarized"


        densgrid=num.zeros([nspin,N3,N2,N1],num.Float)
        densgrid[spin, :,:,:]=num.transpose(density)
        print 'densgrid ',densgrid.shape


        # set the other spin
        if nspin==2:
            other = 1
            if spin==1:
               other = 0
            tmp = self.GetNetCDFEntry('ChargeDensity')
            if tmp is not None:
               densgrid[other,:,:,:]=tmp[other]

        densgrid[:,:,:,:] *= vol

	self.SetNetCDFEntry("ChargeDensity",value=densgrid,
                             dim=['number_of_spin',
                            'hardgrid_dim3',
                            'hardgrid_dim2',
                            'hardgrid_dim1'])

      

        # temporary fix so density is not removed
        self.keepdensity = True

        # add softgrid dimensions by writing to dummy variables
        # temporary fix until dacapo can handle only receiving the
        # hardgrid dimensions.
        n = 1
        for dim in [N1,N2,N3]: 
            grid1=num.zeros([dim],num.Float)
            self.SetNetCDFEntry('dummy'+str(n),
                                value=grid1,
                                dim=['softgrid_dim'+str(n)])
            n += 1

    def SetWavefunctions(self,wflist):
        # for now expect a list with the following layout
        # [Nspin,Nkpt,Nbands,Npws,2]
        self.SetNetCDFEntry("WaveFunction",value=wflist,
                           dim= ['number_IBZ_kpoints',
                                 'number_of_spin',
                                 'number_of_bands',
                                 'number_plane_waves',
                                 'real_complex'] )
        self.keepwavefunctions = True
        

    def DipoleCorrection(self,MixingParameter=0.2,InitialValue=0.0,
                         AdditiveDipoleField=0.0):
        """ This methods helps setting up the Dipole Correction scheme
        See the NetCDF documentation for the details.
        From John Kitchen's DipoleCorrection class.
        """

        self.SetNetCDFEntry('DipoleCorrection') 
        self.SetNetCDFEntry('DipoleCorrection',att='MixingParameter',
                            value=MixingParameter)
        self.SetNetCDFEntry('DipoleCorrection',att='InitialValue',
                            value=InitialValue)
        self.SetNetCDFEntry('DipoleCorrection',att='AdditiveDipoleField',
                            value=AdditiveDipoleField)

    def GetDipoleCorrection(self):
        '''
        Get Dipole correction settings.
        '''
        mp = self.GetNetCDFEntry('DipoleCorrection',
                                 att='MixingParameter')
        iv = self.GetNetCDFEntry('DipoleCorrection',
                                 att='InitialValue')
        adf=self.GetNetCDFEntry('DipoleCorrection',
                                att='AdditiveDipoleField')

        if mp is not None and iv is not None and adf is not None:
            return {'MixingParameter':mp,
                    'InitialValue':iv,
                    'AdditiveDipoleField':adf}
        else:
            return None
        
    def SetElectrostaticDecoupling(self,numberofgaussians=3,ecutoff=100,
                                   widthofgaussian=0.35):
        """ Add electrostatic decoupling, Jan Rossmeisl
        """
        self.SetNetCDFEntry("Decoupling",value="")
        self.SetNetCDFEntry("Decoupling",att="NumberOfGaussians",
                             value=numberofgaussians)
        self.SetNetCDFEntry("Decoupling",att="ECutoff",value=ecutoff)
        self.SetNetCDFEntry("Decoupling",att="WidthOfGaussian",
                             value=widthofgaussian)

    def CalculateAtomicDOS(self,
			   energywindow=(-15,5),
			   energywidth=0.2,
			   numberenergypoints=250,
			   cutoffradius=1.0):
	"""Sets up all possible attributes needed to calculate the AtomicDOS
	with reasonable default values."""
	
        # self.RestartDacapo()
        self.SetNetCDFEntry("PrintAtomProjectedDOS")
	self.SetNetCDFEntry("PrintAtomProjectedDOS",
                            att="EnergyWindow", 
			    value = energywindow)
	self.SetNetCDFEntry("PrintAtomProjectedDOS",
                            att="EnergyWidth",
			    value = energywidth)
	self.SetNetCDFEntry("PrintAtomProjectedDOS",
                            att="NumberEnergyPoints",
			    value = numberenergypoints)
	self.SetNetCDFEntry("PrintAtomProjectedDOS",
                            att="CutoffRadius",
			    value = cutoffradius)

    def SetMultiCenters(self,multicenters,
                        energywindow=(-15,5),
                        energywidth=0.2,
                        numberenergypoints=250,
                        cutoffradius=1.0):

        from MultiCenterProjectedDOS import  MultiCenterProjectedDOS
        self.mcenters = MultiCenterProjectedDOS(multicenters,
                            energywindow=energywindow,
                            energywidth=energywidth,
                            numberenergypoints=numberenergypoints,
                            cutoffradius=cutoffradius )



    def GetMultiCenters(self):
        from MultiCenterProjectedDOS import  MultiCenterProjectedDOS
        from Scientific.IO.NetCDF import NetCDFFile

        self.RestartDacapo()

        self.mcenters = MultiCenterProjectedDOS()
        self.mcenters.ReadFromNetCDFFile(NetCDFFile(self.ncfile))
        return self.mcenters.read()



    def AddExternalPotential(self,potential):
        self.SetNetCDFEntry("ExternalPotential",value=potential)

    def GetLDOS(self, **keywords ): 
        """ return an instance of AtomProjectedDOSTool 
        From John Kitchen.
        """
	from Dacapo.AtomProjectedDOS import AtomProjectedDOS
        from Scientific.IO.NetCDF import NetCDFFile
        
	self.RestartDacapo()
        atomdos = AtomProjectedDOS()
        # read in
        f = NetCDFFile(self.ncfile)
        atomdos.ReadFromNetCDFFile(f)
        f.close()
        return atomdos.GetDOS(**keywords) 

    def SetSymmetryOn(self):
	self.SetNetCDFEntry("UseSymmetry",value="Maximum")

    def SetSymmetryOff(self): 
	self.SetNetCDFEntry("UseSymmetry",value="Off")

    def GetSymmetry(self):
        from string import join,strip
        symmetry = self.GetNetCDFEntry("UseSymmetry")
        return strip(join(symmetry,''))
    
    def StayAliveOn(self): 
        # self.RestartDacapo()
        self.StopDacapo()
	self.stayalive = True

    def StayAliveOff(self):
        self.RestartDacapo()
        self.StopDacapo()
	self.stayalive = False

    def SetExternalDynamicsOn(self): 
	self.StayAliveOn()
    

    def Calculate(self):
        import os

        if self.ready and \
            self.GetAtomsCounter()==self.GetListOfAtoms().GetCount():
    	    return

        if self.ready:
    
            # check if we should restart because listofatoms has
            # changed anything besides the positions
            if self._AtomsHasChanged():
                self.RestartDacapo()
            else:
                diff = self.GetListOfAtoms().GetCartesianPositions()-self.positions
                diff = num.sum(num.sum(diff**2))
                if diff<1e-20:
                    # atoms has changed something unimportant like the velocities
                    return

        # make sure ListOfAtoms is defined in NetCDF
	self.WriteListOfAtoms() 

#        if self.infile: 
#             os.system('rm -f '+self.infile)

        if hasattr(self,'restart_dacapo'):
            if self.restart_dacapo == True:
                self.StopDacapo()
                self.restart_dacapo = False
                # delete netcdf entries
                self.DeleteNetCDFEntry('WaveFunctionFFTindex')
                if not hasattr(self,'keepdensity'): 
                    self.DeleteNetCDFEntry('ChargeDensity') 
                self.DeleteNetCDFEntry('EigenValues')
                self.DeleteNetCDFEntry('OccupationNumbers')
                self.DeleteNetCDFEntry('ElectrostaticPotential')
                if not hasattr(self,'keepwavefunctions'):
                    self.DeleteNetCDFEntry('WaveFunction')
                self.DeleteNetCDFEntry('NLProjectorPsi') 
                self.DeleteNetCDFEntry('TotalEnergy')
                self.DeleteNetCDFEntry('TotalFreeEnergy')
		self.DeleteNetCDFEntry('PartialCoreDensity')
                if self.stayalive:
		    import os,string
        	    if os.environ.get('DACAPOSLEEP'):
                    	t = string.atof(os.environ.get('DACAPOSLEEP'))
                    	time.sleep(t)
                    print 'restarting dacapo .. '

        self.CheckCalculation()

        if self.stayalive:
            self.ExecuteExternalDynamics(outfile=self.ncfile,
					 infile=self.infile,
                                         ascii=self.txtfile)
        else:
            os.system('rm -f '+self.ncfile+' ' +self.txtfile)

            self.Execute(infile=self.infile,outfile=self.ncfile,
                         ascii=self.txtfile)
                        
	# now read everything from the output netcdf file into 
        # this calculator. 
        # This is analog to starting the calculation from this 
        # file
        # in this contect ignore the returned atoms, it will
        # be identical to the attached atoms
        atoms = Dacapo.ReadAtoms(self.ncfile,calc=self,save_memory=self.save_memory)

	self.ready = True
	self.SetAtomsCounter(self.GetListOfAtoms().GetCount())
        self.positions = self.GetListOfAtoms().GetCartesianPositions()

        ### added because on some systems
        ### it appears that dacapo starts up before the
        ### netcdf file is written and this results in errors
        ### This sleep command appears to alleviate the issue.
        ### set the env var DACAPOSLEEP in the script with
        ### os.environ['DACAPOSLEEP']='10'
        ### or in your .cshrc
        ### setenv DACAPOSLEEP 10
        import os,string
	if os.environ.get('DACAPOSLEEP'):
	    t = string.atof(os.environ.get('DACAPOSLEEP'))
	    time.sleep(5)

    def GetPotentialEnergy(self,force_consistent=False):
        """Return the energy for the current state of the ListOfAtoms."""
        self.Calculate()
        try: 
            if force_consistent:
                return self.GetNetCDFEntry('TotalFreeEnergy')[-1]
            else:
                return self.GetNetCDFEntry('TotalEnergy')[-1]
        except TypeError:
            raise RuntimeError('Error in calculating the total energy\n' +
                               'Check ascii out file for error messages')


    def GetFreeEnergy(self):
        """Return the free energy for the current state of the ListOfAtoms."""
        self.Calculate()
        try:
            return self.GetNetCDFEntry('TotalFreeEnergy')[-1]
        except TypeError:
            raise RuntimeError('Error in calculating the total energy\n' +
                               'Check ascii out file for error messages')

    def GetCartesianForces(self):
        """Return the forces for the current state of the ListOfAtoms."""
        self.Calculate()
	return self.GetNetCDFEntry('DynamicAtomForces')[-1]

    def GetStress(self):
        """Return the stress for the current state of the ListOfAtoms.
        If the calculation is not up-to-date or the stress has not
        been calculated the calculation is rerun"""

	if self.GetNetCDFEntry("NetCDFOutputControl",
			       att = 'PrintTotalStress') == 'Yes':
	    self.Calculate()
	    return self.GetNetCDFEntry('TotalStress')
	else:
	    self.CalculateStress()
	    self.Calculate()
	    return self.GetNetCDFEntry('TotalStress')
    
    def CalculateStress(self):
        """ set NetCDF entry so stress will be calculated """
        self.SetNetCDFEntry("NetCDFOutputControl",
			    att = 'PrintTotalStress',value="Yes")

    def WriteAsNetCDFFile(self,filename):
        """Method to create a netcdf file from the current calculator. 

        This method asks all the attributes of the instance
        to write them down in the file with the given filename.
        """
        from Scientific.IO.NetCDF import NetCDFFile

	# remove stuff we do not want to write to dacapo
        # because it is output only
        self.DeleteNetCDFEntry('WannierAugFactor')
        self.DeleteNetCDFEntry('TypeNLProjectorm')
        self.DeleteNetCDFEntry('TypeNLProjectorl')
        self.DeleteNetCDFEntry('NLProjectorPsi')
        self.DeleteNetCDFEntry('NumberOfNLProjectors')

	self.WriteListOfAtoms()
	
        file=NetCDFFile(filename,'w')
        file.history='Dacapo'
        self.WriteToNetCDFFile(file)
	
        file.close()
            
    def GetXCEnergies(self,*functional):
        """
        This function returns the self-consistent energy and/or
	energies associated with various functionals. 
        The functionals are currently PZ,VWN,PW91,PBE,revPBE, RPBE.
        The different energies may be useful for calculating improved
	adsorption energies as in B. Hammer, L.B. Hansen and
	J.K. Norskov, Phys. Rev. B 59,7413. 
        Examples: 
        GetXCEnergies() #returns the self-consistent total energy
	
        GetXCEnergies('PBE') # returns the PBE total energy
        GetXCEnergies('PW91','PBE','revPBE') # returns a
	# list of energies in the order asked for 
        """
	
        # make list of indices based on functionals
        xcreference_list=['PZ','VWN','PW91','PBE','revPBE','RPBE'] 
        func_list=[]
        if len(functional)==0:
            return self.GetPotentialEnergy()
        else:
            for func in functional:
                try:
                    #throws an exception if the index isn;t found
                    func_list.append(xcreference_list.index(func))
                    
                except:
                    print 'The "%s" functional is not supported.' % func
                    pass

            if len(func_list)==1:
                # This is just so it returns a scalar number if only
		# one functional is selected
                # The energy is taken at the last ionic step
                return self.GetNetCDFEntry('EvaluateTotalEnergy')[-1,func_list[-1]]
            else:
                # map the list onto self._potentialenergy_
                #return map(lambda x:self._FunctionalEnergies_[x],func_list)
                return [self.GetNetCDFEntry('EvaluateTotalEnergy')[-1,func_list[n]] for n in range(len(func_list))]
            
    def GetXCEnsembleEnergies(self):
        self.Calculate()
        return self.GetNetCDFEntry('EnsembleXCEnergies')

    def GetEnsembleCoefficients(self):
        self.Calculate()
        E = self.GetPotentialEnergy()
        xc = self.GetNetCDFEntry('EnsembleXCEnergies')
        Exc = xc[0]
        exc_c = self.GetNetCDFEntry('EvaluateCorrelationEnergy')
        exc_e =  self.GetNetCDFEntry('EvaluateExchangeEnergy')
        exc = exc_c + exc_e
        if self.GetXCFunctional() == 'RPBE':
                Exc = exc[-1][-1]

        E0 = xc[1]       # Fx = 0

        diff0 = xc[2] # - Exc
        diff1 = xc[3] # - Exc
        diff2 = xc[4] # - Exc
        coefs = (E + E0 - Exc,diff0-E0 ,diff1-E0,diff2-E0)
        print 'ensemble: (%.9f, %.9f, %.9f, %.9f)'% coefs
        return num.array(coefs)


    def SetNetCDFEntry(self,variable,value=None,dim=None,att=None,series=None):
        """ define a general netcdf entry for this calculation.
        The entry is added as calc._<variable> = NetCDF.Entry,
        If the entry is allready defined, only the value are
        modified. """
        from Dacapo import NetCDF

        # find name of attribute this entry should be added under
        attname = self.GetAttrName(variable)

        # check if this is allready added
        if hasattr(self,attname):
            entry = getattr(self,attname)
            if not att:
                # set the new value for the variable
                entry.SetValue(value)
            else:
                # get current list of attributes
                attributes = entry.GetAttributes()
                # update with the new attributes, possible overwriting
                # an existing attribute
                attributes.update({att:value})
                entry.SetAttributes(attributes)
        else:
            # define a new NetCDF entry
            if not att:
		if series:
                     entry = NetCDF.Series(name=variable,dimensionnames=dim, 
					   value=value)
		else:
                     entry = NetCDF.Entry(name=variable,dimensionnames=dim, 
					  value = value)
            else:
		if series: 
                     entry = NetCDF.Series(name=variable,dimensionnames=dim)
		else:
                     entry = NetCDF.Entry(name=variable,dimensionnames=dim)
                entry.SetAttributes({att:value})
                
            # add this to the Calculation object under attname
            setattr(self,attname,entry)

        self.ready = False

    def GetNetCDFEntry(self,variable,att=None):
        """ return the value of NetCDF entry <variable> or
        the value of <variable>.<attribute> if attribute is given, 
        None is returned if variable or att does not exsist.
        """

        # get the attribute name to look for
        attname = self.GetAttrName(variable)

        try:
            entry = getattr(self,attname)
        except AttributeError:
            return None

        if not att:
            if hasattr(entry,'GetValue'): 
                return entry.GetValue()
            else:
                return entry.GetSize()
        else:
            try:
                return entry.GetAttributes()[att].toscalar()
            except (AttributeError,KeyError):
                try: 
                    return entry.GetAttributes()[att]
                except KeyError:
                    return None

    def GetAttrName(self,variable):
        return 'nc' + variable

    def DeleteNetCDFEntry(self,variable):
        
        # get the attribute name to look for
        attname = self.GetAttrName(variable)
        try:
            delattr(self,attname)
        except AttributeError:
            pass

        self.ready = False
    
    def WriteListOfAtoms(self):
        """Method to write ListOfAtoms data to a netcdffile.   
	No writing is done, this is done by the WriteAsNetCDF method.  """
   
        from Dacapo import NetCDF 

	atoms = self.GetListOfAtoms()
        
        ## write positions !! Cartesian-> Scaled
        attname = self.GetAttrName("DynamicAtomPositions")
        pos = atoms.GetCartesianPositions()
        invcell = LinAlg.inverse(atoms.GetUnitCell())
        scaledpos1 = num.matrixmultiply(pos,invcell)

        scaledpos = self.GetNetCDFEntry("DynamicAtomPositions")
        if scaledpos is not None:
            try: 
            	scaledpos[-1] = scaledpos1
	    except ValueError: 
                # stop dacapo since the number of atoms has been changed
                self.StopDacapo()
                scaledpos = [num.array(scaledpos1)]
		# also remove atoms, wavefunction chargedensity
                self.DeleteNetCDFEntry('DynamicAtomForces')
                self.DeleteNetCDFEntry('WaveFunctionFFTindex')
                self.DeleteNetCDFEntry('ChargeDensity')
                self.DeleteNetCDFEntry('WaveFunction')
                self.DeleteNetCDFEntry('StructureFactor')

        else:
            scaledpos = [num.array(scaledpos1)]
            
	self.SetNetCDFEntry("DynamicAtomPositions",
                            value=scaledpos,
	 		    dim=['number_of_dynamic_atoms'], 
                            series=True) 
 
        unitcell = self.GetNetCDFEntry('UnitCell')
        if unitcell is not None:
            unitcell[-1] = atoms.GetUnitCell()
        else:
            unitcell     = [atoms.GetUnitCell()] 
            
        # write unitcell
        self.SetNetCDFEntry("UnitCell",value=unitcell,series=True ) 

        # write DynamicAtomSpecies for each atom
        dynnames=[]
        for atom in atoms:
            # length of chemicalsymbol must 2 char!
            dynnames.append("%2s" % atom.GetChemicalSymbol())

        self.SetNetCDFEntry("DynamicAtomSpecies",value=dynnames,
                            dim=['number_of_dynamic_atoms'])

        # write tags for atoms, using AtomsTag
        # default 0 
        tags = [atom.GetTag() for atom in atoms] 
        self.SetNetCDFEntry("AtomTags",value=tags,
                            dim=['number_of_dynamic_atoms'])

        ##write magneticmoments
        moments=[]
        thereismoments=0
        for atom in atoms:
            try:
                moments.append(atom.GetMagneticMoment())
                thereismoments=1
            except AttributeError:
                 moments.append(0)
        if thereismoments:
            self.SetNetCDFEntry("InitialAtomicMagneticMoment",value=moments,
                                 dim=['number_of_dynamic_atoms'])

        # write pseudopotential information for each atom
        for atom in atoms:
            symbol = atom.GetChemicalSymbol()
            entryname = 'AtomProperty_' + symbol
            Z = atom.GetAtomicNumber()
            pppath = self.GetPseudoPotential(Z)
	    self.SetNetCDFEntry(entryname,att="PspotFile", 
                                value = pppath) 

    def ReadListOfAtoms(self,index=None):
        """Method to read data from the NetCDF 
        entries attached to the calculator
	into the ListOfAtoms attached to the calculator. 
        The NetCDF entries attached to the calculator are 
        updated. 
        """
	from string import strip,join
        from ASE import Atom,ListOfAtoms

	if index==None:# use the last configuration as default
	    index=-1

	try:  # see if atoms are defined
	    atoms = self.GetListOfAtoms()
        except AttributeError: 
            # create a empty list of atoms 
            atoms = ListOfAtoms([]) 
            
	#read unitcell
	unitcell=self.GetNetCDFEntry("UnitCell")[index]
	atoms.SetUnitCell(unitcell,fix=True)

	## read atoms
	names=self.GetNetCDFEntry("DynamicAtomSpecies")

	# Read the pseudopotentials used in the ncfile
	for name in names:
	    symbol = name.tostring().strip()
	    ncvar = "AtomProperty_%s" % symbol
	    ppot = self.GetNetCDFEntry(ncvar,att="PspotFile")
	    Z=Atom(symbol).GetAtomicNumber()
	    self.SetPseudoPotential(Z,ppot)
	    
	## take only as many atoms as in file
	# del atoms[names.shape[0]:]
	
	## update types 
	for i in range(len(atoms)):
	    if strip(join(names[i].tostring(),''))!=atoms[i].GetChemicalSymbol():
	        atoms[i].SetChemicalSymbol(strip(join(names[i].tostring(),'')))
		    
	## create new atoms
	for i in range(len(atoms),names.shape[0]):
	    atoms.append(Atom(symbol=names[i].tostring().strip()))

       ##read positions
        try:
            scaled=self.GetNetCDFEntry("DynamicAtomPositions")[index]
            # Scaled -> Cartesian
            pos = num.matrixmultiply(scaled,unitcell)
            atoms.SetCartesianPositions(pos)
        except KeyError:
            pass

        ##read partial charges
        # ptch=self.GetNetCDFEntry("PartialCharge")
        # if ptch is not None: 
        #     for i in range(len(atoms)):
        #         atoms[i].SetPartialCharge(ptch[i])

        ##read moments
        mgm=self.GetNetCDFEntry("InitialAtomicMagneticMoment")
        if mgm is not None: 
            for i in range(len(atoms)):
                atoms[i].SetMagneticMoment(mgm[i])
 
        try: 
	    self.GetListOfAtoms() 
	except AttributeError: 
            atoms.SetCalculator(self)

        # read AtomTags 
        tags = self.GetNetCDFEntry("AtomTags")
	if tags is not None: 
            for i in range(len(atoms)): 
                atoms[i].SetTag(tags[i])

    def SetAtomsCounter(self,counter):
        self._atomscounter = counter

    def GetAtomsCounter(self):
        return self._atomscounter

    def GetZIBlochMatrix(self): 
	""" Calculate the localization matrix for wannier functions """ 
        from DacapoWannier import DacapoWannier 
        self.StopDacapo()
        return DacapoWannier(self).GetZIBlochMatrix() 

    def _AtomsHasChanged(self):
        """ check if anything besides the positions in the listofatoms
        has changed """
        import copy

        atoms = self.GetListOfAtoms()
        returnvalue = False
        
        if hasattr(self,'atomsinfo'):
            # compare

            unitcell = atoms.GetUnitCell()
            if min(min(unitcell==self.atomsinfo['unitcell']))==0:
                self.DeleteNetCDFEntry('softgrid_dim1')
                self.DeleteNetCDFEntry('softgrid_dim2')
                self.DeleteNetCDFEntry('softgrid_dim3')
                self.DeleteNetCDFEntry('hardgrid_dim1')
                self.DeleteNetCDFEntry('hardgrid_dim2')
                self.DeleteNetCDFEntry('hardgrid_dim3')
                returnvalue = True

            if len(atoms)!=len(self.atomsinfo['atomicnumbers']):
                returnvalue = True
                self.DeleteNetCDFEntry('DynamicAtomPositions')
                self.DeleteNetCDFEntry('DynamicAtomSpecies')
                self.DeleteNetCDFEntry('UnitCell')
                self.DeleteNetCDFEntry('DynamicAtomForces')
                self.DeleteNetCDFEntry('EvaluateTotalEnergy')
                self.DeleteNetCDFEntry('EvaluateExchangeEnergy')
                self.DeleteNetCDFEntry('EvaluateCorrelationEnergy')
                self.DeleteNetCDFEntry('FermiLevel')
                self.DeleteNetCDFEntry('StructureFactor')
            else: 
                if min(atoms.GetAtomicNumbers()==self.atomsinfo['atomicnumbers'])==0:
                    returnvalue = True

        
        # update atomsinfo
        self.atomsinfo = {}
        self.atomsinfo.update({'unitcell': copy.copy(atoms.GetUnitCell())})
        self.atomsinfo.update({'atomicnumbers': copy.copy(atoms.GetAtomicNumbers())})

        return returnvalue

    def StopDacapo(self):
        """ Stop dacapo and wait until is has finished writing the netcdf file """
        if self._dacapo_is_running == 'true':
            self.ExecuteExternalDynamics(stopprogram='true')
            # atoms = Dacapo.ReadAtoms(self.ncfile,calc=self)

        self._dacapo_is_running ='false'
        # self.restart_dacapo = False

    def GetElectronicStates(self):
        """ return a dacapo ElectronicStates object
        """
        from Dacapo.ElectronicStates import ElectronicStates
	
	if not hasattr(self,'loe'):
            loe = ElectronicStates()
            loe.InitializeFromCalculator(self)
	    self.loe=loe
        return self.loe
    
    def SetSerialExecutable(self,executable):
        '''sets the binary executable for serial dacapo calculations

        overwrites the default executable that should be set in your
        .bashrc or .cshrc file
        '''
        os.environ['DACAPOEXE_SERIAL']=executable

    def GetSerialExecutable(self):
        return os.environ.get('DACAPOEXE_SERIAL')
        
    def SetParallelExecutable(self,executable):
        '''sets the binary executable for parallel dacapo calculations

        overwrites the default executable that should be set in your
        .bashrc or .cshrc file
        '''
        os.environ['DACAPOEXE_PARALLEL']=executable

    def GetParallelExecutable(self):
        return os.environ.get('DACAPOEXE_PARALLEL')

    def SetRunScript(self,runscript='dacapo.run'):
        '''
        sets the executable script that will be used to run a calculation.

        runscript should be an executable (binary or script) that has
        these features:
        1. It takes all the arguments that the binary dacapo executable
           takes:
           yourscript netCDF_input_file [netCDF_output_file] [-innc netCDF_input_file] [-outnc netCDF_out_file] [-out ASCII_output] [-scratch scratch_dir ] [-stop stop_file_name]

           should run a dacapo calculation with these arguments
           
        2. If it runs the parallel dacapo executable it should launch
           the parallel environment, such as lamd

        3. The script should ideally return 0 if the calcualtion was
           ok and something not zero if any errors were detected.

        the default script is dacapo.run which is located in
        Dacapo/Tools. It automatically detects parallel environments
        for PBS, Sun grid engine and the LoadLeveler queue systems
        and runs the parallel executable if it finds them or the
        serial executable otherwise.

        here is a minimal example to always use a file called
        mynodefile that contains the nodes to run your job on with
        your own specially compiled dacapo mpi executable.

        #!/bin/tcsh -fx
        set nprocs = `wc -l mynodefile`
        recon mynodefile
        lamboot mynodefile
        mpirun -O -np $nprocs dacapo-2.7.5_lam706_special.run $argv
        lamhalt
        #end

        see dacapo.run for a more sophisticated example
        '''
        self.runscript = runscript

    def GetRunScript(self):
        return self.runscript

 
    def SetExecutable(self,executable):
        '''
        sets executable that is run when a dacapo calculation is run

        it must be an executable object either on your path or an
        absolute path. the executable should take all the arguments
        dacapo takes, and if it is a parallel job, then the executable
        should launch lamd or whatever parallel lib you use. Currently
        this is all managed in the runscript dacapo.run.

        See the SetRunScript method above, this is now the preferred
        way to do this.

        05/26/05 jrk
        '''
        self.SetRunScript(executable)
        self.GetJobType().SetExecutable(executable)
        self.ready = False

    def GetExecutable(self):
        return self.GetJobType().GetExecutable()

    def GetValenceElectrons(self):
	'''
	Returns the number of valence electrons from the psp files.
	'''
	from Dacapo.PseudoPotentials import GetValenceElectrons
	valelec = 0
	for atom in self.GetListOfAtoms():
	    psp = self.GetPseudoPotential(atom.GetAtomicNumber())
	    valelec += GetValenceElectrons(psp)
	return valelec


    def SetKeywords(self,keywords):
	'''
	keywords should be a string of keywords

	'fcc 2x2 test'

	you can delimit however you want, it depends on what you want
	to do with these later. These are intended for use in searches
        for example combined with regexp to find files with particular
        keywords.
	'''
	self.SetNetCDFEntry('Keywords',value=str(keywords))

    def GetKeywords(self):
        '''
        returns keywords from the netcdf file
        '''
	k = self.GetNetCDFEntry('Keywords')
	from string import strip,join

	k = strip(join(k,''))

        return k

    def InitialWannier(self,data,atoms,kpointgrid):
        from InitialWannier import InitialWannierFunctions
        griddims = (self.GetNetCDFEntry('softgrid_dim1'),
                   self.GetNetCDFEntry('softgrid_dim2'),
                   self.GetNetCDFEntry('softgrid_dim3'))
        
        fftindex = self.GetNetCDFEntry('WaveFunctionFFTindex') 
                   
	return InitialWannierFunctions(data,griddims,atoms,fftindex,kpointgrid)

    def GetWannierLocalizationMatrix(self,G_I,nbands,dirG=0,kpoint=0,nextkpoint=0,spin=0):
        from DacapoWannier import DacapoWannier
        
        if not hasattr(self,'wannier'):
            self.wannier = DacapoWannier(self,nbands,spin)
	if self.wannier.GetSpin()!=spin:  
            self.wannier = DacapoWannier(self,nbands,spin)

        return self.wannier.GetZIBlochMatrix(dirG,kpoint,nextkpoint,G_I)


    def SetSoftGridDimension(self,dims):
        self.RestartDacapo()
        n = 1
        for dim in dims: 
            grid1=num.zeros([dim],num.Float)
            self.SetNetCDFEntry('dummy_soft'+str(n),
                                value=grid1,
                                dim=['softgrid_dim'+str(n)])
            n += 1
   
    def SetHardGridDimension(self,dims):
        self.RestartDacapo()
        n = 1
        for dim in dims: 
            grid1=num.zeros([dim],num.Float)
            self.SetNetCDFEntry('dummy_hard'+str(n),
                                value=grid1,
                                dim=['hardgrid_dim'+str(n)])
            n += 1


    def GetFFTIndexArray(self):
        self.Calculate()
        fftindex = self.GetNetCDFEntry('WaveFunctionFFTindex')
        return fftindex

    def GetHSMatrix(self,all=False):

        # set non consistent flag
        self.SetNetCDFEntry("ElectronicMinimization")
        self.SetNetCDFEntry("ElectronicMinimization",
                            att='Method',
                            value="NonSelfconsistent")

        # do not remove wavefunctions
        self.restart_dacapo = False
        self.Calculate()

        hmat1=self.GetNetCDFEntry("HMatrix")
        smat1=self.GetNetCDFEntry("SMatrix")

        N = len(hmat1[:,0,0])
        hmat = num.zeros((N,N),num.Complex)
        smat = num.zeros((N,N),num.Complex)
        hmat.real = hmat1[:,:,0]
        hmat.imag = hmat1[:,:,1]
        smat.real = smat1[:,:,0]
        smat.imag = smat1[:,:,1]

        if all:
           hartree = self.GetNetCDFEntry('HartreeMatrix')
           xc = self.GetNetCDFEntry('XCMatrix')
           hartree_mat = num.zeros((N,N),num.Complex)
           xc_mat = num.zeros((N,N),num.Complex)
           hartree_mat.real = hartree[:,:,0]
           hartree_mat.imag = hartree[:,:,1]
           xc_mat.real = xc[:,:,0]
           xc_mat.imag = xc[:,:,1]


        if all:
           return hmat,smat,hartree_mat,xc_mat
        else:
           return hmat,smat



        # clean up
        self.DeleteNetCDFEntry("ElectronicMinimization")

        return hmat,smat


class Calculator(Dacapo):
    """ alias for Dacapo """
    pass

