from ASE.Utilities import VectorSpaces,Vector
from ASE.Utilities.VectorSpaces import BravaisLattice
from Dacapo import FunctionSpaces
from ASE.Utilities import ArrayTools
import Numeric
import LinearAlgebra


class Operator:

	def __init__(self,space,representation=None):
		self.SetSpace(space)
		if representation is not None:
			self.SetRepresentation()

	def SetSpace(self,space):
		self.Space=space

	def GetSpace(self):
		return self.Space	

	def SetRepresentation(self,representation):
		self.representation=representation

	def GetRepresentation(self):
		return self.representation

	def UnregisterRepresentation(self):
		# See, if there is something to remove
		if hasattr(self,'representation'):
			delattr(self,'representation')

	def Operate(self,state):
		return self.GetSpace().Operate(state=state,operatorrepresentation=self.GetRepresentation())

	def InnerProduct(self,state1,state2):
		return self.GetSpace().InnerProduct(state1=state1,operatorrepresentation=self.GetRepresentation(),state2=state2)
		

class FunctionSpaceOperator(Operator):

	def __init__(self,space,unitcell=None):
		Operator.__init__(self,space=space)
		if unitcell is not None:
			self.SetUnitCell(unitcell)

	def __add__(self,other):
		"""Reimplemented from Operator"""
		if self.GetSpace()==other.GetSpace() and self.GetUnitCell()==other.GetUnitCell():
			# SaveMemory is always ON, so that parts do not
			# have a pointer to representation
			self.SaveMemoryOn()
			other.SaveMemoryOn()
			return OperatorCollection(parts=(self,other),function=Numeric.add,space=self.GetSpace(),unitcell=self.GetUnitCell())
		else:
			raise ValueError, "__add__ not defined for this operation"

	def __radd__(self,other):
		# Addition is commutative
		return self+other

	def __mul__(self,other):
		# Implementing the action of an operator:
		if type(other) is type(Numeric.array([])):
			return self.Operate(other)
		# Implementing multiplication by a constant
		elif type(other) in [type(1),type(1.0),type(1.0j)]:
			# The part does not keep a pointer to representation
			self.SaveMemoryOn()
			newoperator=OperatorCollection(parts=[self],space=self.GetSpace(),unitcell=self.GetUnitCell())
			newoperator.SetWeights([other])
			return newoperator
		else:
			raise ValueError, "__mul__ not defined for this opration"

	def __rmul__(self,other):
		# Multiplication is commutative:
		return self*other

	def SetUnitCell(self,unitcell):
		self.unitcell=unitcell

	def GetUnitCell(self):
		if not hasattr(self,"unitcell"):
			# The default space is an orthogonal unitcell
			# with unit length between neighboring grid points.
			self.unitcell=VectorSpaces.VectorSpaceWithBasis()
			dimensions=self.GetSpace().GetDimensions()
			basis=Numeric.identity(len(dimensions))*Numeric.array(dimensions,Numeric.Float)
			self.unitcell.SetBasis(basis)
		return self.unitcell

	def GetVolumeOfGridUnitCell(self):
		# Calculate the volume of the grid unit cell
		# Volume of unit cell:
		volume=self.GetUnitCell().GetVolumeOfCell()
		# Number of cells:
		Ncells=Numeric.multiply.reduce(self.GetSpace().GetDimensions())
		# Using float to avoid integer division
		return float(volume)/Ncells

	def SaveMemory(self):
		# Specifying default
		if not hasattr(self,'savemem'):
			self.SaveMemoryOn()
		return self.savemem	

	def SaveMemoryOn(self):
		self.savemem=1

	def SaveMemoryOff(self):
		self.savemem=0

	def GetRepresentation(self):
		"""Reimplemented from Operator"""
		# If the representation is not set: Try to update
		if not hasattr(self,'representation'):
			self.UpdateRepresentation()
		representation=Operator.GetRepresentation(self)
		# If SaveMemory: Do not keep representation
		if self.SaveMemory():
			self.UnregisterRepresentation()
		return representation

	def UpdateRepresentation(self,representation):
		# Scaling the representation according to the unit cell
		# Using integral dr = volume_cell integral dx_i,
		# where x_i are the coordinates of the space,
		# i.e. unit length between grid points
		# NOTE: volume_cell = det(Jacobian)
		volume_cell=self.GetVolumeOfGridUnitCell()
		# Multiplying with det(Jacobian)
		Numeric.multiply(representation,volume_cell,representation)
		self.SetRepresentation(representation)

	def HasInnerProductImplemented(self):
		# Does operator has inner product implemented in space ?
		return 0


	def InnerProduct_Implemented(self,innerproduct):
		# If the inner product is implemented it is carried by the
		# function space and carried out in the coordinates of the
		# space. Hence the transformation is necessary:
		# integral dr = integral det(Jacobian) dx =
		# grid_volume * integral dx, x: coordinates function space
		volume_cell=self.GetVolumeOfGridUnitCell()
		innerproduct=volume_cell*innerproduct
		return innerproduct

	def Operate(self,state):
		# An operator works as: Ax|psi> = O^{-1} <i|A|j> psi,
		# where O=<i|1|j> is the overlap
		newstate=Operator.Operate(self,state)
		# Diving with the normalization of the unitcell
		Numeric.divide(newstate,self.GetVolumeOfGridUnitCell(),newstate)
		return newstate

class OperatorCollection(FunctionSpaceOperator):

	def __init__(self,space,unitcell=None,parts=None,function=None,weights=None):
		if parts is not None:
			self.SetParts(parts)
		if function is not None:
			self.SetFunction(function)
		if weights is not None:
			self.SetWeights(weights)
		if unitcell is not None:
			self.SetUnitCell(unitcell)
		FunctionSpaceOperator.__init__(self,space=space)

	def SetParts(self,parts):
		self.parts=list(parts)
		# This will force an update using GetRepresentation
		self.UnregisterRepresentation()

	def GetParts(self):
		# Default is an empty tuple
		if not hasattr(self,"parts"):
			self.parts=()
		return tuple(self.parts)

	def AddPart(self,part):
		self.parts.append(part)
		# This will force an update using GetRepresentation
		self.UnregisterRepresentation()

	def RemovePart(self,part):
		self.parts.remove(part)
		# This will force an update using GetRepresentation
		self.UnregisterRepresentation()

	def SetSpace(self,space):
		# Space is set
		FunctionSpaceOperator.SetSpace(self,space)
		# Propagating the space to parts
		for part in self.GetParts():
			part.SetSpace(space)
		# Force an update next time GetRepresentation is wanted
		self.UnregisterRepresentation()

	def SetUnitCell(self,unitcell):
		# Unit cell is set:
		FunctionSpaceOperator.SetUnitCell(self,unitcell)
		# Propagating the unit cell to the parts:
		for part in self.GetParts():
			part.SetUnitCell(unitcell)
		# Force an update next time representation is wanted:
		self.UnregisterRepresentation()

	def SetWeights(self,weights):
		self.weights=weights
		# Force an update next time GetRepresentation is used
		self.UnregisterRepresentation()

	def GetWeights(self):
		if not hasattr(self,"weights"):
			# Default: All parts are weighted equally
			return tuple(len(self.GetParts())*[1])
		return tuple(self.weights)

	def SetFunction(self,function):
		self.function=function
		# This will force an update using GetRepresentation
		self.UnregisterRepresentation()

	def GetFunction(self):
		return self.function

	def UpdateRepresentation(self):
		parts=self.GetParts()
		weights=self.GetWeights()

		# Finding representation of 1st. part:
		# If weight is 1: No need to multiply
		if weights[0]==1:
			representation=parts[0].GetRepresentation()
		else:
			representation=weights[0]*parts[0].GetRepresentation()
		# Looping over the remaining parts
		for i in range(len(self.GetParts()))[1:]:
			function=self.GetFunction()
			if weights[i]==1:
				representation=function(representation,parts[i].GetRepresentation())
			else:
				representation=function(representation,weights[i]*parts[i].GetRepresentation())
		FunctionSpaceOperator.SetRepresentation(self,representation)

	def HasInnerProductImplemented(self):
		# Returns 1 only if all parts have been implemented in space
		parts=self.GetParts()
		parts_implemented=map(lambda operator:operator.HasInnerProductImplemented(),parts)
		if parts_implemented==len(parts)*[1]:
			return 1
		else:
			return 0

	def InnerProduct_Implemented(self,state1,state2):
		parts=self.GetParts()
		weights=self.GetWeights()

		# If weight is 1: No need to multiply
		if weights[0]==1:
			innerproduct=parts[0].InnerProduct(state1,state2)
		else:
			innerproduct=weights[0]*parts[0].InnerProduct(state1,state2)
		# Looping over remaining parts		
		for i in range(len(self.GetParts()))[1:]:
			function=self.GetFunction()
			# If weight is 1: No need to multiply
			if weights[i]==1:
				innerproduct=function(innerproduct,parts[i].InnerProduct(state1,state2))
			else:
				innerproduct=function(innerproduct,weights[i]*parts[i].InnerProduct(state1,state2))
		return innerproduct

	
	def InnerProduct(self,state1,state2):
		if self.HasInnerProductImplemented():
			return self.InnerProduct_Implemented(state1,state2)
		else:	
			return FunctionSpaceOperator.InnerProduct(self,state1,state2)
		
class NonLocalOperatorCollection(OperatorCollection):
 	
	def UpdateRepresentation(self):
		allparts=self.GetParts()
		parts=[]
		for part in allparts:
			if part.NeedUpdate()==1:
				parts.append(part)
		# Finding representation of 1st. part:
		if parts!=[]:
			representation=parts[0].GetRepresentation()
		else:
			# Dummy value
			representation=allparts[0].GetRepresentation()
			
		# Looping over the remaining parts
		for i in range(len(parts))[1:]:
			function=self.GetFunction()
			#representation=function(representation,parts[i].GetRepresentation())
			representation=Numeric.add(representation,parts[i].GetRepresentation(),representation)
		FunctionSpaceOperator.SetRepresentation(self,representation)


	def InnerProduct_Implemented(self,state1,state2):
		allparts=self.GetParts()
		parts=[]
		for part in allparts:
			if part.NeedUpdate()==1:
				parts.append(part)
	
		innerproduct=parts[0].InnerProduct(state1,state2)
		
		# Looping over remaining parts		
		for i in range(len(parts))[1:]:
			function=self.GetFunction()
			innerproduct=function(innerproduct,parts[i].InnerProduct(state1,state2))
			
		return innerproduct
	
	
class Identity(FunctionSpaceOperator):

	def SetSpace(self,space):
		FunctionSpaceOperator.SetSpace(self,space)
		self.UnregisterRepresentation()
		
	def UpdateRepresentation(self):
		FunctionSpaceOperator.UpdateRepresentation(self,self.GetSpace().GetIdentityRepresentation())

	def HasInnerProductImplemented(self):
		# If space has the method Identity_InnerProduct return 1
		if hasattr(self.GetSpace(),"Identity_InnerProduct"):
			return 1
		else:
			return 0

	def InnerProduct(self,state1,state2):
		# Does a direct implementation exist ? 
		if self.HasInnerProductImplemented(): # Yes, use it
			innerproduct=self.GetSpace().Identity_InnerProduct(state1,state2)
			# Scaling with unit cell volume
			return FunctionSpaceOperator.InnerProduct_Implemented(self,innerproduct)
		else: # Use the default
			innerproduct=FunctionSpaceOperator.InnerProduct(self,state1,state2)
		return innerproduct	
		     

class ScaledGradient(FunctionSpaceOperator):

	def SetSpace(self,space):
		FunctionSpaceOperator.SetSpace(self,space)
		self.UnregisterRepresentation()

	def _ScaledGradientFromGradientGridUnitLength(self,matrix):
		# Matrix is assumed to have the form: matrix(i,a)
		# where i corresponds to the dimensions of the space
		dimensions=self.GetSpace().GetDimensions()
		for i in range(len(dimensions)):
			matrix[i]=matrix[i]*dimensions[i]
		return matrix	

	def UpdateRepresentation(self):
		space=self.GetSpace()
		dimensions=space.GetDimensions()
		scaledgradient=space.GetGradientRepresentation()
		FunctionSpaceOperator.UpdateRepresentation(self,self._ScaledGradientFromGradientGridUnitLength(scaledgradient))

	def HasInnerProductImplemented(self):
		# If space has the method Gradient_InnerProduct return true
		if hasattr(self.GetSpace(),"Gradient_InnerProduct"):
			return 1
		else:
			return 0

	def InnerProduct_Implemented(self,state1,state2):
		# Calculating the inner product via implementation in space
		innerproduct=self.GetSpace().Gradient_InnerProduct(state1,state2)
		# Scaling with unit cell volume
		innerproduct=FunctionSpaceOperator.InnerProduct_Implemented(self,innerproduct)
		return self._ScaledGradientFromGradientGridUnitLength(innerproduct)

	def InnerProduct_Default(self,state1,state2):
		# Calculating the inner product from the representation
		innerproduct=[]
		# Looping over components of the gradient
		for representation in self.GetRepresentation():
			newinnerproduct=self.GetSpace().InnerProduct(state1=state1,operatorrepresentation=representation,state2=state2)
			innerproduct.append(newinnerproduct)
		return Numeric.array(innerproduct)

	def InnerProduct(self,state1,state2):
		dimensions=self.GetSpace().GetDimensions()
		# Does a direct implementation exist ?
		if self.HasInnerProductImplemented(): # Yes, use it
			return self.InnerProduct_Implemented(state1,state2)
		else: # No use the default
			return self.InnerProduct_Default(state1,state2)

class CartesianGradient(ScaledGradient):

	def GetUnitCell(self):
		if not hasattr(self,"unitcell"):
			# The default space is a orthogonal unitcell
			# with unit length between neighboring grid points.
			self.unitcell=VectorSpaces.VectorSpaceWithBasis()
			dimensions=self.GetSpace().GetDimensions()
			basis=Numeric.identity(len(dimensions))*Numeric.array(dimensions,Numeric.Float)
			self.unitcell.SetBasis(basis)
		return self.unitcell

	def SetUnitCell(self,unitcell):
		self.unitcell=unitcell

	def _CartesianGradientFromScaledGradient(self,scaledmatrix):
		# **NOTE** : Scaled matrix must be : (i,dims)
		# where i refers to one of the scaled dimensions.

		# Calculating cartesian gradient as:
		# grad_r=sum_i grad_i a^i, where a^i is the reciprocal vector i

		# The inverse basis, a^i=invbasis[i]
		invbasis=Numeric.transpose(LinearAlgebra.inverse(self.GetUnitCell().GetBasis()))
		# Contribution due to reciprocal vector: a^0=invbasis[0]
		cartesianmatrix=Numeric.multiply.outer(invbasis[0],scaledmatrix[0])
		# Remaining contributions: a^i=invbasis[i]
		for i in range(1,len(invbasis)):
			Numeric.add(cartesianmatrix,Numeric.multiply.outer(invbasis[i],scaledmatrix[i]),cartesianmatrix)
		return cartesianmatrix

	def UpdateRepresentation(self):
		# **NOTE** : The representation attribute is in scaled coor.
		# First update the scaled gradient
		ScaledGradient.UpdateRepresentation(self)
		scaledrep=self.GetRepresentation()
		# and translate the result to Cartesian coordinates
		FunctionSpaceOperator.UpdateRepresentation(self,self._CartesianGradientFromScaledGradient(scaledrep))

	def InnerProduct_Default(self,state1,state2):
		# This method is to be used if it is not implemented in space
		innerproduct=[]
		# Looping over components of the gradient
		for representation in self.GetRepresentation():
			newinnerproduct=self.GetSpace().InnerProduct(state1=state1,operatorrepresentation=representation,state2=state2)
			innerproduct.append(newinnerproduct)
		return Numeric.array(innerproduct)	

	def InnerProduct(self,state1,state2):
		# Does a fast implementation exist for scaled gradient ?
		if self.HasInnerProductImplemented(): # Yes, use it
			innerproduct_scaled=ScaledGradient.InnerProduct_Implemented(self,state1,state2)
			# Scaling with volume of unit cell
			innerproduct=FunctionSpaceOperator.InnerProduct_Implemented(self,innerproduct)
			return self._CartesianGradientFromScaledGradient(innerproduct_scaled)
		else: # No use, the default
			return self.InnerProduct_Default(state1,state2)

class Laplacean(FunctionSpaceOperator):

	def __init__(self,space,unitcell=None):
		if unitcell is not None:
			self.SetUnitCell(unitcell)
		FunctionSpaceOperator.__init__(self,space)	

	def SetSpace(self,space):
		FunctionSpaceOperator.SetSpace(self,space)
		# Force an update when GetRepresentation is used
		self.UnregisterRepresentation()

	def GetMetric(self):
		# array: Avoid the original unit cell from being manipulated
		unitcell=Numeric.array(self.GetUnitCell().GetBasis())
		dimensions=self.GetSpace().GetDimensions()
		# Scaling the unit cell to represent the unit cell of
		# the grid points
		for i in range(len(dimensions)):
			unitcell[i]=unitcell[i]/dimensions[i]
		# Calculating metric
		metric=Numeric.matrixmultiply(unitcell,Numeric.transpose(unitcell))
		return metric

	def UpdateRepresentation(self):
		metric=self.GetMetric()
		FunctionSpaceOperator.UpdateRepresentation(self,self.GetSpace().GetLaplaceRepresentation(metric=metric))
		
	def HasInnerProductImplemented(self):
		# If space has the method Laplace_InnerProduct return true
		if hasattr(self.GetSpace(),"Laplace_InnerProduct"):
			return 1
		else:
			return 0

	def InnerProduct_Implemented(self,state1,state2):
		innerproduct=self.GetSpace().Laplace_InnerProduct(state1=state1,state2=state2,metric=self.GetMetric())
		# Scaling with unit cell volume
		return FunctionSpaceOperator.InnerProduct_Implemented(self,innerproduct)


	def InnerProduct(self,state1,state2):
		# Does the space have an implementation for the inner product
		if self.HasInnerProductImplemented():   # Yes, use it
			return self.InnerProduct_Implemented(state1,state2)
		else: # No, use the default
			return FunctionSpaceOperator.InnerProduct(self,state1,state2)


class LocalPotential(FunctionSpaceOperator):

	def __init__(self,space,unitcell=None,realspacepotential=None):
		if realspacepotential is not None:
			self.Potential=realspacepotential
		FunctionSpaceOperator.__init__(self,space=space,unitcell=unitcell)

	def SetSpace(self,space):
		FunctionSpaceOperator.SetSpace(self,space)
		self.UnregisterRepresentation()

	def GetRealSpacePotential(self):
		return self.Potential

	def SetRealSpacePotential(self,realspacepotential):
		self.Potential=realspacepotential
		# This will force an update when GetRepresentation
		self.UnregisterRepresentation()

	def UpdateRepresentation(self):
		FunctionSpaceOperator.UpdateRepresentation(self,self.GetSpace().GetLocalPotentialRepresentation(self.GetRealSpacePotential()))		

	def HasInnerProductImplemented(self):
		# If space has the method Laplace_InnerProduct return true
		if hasattr(self.GetSpace(),"LocalPotential_InnerProduct"):
			return 1
		else:
			return 0

	def InnerProduct_Implemented(self,state1,state2):
		innerproduct=self.GetSpace().LocalPotential_InnerProduct(state1=state1,state2=state2,potential=self.GetRealSpacePotential())
		# Scaling with unit cell volume
		return FunctionSpaceOperator.InnerProduct_Implemented(self,innerproduct)

	def InnerProduct(self,state1,state2):
		# Does the space have an implementation for the inner product
		if self.HasInnerProductImplemented():   # Yes, use it
			return self.InnerProduct_Implemented(state1,state2)
		else: # No, use the default
			return FunctionSpaceOperator.InnerProduct(self,state1,state2)


    
class NonLocalOperator(FunctionSpaceOperator):

    def __init__(self,space,matrixelements,unitcell,listofrealspaceprojectors1=None,listofrealspaceprojectors2=None,listofprojectorcoordinates1=None,listofprojectorcoordinates2=None):
        FunctionSpaceOperator.__init__(self,space=space,unitcell=unitcell)
        self.SetMatrixElements(matrixelements)
        if listofprojectorcoordinates1 is not None:
            self.SetListOfProjectorCoordinates1(listofprojectorcoordinates1)
        if listofprojectorcoordinates2 is not None:
            self.SetListOfProjectorCoordinates2(listofprojectorcoordinates2)
        if listofrealspaceprojectors1 is not None:
            self.SetListOfRealSpaceProjectors1(listofrealspaceprojectors1)
        if listofrealspaceprojectors2 is not None:
            self.SetListOfRealSpaceProjectors2(listofrealspaceprojectors2)

    def SetUnitCell(self,unitcell):
	"""Overloaded from FunctionSpaceOperator"""
	# Propagate change to identity operator:
	self.GetIdentityOperator().SetUnitCell(unitcell)
	# and set the unit cell
	FunctionSpaceOperator.SetUnitCell(self,unitcell)

    def GetListOfProjectorCoordinates1(self):
        if not hasattr(self,"listofprojectorcoordinates1"):
	# If no projector coordinates: Create them from realspace values
	    self.listofprojectorcoordinates1=self.GetSpace().CoordinatesFromFunctionValues(self.GetListOfRealSpaceProjectors2())
        return self.listofprojectorcoordinates1

    def SetListOfProjectorCoordinates1(self,listofprojectorcoordinates1):
        self.listofprojectorcoordinates1=listofprojectorcoordinates1
        
    def GetListOfProjectorCoordinates2(self):
        if not hasattr(self,"listofprojectorcoordinates2"):
	# If no projector coordinates: Create them from realspace values
		self.listofprojectorcoordinates2=self.GetSpace().CoordinatesFromFunctionValues()
        return self.listofprojectorcoordinates2

    def SetListOfProjectorCoordinates2(self,listofprojectorcoordinates2):
        self.listofprojectorcoordinates2=listofprojectorcoordinates2

    def GetListOfRealSpaceProjectors1(self):
        # First index is transport direction
        return self.listofrealspaceprojectors1
    
    def SetListOfRealSpaceProjectors1(self,listofrealspaceprojectors1):
        self.listofrealspaceprojectors1=listofrealspaceprojectors1

    def GetListOfRealSpaceProjectors2(self):
        # First index is transport direction
        return self.listofrealspaceprojectors2
    
    def SetListOfRealSpaceProjectors2(self,listofrealspaceprojectors2):
        self.listofrealspaceprojectors2=listofrealspaceprojectors2
        
    def SetMatrixElements(self,matrixelements):
        self.matrixelements=matrixelements

    def GetMatrixElements(self):
        return self.matrixelements

    def GetIdentityOperator(self):
	if not hasattr(self,"identity"):
		self.identity=Identity(space=self.GetSpace(),unitcell=self.GetUnitCell())
	return self.identity	
	
    def HasInnerProductImplemented(self):
        return 1

    def SetNeedUpdate(self,value):
        self.needupdate=value
	    
    def NeedUpdate(self):
        return self.needupdate

    def UpdateRepresentation(self):
	# Using that: <j|V_{nl}|i> = sum_{bb'} <j|b> V_{bb'} <b'|i>,
	# where <j| and |i> refer to the basis set of the space
	# Finding <j|b>:
	projectors1=self.GetListOfProjectorCoordinates1()
	# ... and <i|b>
        projectors2=self.GetListOfProjectorCoordinates2()
	# The scaling factor (det(Jacobian)) due to two transverse integrals:
	scaling=pow(self.GetVolumeOfGridUnitCell(),2)
	matrixelements=scaling*self.GetMatrixElements()

	# The projectors have the shape projectorsi(a,j) where a is the index
	# Transforming: projectors1(a,j)->projectors1(j,a)
	projectors1=FunctionSpaces.ReverseCyclicPermutation(projectors1)
	# Doing the product <j|b>V_{bb'}=projectors1(j,b)xV_{bb'}
	representation=Numeric.matrixmultiply(projectors1,matrixelements)
	# and finally: sum_{bb'} <j|b> V_{bb'} <b'|i> =
	#              representation(j,b')xprojectors2^{*}(b',i)
	representation=Numeric.matrixmultiply(representation,Numeric.conjugate(projectors2))
	# Reshaped to Block form
	FunctionSpaceOperator.SetRepresentation(self,FunctionSpaces.BlockFormFromMatrix(representation,self.GetSpace().GetDimensions()))

    def InnerProduct(self,state1,state2):
	# Using: <state1|V_{nl}|state2> =
	#        sum_{bb'} <state1|b> V_{bb'} <b'|state2>
	# NOTE: statei has the shape (a,i) where a: state index and i: basis

	identityoperator=self.GetIdentityOperator()
	# Transforming vectors to block form:
	projectors1=FunctionSpaces.BlockFormFromVector(self.GetListOfProjectorCoordinates1(),dimensions=self.GetSpace().GetDimensions())
	projectors2=FunctionSpaces.BlockFormFromVector(self.GetListOfProjectorCoordinates2(),dimensions=self.GetSpace().GetDimensions())
	# Doing the inner products:
	# <state1|b> of shape (a1,b)
	product1=identityoperator.InnerProduct(state1,projectors1)
	# <b'|state2> of shape (b',a2)
	product2=identityoperator.InnerProduct(projectors2,state2)
	# finally doing: <state1|b> V_{bb'} <b'|state2> =
	#                product1(a1,b)xV_{bb'}xproduct2(b',a2)
	return Numeric.matrixmultiply(product1,Numeric.matrixmultiply(self.GetMatrixElements(),product2))

class KineticEnergy(Laplacean):

	def __init__(self,space,unitcell=None,unitsystem=None):
		Laplacean.__init__(self,space=space,unitcell=unitcell)
		if unitsystem is not None:
			self.SetUnitSystem(unitsystem)

	def SetUnitSystem(self,unitsystem):
		self.units=unitsystem
		# This will force an update next GetRepresentation()
		self.UnregisterRepresentation()

	def GetUnitSystem(self):
		if not hasattr(self,"units"):
			self.units=Unit.AtomicUnits
		return self.units

	def UpdateRepresentation(self):
		units=self.GetUnitSystem()
		prefactor=-pow(units["hbar"],2)/2.0/units["m_e"]
		# First update the laplacean
		Laplacean.UpdateRepresentation(self)
		representation=Laplacean.GetRepresentation(self)
		# Multiply the laplacean with the prefactor
		Numeric.multiply(representation,prefactor,representation)
		self.SetRepresentation(representation)

	def InnerProduct(self,state1,state2):
		# Does the space have the laplacean implemented
		if self.HasInnerProductImplemented():   # Yes, use it
			innerproduct=Laplacean.InnerProduct_Implemented(self,state1,state2)
			# Convert laplacean to kinetic energy
			units=self.GetUnitSystem()
			prefactor=-pow(units["hbar"],2)/2.0/units["m_e"]
			Numeric.multiply(innerproduct,prefactor,innerproduct)
			return innerproduct
		else: # No, use the default
			return self.GetSpace().InnerProduct(state1=state1,operatorrepresentation=self.GetRepresentation(),state2=state2)


class KineticEnergyPeriodic(OperatorCollection):

	def __init__(self,space,kpoint=None,unitcell=None,unitsystem=None):
		OperatorCollection.__init__(self,space=space)
		# Init of unit cell **must** preceed operator initialization
		if unitcell is not None:
			self.unitcell=unitcell
		if kpoint is not None:
			self.SetKPoint(kpoint)
		if unitsystem is not None:
			self.SetUnitSystem(unitsystem)
		# Define operation for parts
		self.SetFunction(Numeric.add)
		# Finally initialization relevant operators. Operators will
		# not keep their representation.
		identity=Identity(space=space,unitcell=self.GetUnitCell())
		identity.SaveMemoryOn()
		gradient=ScaledGradient(space=space,unitcell=self.GetUnitCell())
		gradient.SaveMemoryOn()
		laplace=Laplacean(space=space,unitcell=self.GetUnitCell())
		laplace.SaveMemoryOn()
		# Finally setting the parts
		self.SetParts([identity,gradient,laplace])

	def GetUnitCell(self):
		if not hasattr(self,"unitcell"):
			# The default space is a orthogonal unit cell
			# with unit length between neighboring grid points.
			unitcell=VectorSpaces.BravaisLattice()
			dimensions=self.GetSpace().GetDimensions()
			basis=Numeric.identity(len(dimensions))*Numeric.array(dimensions,Numeric.Float)
			unitcell.SetBasis(basis)
			OperatorCollection.SetUnitCell(self,unitcell)
		return OperatorCollection.GetUnitCell(self)
		
	def SetUnitSystem(self,unitsystem):
		self.units=unitsystem
		# This will force an update when GetRepresentation
		self.UnregisterRepresentation()

	def GetUnitSystem(self):
		# Atomic units are default
		if not hasattr(self,"units"):
			self.units=Unit.AtomicUnits
		return self.units
		
	def SetKPoint(self,kpoint):
		self.kpoint=kpoint
		# This will force an update when GetRepresentation
		self.UnregisterRepresentation()

	def GetKPoint(self):
		if not hasattr(self,"kpoint"):
			ndim=len(self.GetSpace().GetDimensions())
			self.kpoint=Vector.Vector(ndim*[0.0])
		return self.kpoint

	def _GetGradientPrefactor(self):
		# Doing the scalar product: -2.0j * k dot grad
		# Using that:
		# k dot grad = (ki bi) dot (aj gradj)
		#            =  ki biaj gradj
		# where bi is the reciprocal lattice vectors and
		# and aj obey: ai dot aj = delta_ij, called inverse basis

		# Defining necessary quantities:
		kpoint=self.GetKPoint()
		unitcell=self.GetUnitCell()
		
		# Using: inv_basis=(A^-1)^T => inv_basis[j]=aj
		inv_basis=Numeric.transpose(LinearAlgebra.inverse(unitcell.GetBasis()))
		reciprocalspace=unitcell.GetReciprocalBravaisLattice()

		# Finding the k point coordinates of reciprocal lattice
		k_scaled=reciprocalspace.CoordinatesFromCartesianCoordinates(kpoint.GetCartesianCoordinates())

		# The scale matrix corresponds to biaj
		scalematrix=Numeric.matrixmultiply(reciprocalspace.GetBasis(),Numeric.transpose(inv_basis))

		# Finding ki biaj
		scalecoefficients=Numeric.matrixmultiply(k_scaled,scalematrix)
		# and finally multiplying with the factor -2.0j
		return -2.0j*scalecoefficients

	def _GetKineticEnergyPrefactor(self):
		# Calculating pow(hbar,2)/2m
		units=self.GetUnitSystem()
		return pow(units["hbar"],2)/2.0/units["m_e"]

	def UpdateRepresentation(self):
		unitcell=self.GetUnitCell()
		identity,gradient,laplace=self.GetParts()
		kpoint=self.GetKPoint()
		k_length=kpoint.Length()

		# Examining three operators individually
		
		# Laplace:
		representation=-laplace.GetRepresentation()
		if self.SaveMemory():
			laplace.UnregisterRepresentation()

		# Identity:
		if k_length!=0: 
			representation=Numeric.add(pow(k_length,2)*identity.GetRepresentation(),representation)
		# **IF** kpoint is not Gamma represention is cast to Float:
		# (Necessary since gradient contribution is imaginary)
		
			representation=Numeric.asarray(representation,Numeric.Complex)
			if self.SaveMemory():
				identity.UnregisterRepresentation()

		# Gradient:
		# Finding -2.0j*dot(k,gradient)
		# Temporary solution !!
		if k_length!=0:
			gradientrepresentation=gradient.GetRepresentation()
			if self.SaveMemory():
				gradient.UnregisterRepresentation()
				
			# Finding the prefactor for scaled gradient:
			gradientprefactor=self._GetGradientPrefactor()	
			for i in range(len(gradientprefactor)):
				# Only necessary if scalecoefficients[i]!=0
				if gradientprefactor[i]!=0:
					Numeric.add(representation,gradientprefactor[i]*gradientrepresentation[i],representation)

					
		# Including pow(hbar,2)/2m
		Numeric.multiply(representation,self._GetKineticEnergyPrefactor(),representation)
		# Finally setting the representation
		self.SetRepresentation(representation)

	def InnerProduct_Implemented(self,state1,state2):
		# Method to be used if all operators are implemented in space
		identity,gradient,laplacean=self.GetParts()

		# Evaluating 3 operators separately
		k_length=self.GetKPoint().Length()

		# Laplacean:
		innerproduct=-laplacean.InnerProduct(state1,state2)
		if self.SaveMemory():
			laplacean.UnregisterRepresentation()

		# Identity:
		if k_length!=0:
			Numeric.add(innerproduct,pow(k_length,2)*identity.InnerProduct(state1,state2),innerproduct)
			if self.SaveMemory():
				identity.UnregisterRepresentation()

		# Gradient:
		if k_length!=0:
			innerproduct_gradient=gradient.InnerProduct(state1,state2)
			if self.SaveMemory():
				gradient.UnregisterRepresentation()
			gradientprefactor=self._GetGradientPrefactor()
			for i in range(len(gradientprefactor)):
				# Only if prefactor!=0
				if gradientprefactor[i]!=0:
					Numeric.add(innerproduct,gradientprefactor[i]*innerproduct_gradient[i],innerproduct)
		# Finally including pow(hbar,2)/2m
		Numeric.multiply(innerproduct,self._GetKineticEnergyPrefactor(),innerproduct)
		return innerproduct

	def InnerProduct(self,state1,state2):
		if self.HasInnerProductImplemented():
			return self.InnerProduct_Implemented(state1,state2)
		else:
			return OperatorCollection.InnerProduct(self,state1,state2)
		

class Translation(FunctionSpaceOperator):

	def __init__(self,space,unitcell=None,translationvector=None):
		if translationvector is not None:
			self.SetTranslationVector(translationvector)
		FunctionSpaceOperator.__init__(self,space=space,unitcell=unitcell)
		
	def GetTranslationVector(self):
		return self.translation

	def SetTranslationVector(self,translationvector):
		self.translation=translationvector

	def GetTranslationCoordinates(self):
		unitcell=self.GetUnitCell()
		trans_cartes=self.GetTranslationVector().GetCartesianCoordinates()
		trans_coor=BravaisLattice(basis=unitcell).CoordinatesFromCartesianCoordinates(trans_cartes)
		return Numeric.multiply(trans_coor,self.GetSpace().GetDimensions())

	def Operate(self,state):
		return self.GetSpace().Translation_Operate(state=state,coordinates=self.GetTranslationCoordinates())
