#!/usr/bin/env python

from Dacapo import Dacapo
from ASE import Atom,ListOfAtoms
from ASE.Filters.HomogeneousStrain import HomogeneousStrain
from ASE.Dynamics.MDMin import MDMin
from ASE.Trajectories.NetCDFTrajectory import NetCDFTrajectory

bulk = ListOfAtoms([Atom('Al', (0, 0, 0))] )

b=4.0
bulk.SetUnitCell([(0,b/2,b/2),
		  (b/2,0,b/2),
		  (b/2,b/2,0)])

calc = Dacapo(kpts=(12,12,12),               
	      planewavecutoff=450,
	      nbands=6,
	      out='OptAl_nosymm.nc')

calc.CalculateStress()
bulk.SetCalculator(calc)

strain = HomogeneousStrain(bulk)
traj = NetCDFTrajectory('OptAl_nosymm.traj',bulk)
traj.Add('Stress')

dyn = MDMin(strain,dt=0.2,fmax=0.0062)
dyn.Attach(traj)

dyn.Converge()


'''
I like to think of stress in terms of GPa, but the
stress units from Dacapo are eV/A^3. The conversion
factor is 1 GPa = 160.21773 ev/A^3

0.0125 eV/A^3 ~ 2.0 GPa
0.0062 eV/A^3 ~ 1.0 GPa
0.0031 eV/A^3 ~ 0.5 GPa

Depending on the material, the total energy
changes by only a few meV below a maximum stress of
2 GPa, but this must be checked.

For this example, we use 1GPa as the maximum
stress.

Notes: the stress tensor is rather sensitive to the planewave cutoff
and kpt grid. It does converge with planewaves, but at much higher
cutoffs than the total energy does. With kpt-grids, it can oscillate
(with elements changing signs, magnitude by factors of 2, going from 0
to non-zero, etc...). This may also depend on your unit cell.

fcc Al at the experimental lattice constant
kpt (pw cutoff = 450 eV, no symmetry constraint)
grid energy  sxx     syy   szz    syz    sxz    sxy
8  -56.5104  0.35   0.35   0.35   2.04   2.04   2.04
9  -56.4776 -0.02  -0.02  -0.02   0.00   0.00   0.00
10 -56.5092  0.25   0.25   0.25  -1.37  -1.37  -1.37
12 -56.5054  0.18   0.18   0.18   0.71   0.71   0.71
13 -56.5029  0.28   0.28   0.28   0.00   0.00   0.00

note the energies look converged, but the stress components are not
really. But, for stresses less than 1 GPa, the total energy changes by
less than 1 meV by reducing the stresses, so this suggests the
experimental lattice constant is an excellent guess for Al! It is not
always this way.

finally, using symmetry reductions can do strange things. I think it
is best not to use it.

                #calcs    lc     total energy
MurnaghanEOS      8     4.048      -56.5076 (no symmetry)
filter w/symm     3     4.079      -56.5046
filter w/nosymm   3     see below  -56.5062

Most recent output from OptAl_nosymm.traj
Energy:  -56.5062473983

Unit Cell vectors (angstroms)
        x       y       z
a1 [-0.0142  2.0266  2.0266]
a2 [ 2.0266 -0.0142  2.0266]
a3 [ 2.0266  2.0266 -0.0142]

Atom, position (in x,y,z),      tag, rmsForce
   Al [ 0.0000  0.0000  0.0000]  0  0.000

Stress (GPa)
[0.871 0.449 0.449]
[0.449 0.871 0.449]
[0.449 0.449 0.871]

note the small diagonal elements of the unit cell! this is
because of the k-point problem with the stress tensor.
'''


