#!/usr/bin/env python

""" h2o dissociation example 

The reaction path is mapped out for removing a H atom 
for water, 
The FixBondLength filter are use.
A trajectory with the minimum energy path is produced. 
This version use the quasi-Newton algorithm, 
re-using the Hessian from the previous bond length. 
"""

import os
from Dacapo import Dacapo
from ASE import Atom, ListOfAtoms
from ASE.Filters.Subset import Subset
from ASE.Filters.FixBondLength import FixBondLength
from ASE.Dynamics.QuasiNewton import QuasiNewton
from ASE.Trajectories.NetCDFTrajectory import NetCDFTrajectory

# setup the h2o molecule.
# setup the calculation spin-polarized: 
# Then the H atom are removed the 
# reactants are both spin-polarized.
atoms = ListOfAtoms([Atom('O', (0, 0, 0),         magmom=1),
                     Atom('H', (0.773, 0.600, 0), magmom=1),
                     Atom('H', (-0.773, 0.600, 0),magmom=1)]) 
                     

atoms.SetUnitCell((4,4,4),fix=True)

calc = Dacapo(planewavecutoff=300,   # in eV
              nbands=6,              # 1 extra empty bands
              out='h2o.nc',txtout='h2o.txt', 
              spinpol = True ) 


atoms.SetCalculator(calc)

# Create a trajectory for the reaction path
path = NetCDFTrajectory('h2o-dissociation.nc', atoms)
# Put the initial state in the trajectory:
path.Update()

# Create a trajectory for the all frames
allframes = NetCDFTrajectory('h2o-all.nc', atoms)
# Put the initial state in the trajectory:
allframes.Update()

# fix one H atom 
subset = Subset(atoms,indices=[0,1]) 
# fix the bond length
fixbond  = FixBondLength(subset,0,1)

# increase the bond length in steps of 0.1A
bondlength = 0.95
for n in range(10): 
    dyn = QuasiNewton(fixbond,fmax=0.1,diagonal=50.,logfilename='qn_'+str(n)+'.log') 
    if n>0: 
	dyn.ReadHessian('hessian1.pickle')

    dyn.Attach(allframes)
    fixbond.SetBondLength(bondlength) 
    calc.SetNetCDFFile('out_' + str(bondlength) + '.nc') 
    calc.SetTxtFile('out_' + str(bondlength) + '.txt') 
    print 'Minmizing for bond length ',bondlength
    dyn.Converge()
    dyn.WriteHessian('hessian1.pickle')
    path.Update()
    bondlength +=0.1



